#include  "soccer_viewer_option_analyser.h"
#include  "debug_server_connection.h"
#include  "ref_count_ptr.h"
#include  <iostream>
#include  <utility>

void   Soccer_Viewer_Option_Analyser::usage( std::ostream &  ostr ) const
{
	ostr << "Usage: "
	     << this -> program_name() << " [option ...]" << std::endl
	     << "Possible options are:" << std::endl
	     << "      --help" << std::endl
	     << "  -h, --host HOST" << std::endl
	     << "  -p, --port, --server-port PORT" << std::endl
	     << "  -P, --debug-port PORT" << std::endl
	     << "      --server-protocol {1,2}" << std::endl
	     << "  -l, --log-file {FILENAME,FILENAME.{gz,bz2,Z},-}"
							     << std::endl
	     << "      --show-view-angle" << std::endl
	     << "      --hide-view-angle" << std::endl
	     << "      --left-team-config {FILENAME,-}" << std::endl
	     << "      --right-team-config {FILENAME,-}" << std::endl
	     << "      --ignore-debug" << std::endl
	     << "      --auto-drop" << std::endl
	     << "      --auto-drop-on" << std::endl
	     << "      --auto-drop-off" << std::endl
	     << "      --auto-kick-off" << std::endl
	     << "      --auto-kick-off-on" << std::endl
	     << "      --auto-kick-off-off" << std::endl
	     << "      --auto-quit" << std::endl
	     << "      --auto-quit-on" << std::endl
	     << "      --auto-quit-off" << std::endl
	     << "      --server-process-id PID" << std::endl
	     << "      --default-window-size WIDTHxHEIGHT" << std::endl
	     << " and gtk options (--display etc.)" << std::endl;
}


//
// local class for --default-window-size option
//
class  Window_Size_Option_Descripter
	: public Option_Analyser::Option_Descripter
{
protected:
	mutable	std::pair<int, int> *	size;

public:
	Window_Size_Option_Descripter( std::pair<int, int> *  s )
		: size( s ) {}

	virtual	~Window_Size_Option_Descripter() {}

	virtual	int	execute( const std::vector<std::string> &  arg ) const
	{
		if ( arg.size() < 1 )
		{
			return( -1 );
		}

		int	window_width  = 0;
		int	window_height = 0;
		int	n = 0;

		if ( std::sscanf( arg[0].c_str() ,
				  "%dx%d%n" ,
				  &window_height , &window_width , &n ) == 2
		     && arg[0][n] == '\0' )
		{
			*(this -> size) = std::pair<int, int>( window_width ,
							       window_height );

			return( 1 );
		}
		else
		{
			std::cerr << "Bad window size \"" << arg[0] << "\"."
				  << std::endl;

			std::cerr << std::endl;

			return( -1 );
		}
	}
};


//
// local class for --log-file option
//
class  File_Name_Option_Descripter
	: public Option_Analyser::String_Option_Descripter
{
protected:
	bool *		log_source;

public:
	File_Name_Option_Descripter( std::string *  file_name ,
				     bool *  log_source )
		: Option_Analyser::String_Option_Descripter( file_name ) ,
		  log_source( log_source ) {}

	virtual	~File_Name_Option_Descripter() {}

	virtual	int	execute( const std::vector<std::string> &  arg ) const
	{
		*(this -> log_source) = true;

		return( Option_Analyser::String_Option_Descripter
			::execute( arg ) );
	}
};


Soccer_Viewer_Option_Analyser::Soccer_Viewer_Option_Analyser
			( const std::string &  default_server_host_name ,
			  ip_port_number_t  default_server_port_number ,
			  ip_port_number_t  default_debug_port_number )
{
	//
	// Initialize
	//
	this -> help_flag                   = false;
	this -> server_host                 = default_server_host_name;
	this -> server_port                 = default_server_port_number;
	this -> debug_port                  = default_debug_port_number;
	this -> serv_protocol_version       = 2;
	this -> log_file_source_flag        = false;
	this -> log_file_name_str           = "";
	this -> draw_player_view_angle_flag = true;
	this -> left_team_conf_flag         = false;
	this -> left_team_conf_file_name    = "";
	this -> right_team_conf_flag        = false;
	this -> right_team_conf_file_name   = "";
	this -> receive_debug_flag          = true;
	this -> auto_drop_flag              = true;
	this -> auto_kick_off_flag          = false;
	this -> auto_quit_flag              = false;
	this -> server_pid                  = -1;
	this -> window_size                 = std::pair<int, int>( 0 , 0 );


	//
	// Add Options
	//
	this -> add_long_option_flag_on(       "--help"       , &help_flag );
	this -> add_option_string      ( 'h' , "--host"       , &server_host );
	this -> add_long_option_string (       "--server-host", &server_host );
	this -> add_option_integer     ( 'p' , "--port"       , &server_port );
	this -> add_long_option_integer(       "--server-port", &server_port );
	this -> add_option_integer     ( 'P' , "--debug-port" , &debug_port );

	this -> add_long_option_integer(       "--server-protocol" ,
					       &serv_protocol_version );
	this -> add_option             ( 'l' , "--log-file" ,
					 new File_Name_Option_Descripter
					     ( &log_file_name_str ,
					       &log_file_source_flag ) );

	this -> add_long_option_flag_on(       "--show-view-angle" ,
					       &draw_player_view_angle_flag );
	this -> add_long_option_flag_off(      "--hide-view-angle" ,
					       &draw_player_view_angle_flag );

	this -> add_long_option        ( "--left-team-config" ,
					 new File_Name_Option_Descripter
					     ( &left_team_conf_file_name ,
					       &left_team_conf_flag ) );
	this -> add_long_option        ( "--right-team-config" ,
					 new File_Name_Option_Descripter
					     ( &right_team_conf_file_name ,
					       &right_team_conf_flag ) );

	this -> add_long_option_flag_off(      "--ignore-debug" ,
					       &receive_debug_flag );

	this -> add_long_option_flag_on (     "--auto-drop" ,
					      &auto_drop_flag );
	this -> add_long_option_flag_on (     "--auto-drop-on" ,
					      &auto_drop_flag );
	this -> add_long_option_flag_off(     "--auto-drop-off" ,
					      &auto_drop_flag );
	this -> add_long_option_flag_on (     "--auto-kick-off" ,
					      &auto_kick_off_flag );
	this -> add_long_option_flag_on (     "--auto-kick-off-on" ,
					      &auto_kick_off_flag );
	this -> add_long_option_flag_off(     "--auto-kick-off-off" ,
					      &auto_kick_off_flag );
	this -> add_long_option_flag_on (     "--auto-quit" ,
					      &auto_quit_flag );
	this -> add_long_option_flag_on (     "--auto-quit-on" ,
					      &auto_quit_flag );
	this -> add_long_option_flag_off(     "--auto-quit-off" ,
					      &auto_quit_flag );

	this -> add_long_option_integer(       "--server-process-id",
					       &server_pid );

	this -> add_long_option
		( "--default-window-size" ,
		  new Window_Size_Option_Descripter( &window_size ) );
}

Soccer_Viewer_Option_Analyser::~Soccer_Viewer_Option_Analyser()
{
}

bool   Soccer_Viewer_Option_Analyser::check() const
{
	return( this -> rest().size() == 0 );
}

bool   Soccer_Viewer_Option_Analyser::help() const
{
	return( this -> help_flag );
}

std::string  Soccer_Viewer_Option_Analyser::server_hostname() const
{
	return( this -> server_host );
}

ip_port_number_t  Soccer_Viewer_Option_Analyser::server_port_number() const
{
	return( static_cast<ip_port_number_t>( this -> server_port ) );
}

ip_port_number_t  Soccer_Viewer_Option_Analyser::debug_port_number() const
{
	if ( this -> debug_port == 0 )
	{
		return( this -> server_port_number()
			  + Debug_Server_Connection::DEFAULT_PORT_OFFSET );
	}
	else
	{
		return( static_cast<ip_port_number_t>( this -> debug_port ) );
	}
}

int    Soccer_Viewer_Option_Analyser::server_protocol_version() const
{
	return( this -> serv_protocol_version );
}

bool   Soccer_Viewer_Option_Analyser::log_file_source() const
{
	return( this -> log_file_source_flag );
}

std::string  Soccer_Viewer_Option_Analyser::log_file_name() const
{
	return( this -> log_file_name_str );
}

bool   Soccer_Viewer_Option_Analyser::draw_player_view_angle() const
{
	return( this -> draw_player_view_angle_flag );
}

bool   Soccer_Viewer_Option_Analyser::have_left_team_config() const
{
	return( this -> left_team_conf_flag );
}

std::string  Soccer_Viewer_Option_Analyser::left_team_config_file_name() const
{
	return( this -> left_team_conf_file_name );
}

bool   Soccer_Viewer_Option_Analyser::have_right_team_config() const
{
	return( this -> right_team_conf_flag );
}

std::string  Soccer_Viewer_Option_Analyser::right_team_config_file_name() const
{
	return( this -> right_team_conf_file_name );
}

bool   Soccer_Viewer_Option_Analyser::receive_debug() const
{
	return( this -> receive_debug_flag );
}

bool   Soccer_Viewer_Option_Analyser::auto_drop() const
{
	return( this -> auto_drop_flag );
}

bool   Soccer_Viewer_Option_Analyser::auto_kick_off() const
{
	return( this -> auto_kick_off_flag );
}

bool   Soccer_Viewer_Option_Analyser::auto_quit() const
{
	return( this -> auto_quit_flag );
}

pid_t  Soccer_Viewer_Option_Analyser::server_process_id() const
{
	return( static_cast<pid_t>( this -> server_pid ) );
}

std::pair<int, int>  Soccer_Viewer_Option_Analyser::default_window_size()
	const
{
	return( this -> window_size );
}
