#include  "time_controller.h"
#include  <algorithm>

const int	Time_Controller::FAST_SIMULATOR_STEP_MSEC      = 20 /* msec */;
const int	Time_Controller::VERY_FAST_SIMULATOR_STEP_MSEC =  1 /* msec */;
const int	Time_Controller::N_STEP_DELAY = 3 /* step */;

Time_Controller::Time_Controller( Active_Repository &  rep ,
				  Abstract_Soccer_Viewer &  v ,
				  bool  log_file_source )
	: viewer( v ) ,
	  active_repository( rep ) ,
	  repository_changed_receiver
		( new Active_Repository_Changed_Event_Receiver( *this ) ) ,
	  callback( *this ) ,
	  updater( callback ) ,
#if 0
	  live_mode_flag( ! log_file_source ) ,
	  log_count( log_file_source ? 1 : 0 ) ,
#else
	  live_mode_flag( true ) ,
	  log_count( 0 ) ,
#endif
	  log_wait( 0 ) , log_forward( true )
{
	for ( int i = 0  ;  i < Time_Controller::N_STEP_DELAY  ;  i ++ )
	{
		prev_server_time_queue.push( 0 );
	}

	active_repository.add_changed_event_receiver
				( repository_changed_receiver );
}

long   Time_Controller::snapshot_time() const
{
	if ( this -> live_mode() )
	{
		return( prev_server_time_queue.front() );
	}
	else
	{
		return( this -> log_count );
	}
}


ref_count_ptr<const View_Snapshot>  Time_Controller::snapshot() const
{
	if ( this -> live_mode() )
	{
		return( this -> current_snapshot() );
	}
	else
	{
		return( this -> log_snapshot() );
	}
}

ref_count_ptr<const View_Snapshot>  Time_Controller::current_snapshot() const
{
	std::map< long , ref_count_ptr<View_Snapshot> >::const_iterator	it;

	it = active_repository.get_game_log()
		.find( this -> prev_server_time_queue.front() );

	if ( it != active_repository.get_game_log().end() )
	{
		return( it -> second );
	}
	else
	{
		return( ref_count_ptr<View_Snapshot>( 0 ) );
	}
}

ref_count_ptr<const View_Snapshot>  Time_Controller::log_snapshot() const
{
	std::map< long , ref_count_ptr<View_Snapshot> >::const_iterator	it;

	it = active_repository.get_game_log()
		.find( this -> log_count );

	if ( it != active_repository.get_game_log().end() )
	{
		return( it -> second );
	}
	else
	{
		return( ref_count_ptr<View_Snapshot>( 0 ) );
	}
}

ref_count_ptr<const Debug_Client_Field_Recog>
	Time_Controller::client_view_data( const  S_Side_LR &  side ,
					   int  player_number ) const
{
	if ( this -> snapshot()
	  && this -> snapshot() -> client_view_data( side , player_number ) )
	{
		return( this -> snapshot()
				-> client_view_data( side , player_number ) );
	}
	else
	{
		return( ref_count_ptr<Debug_Client_Field_Recog>( 0 ) );
	}
}

ref_count_ptr<const Monitor_View_Data>   Time_Controller::monitor_view_data()
	const
{
	if ( this -> snapshot()
	  && this -> snapshot() -> monitor_view_data() )
	{
		return( this -> snapshot() -> monitor_view_data() );
	}
	else
	{
		return( ref_count_ptr<Monitor_View_Data>( 0 ) );
	}
}

bool   Time_Controller::live_mode() const
{
	return( this -> live_mode_flag );
}



int    Time_Controller::send_start()
{
	return( this -> active_repository.send_start() );
}

int    Time_Controller::send_foul( double  x ,  double  y ,  S_Side_LR  side ,
				   bool  change_to_live_mode )
{
	if ( change_to_live_mode )
	{
		this -> live_mode_flag = true;
		this -> log_wait = 0;
	}

	return( active_repository.send_foul( x , y , side ) );
}


Time_Controller::Active_Repository_Changed_Event_Receiver
	    ::Active_Repository_Changed_Event_Receiver( Time_Controller &  tc )
	: tc( tc )
{
}

void   Time_Controller::Active_Repository_Changed_Event_Receiver::notify()
{
	tc.repository_changed();
}

void   Time_Controller::repository_changed()
{
	this -> prev_server_time_queue.pop();
	this -> prev_server_time_queue
		.push( active_repository.get_server_time() );

	if ( this -> live_mode() )
	{
		this -> log_count = prev_server_time_queue.front();
	}

	this -> viewer.display();
}

// callback
Time_Controller::Timer_Callback::Timer_Callback( Time_Controller &  tc )
	: tc( tc )
{
}

gint   Time_Controller::Timer_Callback::operator() ()
{
	return( this -> tc.timed_out() );
}

int    Time_Controller::timed_out()
{
#if 0
	if ( this -> live_mode() )
	{
		this -> viewer.display();

		return( Active_Repository::SIMULATOR_STEP_MSEC );
	}
#endif

	if ( this -> log_wait != 0 )
	{
		if ( this -> log_wait < 0 )
		{
			this -> log_wait = 0;
		}

		if ( this -> log_forward )
		{
			if ( this -> log_count
			     <= this -> prev_server_time_queue.front() )
			{
				this -> forward_log();
			}
			else
			{
				this -> live_mode_flag = true;
				this -> log_wait = 0;
			}
		}
		else
		{
			this -> backward_log();
		}
	}

	return( this -> log_wait );
}

void   Time_Controller::forward_log()
{
	this -> log_count ++;

	if ( this -> log_count < 0 )
	{
		this -> log_count = 0;
	}

	this -> viewer.display();
}

void   Time_Controller::backward_log()
{
	this -> log_count --;

	if ( this -> log_count < 0 )
	{
		this -> log_count = 0;
		this -> log_wait = 0;
	}

	this -> viewer.display();
}

void   Time_Controller::stop_log()
{
	this -> viewer.display();
}

void   Time_Controller::set_log( long  t )
{
	this -> log_count = t;

	if ( this -> log_count < 0 )
	{
		this -> log_count = 0;
	}

	this -> viewer.display();
}


void   Time_Controller::set_backward_very_fast()
{
	this -> live_mode_flag = false;
	this -> log_forward = false;

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = VERY_FAST_SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = VERY_FAST_SIMULATOR_STEP_MSEC;
	}
}

void   Time_Controller::set_backward_fast()
{
	this -> live_mode_flag = false;
	this -> log_forward = false;

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = FAST_SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = FAST_SIMULATOR_STEP_MSEC;
	}
}


void   Time_Controller::set_backward()
{
	this -> live_mode_flag = false;
	this -> log_forward = false;

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = Active_Repository::SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = Active_Repository::SIMULATOR_STEP_MSEC;
	}
}


void   Time_Controller::set_backward_step()
{
	this -> live_mode_flag = false;
	this -> log_wait = 0;

	this -> backward_log();
}

void   Time_Controller::set_stop()
{
	this -> live_mode_flag = false;
	this -> log_wait = 0;
	this -> stop_log();
}

void   Time_Controller::set_forward_step()
{
	this -> live_mode_flag = false;
	this -> log_wait = 0;

	if ( this -> monitor_view_data() )
	{
		if ( this -> monitor_view_data() -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Before_Kick_Off )
		{
			this -> send_start();

			this -> updater.set_timer
				( Active_Repository::SIMULATOR_STEP_MSEC
						      * (N_STEP_DELAY + 1) );
			this -> log_wait = -1;
		}
	}

	this -> forward_log();
}

void   Time_Controller::set_forward()
{
	this -> live_mode_flag = false;
	this -> log_forward = true;

	if ( this -> monitor_view_data() )
	{
		if ( this -> monitor_view_data() -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Before_Kick_Off )
		{
			this -> send_start();
		}
	}

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = Active_Repository::SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = Active_Repository::SIMULATOR_STEP_MSEC;
	}
}

void   Time_Controller::set_forward_fast()
{
	this -> live_mode_flag = false;
	this -> log_forward = true;

	if ( this -> monitor_view_data() )
	{
		if ( this -> monitor_view_data() -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Before_Kick_Off )
		{
			this -> send_start();
		}
	}

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = FAST_SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = FAST_SIMULATOR_STEP_MSEC;
	}
}

void   Time_Controller::set_forward_very_fast()
{
	this -> live_mode_flag = false;
	this -> log_forward = true;

	if ( this -> monitor_view_data() )
	{
		if ( this -> monitor_view_data() -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Before_Kick_Off )
		{
			this -> send_start();
		}
	}

	if ( this -> log_wait == 0 )
	{
		this -> log_wait = VERY_FAST_SIMULATOR_STEP_MSEC;

		this -> updater.set_timer( log_wait /* msec */ );
	}
	else
	{
		this -> log_wait = VERY_FAST_SIMULATOR_STEP_MSEC;
	}
}

void   Time_Controller::set_forward_highlight()
{
	for ( std::vector< const std::pair<long,S_Side_LR> >::iterator
		      it = active_repository.get_goal_time().begin()  ;
	      it != active_repository.get_goal_time().end()  ;  it ++ )
	{
		if ( (it -> first) > (this -> log_count + 1) )
		{
			this -> set_time( std::max( it -> first - 1 ,
						    static_cast<long>( 0 ) ) );

			return;
		}
	}

	this -> set_live_mode();
}

void   Time_Controller::set_backward_highlight()
{
	for ( std::vector< const std::pair<long,S_Side_LR> >::reverse_iterator
		      it = active_repository.get_goal_time().rbegin()  ;
	      it != active_repository.get_goal_time().rend()  ;  it ++ )
	{
		if ( (it -> first) < (this -> log_count + 1) )
		{
			this -> set_time( std::max( it -> first - 1 ,
						    static_cast<long>( 0 ) ) );

			return;
		}
	}

	this -> set_time( 0 );
}

void   Time_Controller::set_toggle_live_log()
{
	this -> live_mode_flag = (! this -> live_mode());

	if ( this -> live_mode_flag )
	{
		this -> log_count = active_repository.get_server_time();

		this -> updater
			.set_timer( Active_Repository::SIMULATOR_STEP_MSEC );
	}
	else
	{
		this -> log_wait = 0;
	}

	this -> stop_log();
}

void   Time_Controller::set_live_mode()
{
	this -> live_mode_flag = true;

	this -> log_count = active_repository.get_server_time();

	this -> updater.set_timer( Active_Repository::SIMULATOR_STEP_MSEC );

	this -> stop_log();
}

void   Time_Controller::set_time( long  t )
{
	this -> live_mode_flag = false;
	this -> log_wait = 0;

	this -> set_log( t );
}

bool   Time_Controller::have_server_process_id() const
{
	return( this -> active_repository.have_server_process_id() );
}

void   Time_Controller::kill_server()
{
	this -> active_repository.kill_server();
}

bool   Time_Controller::monitor_connection_responsive() const
{
	return( this -> active_repository.monitor_connection_responsive() );
}

bool   Time_Controller::game_over() const
{
	return( this -> active_repository.game_over() );
}

const SServer_Param &  Time_Controller::sserver_param() const
{
	return( this -> active_repository.sserver_param() );
}

void   Time_Controller::set_auto_drop( bool  f )
{
	this -> active_repository.set_auto_drop( f );
}

bool   Time_Controller::auto_drop_mode() const
{
	return( this -> active_repository.auto_drop_mode() );
}

long   Time_Controller::auto_drop_rest_step() const
{
	return( this -> active_repository.auto_drop_rest_step() );
}

long   Time_Controller::auto_drop_rest_second() const
{
	return( this -> active_repository.auto_drop_rest_second() );
}

void   Time_Controller::set_auto_kick_off( bool  f )
{
	this -> active_repository.set_auto_kick_off( f );
}

bool   Time_Controller::auto_kick_off_mode() const
{
	return( this -> active_repository.auto_kick_off_mode() );
}

long   Time_Controller::auto_kick_off_rest_step() const
{
	return( this -> active_repository.auto_kick_off_rest_step() );
}

long   Time_Controller::auto_kick_off_rest_second() const
{
	return( this -> active_repository.auto_kick_off_rest_second() );
}

void   Time_Controller::set_auto_quit( bool  f )
{
	this -> active_repository.set_auto_quit( f );
}

bool   Time_Controller::auto_quit_mode() const
{
	return( this -> active_repository.auto_quit_mode() );
}

long   Time_Controller::auto_quit_rest_step() const
{
	return( this -> active_repository.auto_quit_rest_step() );
}

long   Time_Controller::auto_quit_rest_second() const
{
	return( this -> active_repository.auto_quit_rest_second() );
}
