#include  "field_canvas.h"
#include  "math_extention.h"
#include  <set>
#include  <cmath>
#include  <cstring>
#include  <algorithm>

using namespace std;

//
// Default Size and Magnify
//
const	gint	Field_Canvas::DEFAULT_WIDTH  = 930;  // pixel
const	gint	Field_Canvas::DEFAULT_HEIGHT = 620;  // pixel

const	gdouble	Field_Canvas::DEFAULT_MAGNIFY = 8.0;


//
// Parameter for Drawing Field
//
const	gint	Field_Canvas::MAX_AUTO_DROP_COLUMN     = 4;
const	gint	Field_Canvas::MAX_AUTO_KICK_OFF_COLUMN = 4;
const	gint	Field_Canvas::MAX_AUTO_QUIT_COLUMN     = 4;

const	gdouble	Field_Canvas::FIELD_HALF_WIDTH  = 34.0;
const	gdouble	Field_Canvas::FIELD_HALF_LENGTH = 52.5;
const	gdouble	Field_Canvas::FIELD_WIDTH  = FIELD_HALF_WIDTH * 2.0;
const	gdouble	Field_Canvas::FIELD_LENGTH = FIELD_HALF_LENGTH * 2.0;

const	gdouble	Field_Canvas::PENALTY_AREA_LENGTH     = 16.5;
const	gdouble	Field_Canvas::PENALTY_AREA_HALF_WIDTH = 20.16;

const	gdouble	Field_Canvas::GOAL_AREA_LENGTH     = 5.5;
const	gdouble	Field_Canvas::GOAL_AREA_HALF_WIDTH = 9.16;

const	gdouble	Field_Canvas::CENTER_CIRCLE_R = 9.15;
const	gdouble	Field_Canvas::CORNER_CIRCLE_R = 1.0;

const	gdouble	Field_Canvas::PENALTY_SPOT_DIST = 11.0;


//
// Canvas_Event_Handler
//
Field_Canvas::Canvas_Event_Handler::Canvas_Event_Handler()
{
}

Field_Canvas::Canvas_Event_Handler::~Canvas_Event_Handler()
{
}

gint   Field_Canvas::Canvas_Event_Handler::button_pressed( GdkEventButton * ,
							   double , double )
{
	return( false );
}

gint   Field_Canvas::Canvas_Event_Handler::button_released( GdkEventButton * ,
							    double , double )
{
	return( false );
}

gint   Field_Canvas::Canvas_Event_Handler::key_pressed( GdkEventKey * )
{
	return( false );
}

gint   Field_Canvas::Canvas_Event_Handler::key_released( GdkEventKey * )
{
	return( false );
}



//
// Field_Canvas
//
Field_Canvas::Field_Canvas( Field_Canvas::Canvas_Event_Handler &  handler ,
			    const SServer_Param &  param ,
			    gdouble  view_point_x ,  gdouble  view_point_y ,
			    gdouble  mag ,
			    bool  inv_x , bool  inv_y )
	: PENALTY_ARC_DIGREE
		( std::acos((PENALTY_AREA_LENGTH - PENALTY_SPOT_DIST)
			    / CENTER_CIRCLE_R)
		  * 180.0 / Math_Extention::PI - 1.0 ) ,
	  param( param ) ,
	  event_handler( handler ) ,
	  ratio( 1.0 / mag ) ,
	  center_x( view_point_x ) , center_y( view_point_y ) ,
	  inverse_x( inv_x ) , inverse_y( inv_y ) ,
	  max_number_char_width( 0 ) ,
	  auto_drop_message_x( 0 ) , auto_drop_message_y( width() ) ,
	  auto_drop_message_width( 0 ) ,
	  auto_kick_off_message_x( 0 ) , auto_kick_off_message_y( width() ) ,
	  auto_kick_off_message_width( 0 ) ,
	  auto_quit_message_x( 0 ) , auto_quit_message_y( width() ) ,
	  auto_quit_message_width( 0 ) ,
	  gc_font_initialized( false )
{
	//
	// set events
	//
	this -> add_events(   GDK_BUTTON_PRESS_MASK
			    | GDK_BUTTON_RELEASE_MASK
			    | GDK_KEY_PRESS_MASK
			    | GDK_KEY_RELEASE_MASK );


	//
	// set size
	//
	this -> size( this -> DEFAULT_WIDTH , this -> DEFAULT_HEIGHT );
}

Field_Canvas::~Field_Canvas()
{
}

void   Field_Canvas::view_change( gdouble  view_point_x ,
				  gdouble  view_point_y ,
				  gdouble  mag ,
				  bool  inv_x ,
				  bool  inv_y )
{
	const double	EPS = 1.0e-10;

	bool	changed = false;

	if ( std::fabs( this -> ratio     - (1.0 / mag)  ) > EPS
	  || std::fabs( this -> center_x  - view_point_x ) > EPS
	  || std::fabs( this -> center_y  - view_point_y ) > EPS
	  || this -> inverse_x != inv_x
	  || this -> inverse_y != inv_y )
	{
		changed = true;
	}

	this -> ratio = 1.0 / mag;
	this -> center_x = view_point_x;
	this -> center_y = view_point_y;
	this -> inverse_x = inv_x;
	this -> inverse_y = inv_y;

	if ( changed )
	{
		this -> configure();
	}
}


void   Field_Canvas::gc_font_initialize()
{
	//
	// GC
	//
	ground_gc.create( get_window() , "indianred" );
	frame_gc .create( get_window() , "antique white" );

	field_gc .create( get_window() , "forestgreen" );
	line_gc  .create( get_window() , "white" );
	goal_gc  .create( get_window() , "black" );
	window_gc.create( get_window() , "white" );

	grid_main_gc    .create( get_window() , "dark slate gray" );
	grid_sub_gc     .create( get_window() , "light slate gray" );
	player_circle_gc.create( get_window() , "dark slate gray" );
	ball_circle_gc  .create( get_window() , "white" );

	ball_gc         .create( get_window() , "white"      );
	debug_ball_gc   .create( get_window() , "royal blue" );
	ball_velocity_gc.create( get_window() , "white"      );
	ball_edge_gc    .create( get_window() , "black"      );
	ball_comment_gc .create( get_window() , "black"      );

	left_field_player_gc            .create( get_window() , "gold"       );
	left_goalie_gc                  .create( get_window() , "green"      );
	right_field_player_gc           .create( get_window() , "red"        );
	right_goalie_gc                 .create( get_window() , "purple"     );
	realized_self_player_gc         .create( get_window() , "royal blue" );
	realized_left_player_gc         .create( get_window() ,
						 "green yellow" );
	realized_right_player_gc        .create( get_window() ,
						 "light sky blue" );
	unknown_player_gc               .create( get_window() , "gray"       );
	player_bg_gc                    .create( get_window() , "black"      );
	player_view_angle_gc            .create( get_window() , "royal blue" );
	player_face_bar_gc              .create( get_window() , "white"      );
	effort_reduced_gc               .create( get_window() , "indian red" );
	left_player_uniform_number_gc   .create( get_window() , "indian red" );
	right_player_uniform_number_gc  .create( get_window() , "gray"       );
	unknown_player_uniform_number_gc.create( get_window() , "gray"       );
	player_name_gc                  .create( get_window() , "white"      );
	catch_fault_gc                  .create( get_window() , "gray"       );
	target_point_gc                 .create( get_window() , "white"      );
	popup_string_gc                 .create( get_window() , "white"  );


	//
	// Font
	//
	player_identification_font.set_load( "9x15bold" );
	comment_font.set_load( "9x15bold" );

	popup_font.set_load
		( "-adobe-helvetica-*-r-normal--24-240-75-75-p-*-iso8859-1" );

	this -> gc_font_initialize_hook();

	gc_font_initialized = true;
}

/* virtual */
void   Field_Canvas::gc_font_initialize_hook()
{
	// no operation
}


gint   Field_Canvas::configure()
{
	if ( ! gc_font_initialized )
	{
		gc_font_initialize();
	}

	//
	// new field_environment_pixmap
	//
	field_environment_pixmap.create( this -> center_x , this -> center_y ,
					 this -> ratio ,
					 this -> inverse_x ,
					 this -> inverse_y ,
					 get_window() , width() , height() );


	//
	// draw field environment
	//
	draw_field_environment_to_pixmap( field_environment_pixmap );


	//
	// new pixmap
	//
	pixmap.entity().set_background( field_environment_pixmap.entity() );

	pixmap.create( this -> center_x , this -> center_y , this -> ratio ,
		       this -> inverse_x , this -> inverse_y ,
		       get_window() , width() , height() );


	//
	// draw background
	//
	this -> get_window().set_back_pixmap(
			     field_environment_pixmap.entity() , false );


	//
	// calculate auto_kick_off, auto_drop and auto_quit message position
	//
	{
		const string	numbers = "0123456789";
		for ( size_t  i = 0  ;  i < numbers.length()  ;  i ++ )
		{
			gint	w = popup_font.char_width( numbers[i] );
			if ( w > this -> max_number_char_width )
			{
				this -> max_number_char_width = w;
			}
		}

		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		// auto drop
		const string	auto_drop_string = "Auto Drop: ";

		popup_font.string_extents( auto_drop_string ,
					   lbearing , rbearing ,
					   text_width ,
					   ascent , descent );

		this -> auto_drop_message_width = rbearing - lbearing;

		this -> auto_drop_message_x
			  = width() - 3
			    - (this -> auto_drop_message_width)
			    - (this -> max_number_char_width)
				* MAX_AUTO_DROP_COLUMN;

		this -> auto_drop_message_y = height() - 8;


		// auto kick off
		const string	auto_kick_off_string = "Auto Kick Off: ";

		popup_font.string_extents( auto_kick_off_string ,
					   lbearing , rbearing ,
					   text_width ,
					   ascent , descent );

		this -> auto_kick_off_message_width = rbearing - lbearing;

		this -> auto_kick_off_message_x
			  = width() - 3
			    - (this -> auto_kick_off_message_width)
			    - (this -> max_number_char_width)
				* MAX_AUTO_KICK_OFF_COLUMN;

		this -> auto_kick_off_message_y = height() - 8;


		// auto quit
		const string	auto_quit_string = "Auto Quit: ";

		popup_font.string_extents( auto_quit_string ,
					   lbearing , rbearing ,
					   text_width ,
					   ascent , descent );

		this -> auto_quit_message_width = rbearing - lbearing;

		this -> auto_quit_message_x
			  = width() - 3
			    - (this -> auto_quit_message_width)
			    - (this -> max_number_char_width)
				* MAX_AUTO_QUIT_COLUMN;

		this -> auto_quit_message_y = height() - 8;
	}

	this -> event_handler.configured();

	return( true );
}

gint   Field_Canvas::configure_event_impl( GdkEventConfigure * )
{
	this -> configure();

	return( true );
}

gint   Field_Canvas::expose_event_impl( GdkEventExpose *  ev )
{
	if ( ev -> count == 0 )
	{
		get_window().draw_pixmap( pixmap.entity().gc() ,
					  pixmap.entity() ,
					  0 , 0 ,
					  0 , 0 ,
					  width() , height() );
		pixmap.entity().drawn();
	}

	return( false );
}

gint   Field_Canvas::button_press_event_impl( GdkEventButton *  ev )
{
	return( this -> event_handler.button_pressed
		( ev ,
		  pixmap.x_pixel_to_coordinate
				  ( static_cast<gint>( ev -> x ) ) ,
		  pixmap.y_pixel_to_coordinate
				  ( static_cast<gint>( ev -> y ) ) ) );
}

gint   Field_Canvas::button_release_event_impl( GdkEventButton *  ev )
{
	return( this -> event_handler.button_released
		( ev ,
		  pixmap.x_pixel_to_coordinate
				  ( static_cast<gint>( ev -> x ) ) ,
		  pixmap.y_pixel_to_coordinate
				  ( static_cast<gint>( ev -> y ) ) ) );
}

gint   Field_Canvas::key_press_event_impl( GdkEventKey *  ev )
{
	return( this -> event_handler.key_pressed( ev ) );
}

gint   Field_Canvas::key_release_event_impl( GdkEventKey *  ev )
{
	return( this -> event_handler.key_released( ev ) );
}

void   Field_Canvas::get_pointer_coordinate( double *  x ,  double *  y )
{
	gint	pointer_x;
	gint	pointer_y;

	this -> get_pointer( pointer_x , pointer_y );

	*x = pixmap.x_pixel_to_coordinate( pointer_x );
	*y = pixmap.y_pixel_to_coordinate( pointer_y );
}


void   Field_Canvas::clear_field()
{
	pixmap.entity().clear();
}


void   Field_Canvas::draw_grid( gdouble  grid_length )
{
	this -> draw_grid_to_pixmap( pixmap , grid_length );
}

void   Field_Canvas::draw_circle
			( const Monitor_View_Data &  monitor_view_data ,
			  gdouble  circle_radius )
{
	this -> draw_circle_to_pixmap( pixmap ,
				       monitor_view_data , circle_radius );
}

void   Field_Canvas::draw_message_center( const string &  message )
{
	gint	lbearing;
	gint	rbearing;
	gint	text_width;
	gint	ascent;
	gint	descent;

	popup_font.string_extents( message ,
				   lbearing , rbearing ,
				   text_width ,
				   ascent , descent );

	gint	message_width = rbearing - lbearing;
	gint	message_height = ascent + descent;

	this -> pixmap.entity().draw_string
		( this -> popup_font ,
		  this -> popup_string_gc ,
		  (this -> pixmap.width() - message_width) / 2 ,
		  (this -> pixmap.height() - message_height) / 2 + ascent ,
		  message );
}


void   Field_Canvas::draw_monitor_view_data
			( const Monitor_View_Data &  monitor_view_data ,
			  bool  draw_view_angle ,
			  const Team_Config  team_config[2] ,
			  gdouble  player_size ,
			  gdouble  ball_size ,
			  bool  use_teamname )
{
	int	offset[2]            = { 0 , MAX_PLAYER };
	int	team_config_index[2] = { 0 , 1 };

	if (   team_config[0].have_player_config
	  && ! team_config[1].have_player_config )
	{
		std::swap( offset[0] , offset[1] );
		std::swap( team_config_index[0] , team_config_index[1] );
	}

	for ( int  team_index = 0  ;  team_index <= 1  ;  team_index ++ )
	{
		for ( int  i = 0  ;  i < MAX_PLAYER  ;  i ++ )
		{
			this -> draw_player_to_pixmap
			( pixmap ,
			  monitor_view_data.player[ offset[team_index] + i ],
			  draw_view_angle ,
			  player_size ,
			  static_cast<char *>(0) ,
			  team_config[team_config_index[team_index]]
					  .player[i].name.c_str() ,
			  team_config[team_config_index[team_index]]
					  .player[i].have_name ,
			  team_config[team_config_index[team_index]]
					  .player[i].color.c_str() ,
			  team_config[team_config_index[team_index]]
					  .player[i].have_color ,
			  false ,
			  static_cast<char *>(0) ,
			  false );
		}
	}

	this -> draw_monitor_ball_to_pixmap( pixmap , ball_size ,
					     monitor_view_data.ball ,
					     monitor_view_data.play_mode );

	this -> draw_popup_to_pixmap( pixmap , use_teamname ,
				      monitor_view_data.play_mode ,
				      monitor_view_data.team[0].team_name ,
				      monitor_view_data.team[1].team_name );
}

void   Field_Canvas::draw_auto_drop_count_down( long  auto_drop_rest_count )
{
	this -> draw_auto_drop_count_down_to_pixmap
				( pixmap , auto_drop_rest_count );
}

void   Field_Canvas::draw_auto_kick_off_count_down
				( long  auto_kick_off_rest_count )
{
	this -> draw_auto_kick_off_count_down_to_pixmap
				( pixmap , auto_kick_off_rest_count );
}

void   Field_Canvas::draw_auto_quit_count_down( long  auto_quit_rest_count )
{
	this -> draw_auto_quit_count_down_to_pixmap
				( pixmap , auto_quit_rest_count );
}

void   Field_Canvas::draw_ball( const Ball_Reference &  ball ,
				S_Side_LR  self_side_lr ,
				gdouble  ball_size ,
				const Play_Mode_Reference &  mode ,
				bool  draw_velocity ,
				const char *  comment ,
				bool  for_debug )
{
	D2_Vector	ball_coordinate;
	D2_Vector	ball_velocity;

	switch( self_side_lr )
	{
	case S_Side_LR::Left_Side:
		ball_coordinate = (+ ball.coordinate());
		ball_velocity   = (+ ball.velocity());
		break;

	case S_Side_LR::Right_Side:
		ball_coordinate = (- ball.coordinate());
		ball_velocity   = (- ball.velocity());
		break;

	case S_Side_LR::Unknown:
	default:
		// error
		return;
		break;
	}


	S_Side	ball_side;

	switch( mode.mode() )
	{
	case Play_Mode::Unknown:
	case Play_Mode::Before_Kick_Off:
	case Play_Mode::Play_On:
	case Play_Mode::Time_Over:
		ball_side = S_Side::Unknown;
		break;

	case Play_Mode::Our_Kick_Off:
	case Play_Mode::Our_Kick_In:
	case Play_Mode::Our_Free_Kick:
	case Play_Mode::Our_Corner_Kick:
	case Play_Mode::Our_Goal_Kick:
	case Play_Mode::Our_Catch:
	case Play_Mode::Opponent_Goal:
		ball_side = S_Side::Our_Side;
		break;

	case Play_Mode::Opponent_Kick_Off:
	case Play_Mode::Opponent_Kick_In:
	case Play_Mode::Opponent_Free_Kick:
	case Play_Mode::Opponent_Corner_Kick:
	case Play_Mode::Opponent_Goal_Kick:
	case Play_Mode::Opponent_Catch:
	case Play_Mode::Our_Goal:
		ball_side = S_Side::Opponent_Side;
		break;
	}


	S_Side_LR	ball_side_lr;

	switch( ball_side )
	{
	case S_Side::Our_Side:
		ball_side_lr = self_side_lr;
		break;

	case S_Side::Opponent_Side:
		ball_side_lr = self_side_lr.reverse();
		break;

	case S_Side::Unknown:
		ball_side_lr = S_Side_LR::Unknown;
		break;
	}


	this -> draw_ball_to_pixmap
		( pixmap , ball_size , ball_coordinate ,
		  draw_velocity && ball.velocity().have_info() ,
		  ball_velocity ,
		  ball_side_lr , comment , for_debug );
}

void   Field_Canvas::draw_player( const Player_Reference &  player ,
				  S_Side_LR  self_side ,
				  const Soccer_Composite_Command &  com ,
				  gdouble  player_size ,
				  const char *  comment ,
				  bool  draw_face_angle ,
				  bool  use_stamina ,
				  const char *  debug_message ,
				  bool  for_debug )
{
	Monitor_View_Data::Player_State	state;

	state.set_valid( true );

	state.set_side( player.side_lr() );

	state.set_uniform_number( (player.player_number() == (-1)) ?
				  0 : player.player_number() );

	state.set_goalie( player.goalie() );

	state.set_kicking
		( com.base_type() == Soccer_Command::Kick_Command );

	state.set_catching
		( com.base_type() == Soccer_Command::Catch_Ball_Command );

	switch( self_side )
	{
	case S_Side_LR::Left_Side:
		if ( player.body_angle().accuracy_check() )
		{
			state.set_body_angle( player.body_angle() );
		}
		else
		{
			if ( player.side_lr() == S_Side_LR::Right_Side )
			{
				state.set_body_angle( Degree( -180.0 ) );
			}
			else
			{
				state.set_body_angle( Degree( 0.0 ) );
			}
		}

		state.set_x( player.coordinate().x() );
		state.set_y( player.coordinate().y() );

		break;

	case S_Side_LR::Right_Side:
		if ( player.body_angle().accuracy_check() )
		{
			state.set_body_angle( (player.body_angle()
					       - Degree( 180.0 ))
					      .normalize() );
		}
		else
		{
			if ( player.side_lr() == S_Side_LR::Left_Side )
			{
				state.set_body_angle( Degree( 0.0 ) );
			}
			else
			{
				state.set_body_angle( Degree( -180.0 ) );
			}
		}

		state.set_x( - player.coordinate().x() );
		state.set_y( - player.coordinate().y() );

		break;

	case S_Side_LR::Unknown:
	default:
		// error
		return;
		break;
	}


	if ( draw_face_angle && player.face_angle().accuracy_check( 0 ) )
	{
		state.set_head_angle( player.face_angle().normalize() );
	}


	if ( use_stamina )
	{
		state.set_stamina( player.stamina() );
	}

	this -> draw_player_to_pixmap( pixmap ,
				       state ,
				       draw_face_angle ,
				       player_size ,
				       comment ,
				       static_cast<char *>(0) , false ,
				       static_cast<char *>(0) , false ,
				       player.self_player() ,
				       debug_message ,
				       for_debug );
}

void   Field_Canvas::update_field()
{
	this -> draw( Gdk_Rectangle( 0 , 0 ,
				     this -> width() , this -> height() ) );
}

void   Field_Canvas::draw_grid_to_pixmap
			( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
			  gdouble  grid_width )
{
	gint	min_x = static_cast<gint>( pix.min_x() / grid_width );
	gint	max_x = static_cast<gint>( pix.max_x() / grid_width );
	gint	min_y = static_cast<gint>( pix.min_y() / grid_width );
	gint	max_y = static_cast<gint>( pix.max_y() / grid_width );

	// up <-> down sub line
	for ( gint  x = min_x  ;  x <= max_x  ; x ++ )
	{
		if ( std::abs( x ) % 2 == 1 )
		{
			pix.draw_line_absolute( grid_sub_gc ,
						x * grid_width , pix.min_y() ,
						x * grid_width , pix.max_y() );
		}
	}

	// left <-> right sub line
	for ( gint  y = min_y  ;  y <= max_y  ; y ++ )
	{
		if ( std::abs( y ) % 2 == 1 )
		{
			pix.draw_line_absolute( grid_sub_gc ,
						pix.min_x() , y * grid_width ,
						pix.max_x() , y * grid_width );
		}
	}

	// up <-> down main line
	for ( gint  x = min_x  ;  x <= max_x  ; x ++ )
	{
		if ( std::abs( x ) % 2 == 0  &&  x != 0 )
		{
			pix.draw_line_absolute( grid_main_gc ,
						x * grid_width , pix.min_y() ,
						x * grid_width , pix.max_y() );
		}
	}

	// left <-> right main line
	for ( gint  y = min_y  ;  y <= max_y  ; y ++ )
	{
		if ( std::abs( y ) % 2 == 0  &&  y != 0 )
		{
			pix.draw_line_absolute( grid_main_gc ,
						pix.min_x() , y * grid_width ,
						pix.max_x() , y * grid_width );
		}
	}


	// center up <-> down line
	if ( min_x * max_x <= 0 )
	{
		pix.draw_line_absolute( line_gc ,
					0 , pix.min_y() ,
					0 , pix.max_y() );

	}

	// center left <-> right line
	if ( min_y * max_y <= 0 )
	{
		pix.draw_line_absolute( line_gc ,
					pix.min_x() , 0 ,
					pix.max_x() , 0 );
	}
}

void   Field_Canvas::draw_circle_to_pixmap
			( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
			  const Monitor_View_Data &  monitor_view_data ,
			  gdouble  circle_radius )
{
	for ( int  i = 0  ;  i < MAX_PLAYER * 2  ;  i ++ )
	{
		if ( ! monitor_view_data.player[i].valid() )
		{
			continue;
		}

		pix.draw_arc_absolute_another_interface
			( player_circle_gc ,
			  false ,
			  monitor_view_data.player[i].x() ,
			  monitor_view_data.player[i].y() ,
			  circle_radius ,
			  circle_radius ,
			  0.0 , 360.0 );
	}

	pix.draw_arc_absolute_another_interface
		( ball_circle_gc ,
		  false ,
		  monitor_view_data.ball.x() ,
		  monitor_view_data.ball.y() ,
		  circle_radius ,
		  circle_radius ,
		  0.0 , 360.0 );
}

void   Field_Canvas::draw_monitor_ball_to_pixmap
	 ( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
	   const gdouble  ball_size ,
	   const D2_Vector &  ball ,
	   const Monitor_View_Data::Monitor_Play_Mode &  mode )
{
	S_Side_LR	side;

	switch( mode )
	{
	case Monitor_View_Data::Monitor_Play_Mode::Kick_In_Left:
	case Monitor_View_Data::Monitor_Play_Mode::Corner_Kick_Left:
	case Monitor_View_Data::Monitor_Play_Mode::Free_Kick_Left:
	case Monitor_View_Data::Monitor_Play_Mode::Offside_Right:
	case Monitor_View_Data::Monitor_Play_Mode::Kick_Off_Left:
		side = S_Side_LR::Left_Side;
		break;

	case Monitor_View_Data::Monitor_Play_Mode::Kick_In_Right:
	case Monitor_View_Data::Monitor_Play_Mode::Corner_Kick_Right:
	case Monitor_View_Data::Monitor_Play_Mode::Free_Kick_Right:
	case Monitor_View_Data::Monitor_Play_Mode::Offside_Left:
	case Monitor_View_Data::Monitor_Play_Mode::Kick_Off_Right:
		side = S_Side_LR::Right_Side;
		break;

	default:
		side = S_Side_LR::Unknown;
		break;
	}

	this -> draw_ball_to_pixmap( pix , ball_size , ball ,
				     false , D2_Vector::origin() ,
				     side );
}

void   Field_Canvas::draw_ball_to_pixmap
	 ( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
	   const gdouble  ball_size ,
	   const D2_Vector &  ball ,
	   bool  draw_velocity ,
	   const D2_Vector &  velocity ,
	   S_Side_LR  side ,
	   const char *  comment ,
	   bool  for_debug )
{
	Gdk_GC *	gc;

	if ( for_debug )
	{
		gc = &debug_ball_gc;
	}
	else
	{
		switch( side )
		{
		case S_Side_LR::Left_Side:
			gc = &left_field_player_gc;
			break;

		case S_Side_LR::Right_Side:
			gc = &right_field_player_gc;
			break;

		case S_Side_LR::Unknown:
		default:
			gc = &ball_gc;
			break;
		}
	}

	// ball
	pix.draw_arc_absolute_another_interface
		( *gc ,
		  true ,
		  ball.x() , ball.y() ,
		  ball_size , ball_size ,
		  0.0 , 360.0 );

	// ball edge
	pix.draw_arc_absolute_another_interface
		( ball_edge_gc ,
		  false ,
		  ball.x() , ball.y() ,
		  ball_size , ball_size ,
		  0.0 , 360.0 );

	// velocity
	if ( draw_velocity )
	{
		const	double	VELOCITY_RATE = 20.0;

		D2_Vector	to = ball + velocity * VELOCITY_RATE;

		this -> pixmap.draw_line_absolute( ball_velocity_gc ,
						   ball.x() , ball.y() ,
						   to.x() , to.y() );
	}

	// comment
	if ( comment )
	{
		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		comment_font.string_extents( comment ,
					     lbearing , rbearing ,
					     text_width ,
					     ascent , descent );

		pix.entity().draw_string(
			comment_font ,
			ball_comment_gc ,
			pix.x_coordinate_to_pixel
			  ( ball.x()
			    + ball_size * (inverse_x ? (-1) : (1)) )
			 - rbearing ,
			pix.y_coordinate_to_pixel
			 ( ball.y()
			   - ball_size * (inverse_y ? (-1) : (1)) ) ,
			comment );
	}
}

void   Field_Canvas::draw_player_to_pixmap
	 ( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
	   const Monitor_View_Data::Player_State &  player ,
	   bool  draw_view_angle ,
	   gdouble  player_size ,
	   const char *  comment ,
	   const char *  player_name ,
	   bool  draw_player_name ,
	   const char *  player_color ,
	   bool  use_player_color ,
	   bool  self_player ,
	   const char *  debug_message ,
	   bool  for_debug )
{
	if ( ! player.valid() )
	{
		return;
	}


	//
	// set player foreground gc
	//
	Gdk_GC *	fg_gc = &unknown_player_gc;
	Gdk_GC *	font_gc = &unknown_player_gc;
	Gdk_GC_Wrapper	player_color_fg_gc;

	if ( use_player_color )
	{
		player_color_fg_gc.create( get_window() );
		player_color_fg_gc.set_foreground( player_color );

		fg_gc = &player_color_fg_gc;
	}
	else if ( self_player )
	{
		fg_gc = &realized_self_player_gc;
	}
	else if ( for_debug )
	{
		switch( player.side() )
		{
		case S_Side_LR::Left_Side:
			fg_gc = &realized_left_player_gc;
			font_gc = &left_player_uniform_number_gc;
			break;

		case S_Side_LR::Right_Side:
			fg_gc = &realized_right_player_gc;
			font_gc = &right_player_uniform_number_gc;
			break;

		case S_Side_LR::Unknown:
			fg_gc = &unknown_player_gc;
			font_gc = &unknown_player_uniform_number_gc;
			break;
		}
	}
	else
	{
		switch( player.side() )
		{
		case S_Side_LR::Left_Side:
			if ( player.goalie() )
			{
				fg_gc = &left_goalie_gc;
			}
			else
			{
				fg_gc = &left_field_player_gc;
			}

			font_gc = &left_player_uniform_number_gc;
			break;

		case S_Side_LR::Right_Side:
			if ( player.goalie() )
			{
				fg_gc = &right_goalie_gc;
			}
			else
			{
				fg_gc = &right_field_player_gc;
			}

			font_gc = &right_player_uniform_number_gc;
			break;

		case S_Side_LR::Unknown:
			fg_gc = &unknown_player_gc;
			font_gc = &unknown_player_uniform_number_gc;
			break;
		}
	}


	//
	// set player background gc
	//
	Gdk_GC *	bg_gc;
	Gdk_GC		player_stamina_bg_gc;

	if ( player.have_stamina() )
	{
		Gdk_Color	player_stamina_bg_color;
		player_stamina_bg_color.set_grey_p
			  ( 1.0 - (player.stamina() / param.STAMINA_MAX()) );
		Gdk_Colormap::get_system().alloc( player_stamina_bg_color );

		player_stamina_bg_gc.create( get_window() );
		player_stamina_bg_gc.set_foreground( player_stamina_bg_color );

		bg_gc = &player_stamina_bg_gc;
	}
	else
	{
		bg_gc = &player_bg_gc;
	}


	if ( for_debug )
	{
		// draw front body
		pix.draw_arc_absolute_another_interface(
			*fg_gc ,
			true ,
			player.x() , player.y() ,
			player_size , player_size ,
			player.body_angle().normalize().degree() - 90.0 ,
			180.0 );

		// draw edge
		pix.draw_arc_absolute_another_interface(
			*fg_gc ,
			false ,
			player.x() , player.y() ,
			player_size , player_size ,
			0.0 , 360.0 );
	}
	else
	{
		// fill background
		pix.draw_arc_absolute_another_interface(
				*bg_gc ,
				true ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );

		// draw front body
		pix.draw_arc_absolute_another_interface(
			*fg_gc ,
			true ,
			player.x() , player.y() ,
			player_size , player_size ,
			player.body_angle().normalize().degree() - 90.0 ,
			180.0 );

		// draw edge
		pix.draw_arc_absolute_another_interface(
			*fg_gc ,
			false ,
			player.x() , player.y() ,
			player_size , player_size ,
			0.0 , 360.0 );


		if ( player.have_effort() && ! player.full_effort() )
		{
			// draw edge
			pix.draw_arc_absolute_another_interface(
				effort_reduced_gc ,
				false ,
				player.x() , player.y() ,
				player_size * 1.2 , player_size * 1.2 ,
				0.0 , 360.0 );
		}

		if ( player.kicking() )
		{
			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				true ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );

			pix.draw_arc_absolute_another_interface(
				*bg_gc ,
				true ,
				player.x() , player.y() ,
				player_size * (3.0 / 4.0) ,
				player_size * (3.0 / 4.0) ,
				player.body_angle().normalize().degree()
				  + 90.0 ,
				180.0 );

			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				true ,
				player.x() , player.y() ,
				player_size * (1.0 / 2.0) ,
				player_size * (1.0 / 2.0) ,
				0.0 , 360.0 );

			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				false ,
				player.x() , player.y() ,
				player_size * (5.0 / 8.0) ,
				player_size * (5.0 / 8.0) ,
				0.0 , 360.0 );
		}

		if ( player.kicking_fault() )
		{
			pix.draw_arc_absolute_another_interface(
				*bg_gc ,
				true ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );

			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				false ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );
		}

		if ( player.player_to_ball() )
		{
			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				false ,
				player.x() , player.y() ,
				player_size * 1.25 , player_size * 1.25 ,
				0.0 , 360.0 );
		}

		if ( player.ball_to_player() )
		{
			pix.draw_arc_absolute_another_interface(
				ball_gc ,
				false ,
				player.x() , player.y() ,
				player_size * 1.3 , player_size * 1.3 ,
				0.0 , 360.0 );
		}

		if ( player.catching() )
		{
			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				true ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );
		}

		if ( player.catching_fault() )
		{
			pix.draw_arc_absolute_another_interface(
				catch_fault_gc ,
				true ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );

			pix.draw_arc_absolute_another_interface(
				*fg_gc ,
				false ,
				player.x() , player.y() ,
				player_size , player_size ,
				0.0 , 360.0 );
		}
	}


	if ( draw_view_angle && player.have_head_angle() )
	{
		if ( player.have_view_width() )
		{
			// draw visible angle
			Angle	absolute_head_angle
				= (player.body_angle()
				   + player.head_angle()).normalize();

			pix.draw_line_absolute(
				player_view_angle_gc ,
				player.x() ,
				player.y() ,
				player.x() + (absolute_head_angle
					      - player.view_width() / 2.0)
					     .cos() * player_size * 0.8 ,
				player.y() + (absolute_head_angle
					      - player.view_width() / 2.0)
					     .sin() * player_size * 0.8 );

			pix.draw_line_absolute(
				player_view_angle_gc ,
				player.x() ,
				player.y() ,
				player.x() + (absolute_head_angle
					      + player.view_width() / 2.0)
					     .cos() * player_size * 0.8 ,
				player.y() + (absolute_head_angle
					      + player.view_width() / 2.0)
					     .sin() * player_size * 0.8 );

			pix.draw_arc_absolute_another_interface(
				player_view_angle_gc ,
				false ,
				player.x() , player.y() ,
				player_size * 0.8 , player_size * 0.8 ,
				(absolute_head_angle
				 - player.view_width() / 2.0)
				.normalize().degree() ,
				player.view_width().normalize().degree() );
		}
		else
		{
			// draw player face bar
			const	double	face_bar_ratio = 2.0;

			Angle	face_angle
				= player.body_angle() + player.head_angle();

			pix.draw_line_absolute
				( player_face_bar_gc ,
				  player.x() ,
				  player.y() ,
				  player.x()
				    + face_angle.cos()
					* player_size * face_bar_ratio ,
				  player.y()
				    + face_angle.sin()
					* player_size * face_bar_ratio );

		}
	}


	gint	uniform_number_ascent = 0;

	// draw uniform number or player name
	if ( player.uniform_number() >= 1 )
	{
		gchar	buf[ sizeof(gint) * CHAR_BIT + strlen("-") + 1 ];

		sprintf( buf , "%d" , player.uniform_number() );

		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	descent;

		player_identification_font.string_extents
						( buf ,
						  lbearing ,
						  rbearing ,
						  text_width ,
						  uniform_number_ascent ,
						  descent );

		pix.entity().draw_string
				( player_identification_font ,
				  *font_gc ,
				  pix.x_coordinate_to_pixel
				  ( player.x()
				    + player_size * (inverse_x ? (-1) : (1)) )
				  - rbearing ,
				  pix.y_coordinate_to_pixel( player.y() ) ,
				  buf );
	}

	if ( draw_player_name )
	{
		const gchar *	name;

		if ( player_name )
		{
			name = player_name;
		}
		else
		{
			name = "";
		}


		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		player_identification_font.string_extents( name ,
							   lbearing ,
							   rbearing ,
							   text_width ,
							   ascent ,
							   descent );

		gint	x_position;

		if ( rbearing - lbearing
		     >= pix.coordinate_distance_to_pixel( player_size ) )
		{
			x_position = pix.x_coordinate_to_pixel( player.x() )
				     - (rbearing - lbearing) / 2;
		}
		else
		{
			x_position
				= pix.x_coordinate_to_pixel
				  ( player.x()
				    + player_size * (inverse_x ? (-1) : (1)) )
				  - rbearing;
		}

		pix.entity().draw_string
				( player_identification_font ,
				  player_name_gc ,
				  x_position ,
				  pix.y_coordinate_to_pixel( player.y() )
				    - (uniform_number_ascent + descent + 1) ,
				  name );
	}


	// draw comment
	gint	comment_ascent = 0;
	if ( comment )
	{
		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		comment_font.string_extents( comment ,
					     lbearing , rbearing ,
					     text_width ,
					     ascent , descent );

		comment_ascent = ascent;

		pix.entity().draw_string
		      ( comment_font ,
			*font_gc ,
			pix.x_coordinate_to_pixel
			  ( player.x()
			    + player_size * (inverse_x ? (-1) : (1)) )
			 - rbearing ,
			pix.y_coordinate_to_pixel
			 ( player.y()
			   - player_size * (inverse_y ? (-1) : (1)) ) ,
			comment );
	}


	// draw debug message
	if ( debug_message )
	{
		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		comment_font.string_extents( debug_message ,
					     lbearing , rbearing ,
					     text_width ,
					     ascent , descent );

		pix.entity().draw_string
		      ( comment_font ,
			*font_gc ,
			pix.x_coordinate_to_pixel
			  ( player.x()
			    + player_size * (inverse_x ? (-1) : (1)) )
			 - rbearing ,
			pix.y_coordinate_to_pixel
			 ( player.y()
			   - (player_size + comment_ascent)
				   * (inverse_y ? (-1) : (1)) ) ,
			debug_message );
	}
}


void   Field_Canvas::draw_offside_line( gdouble  deffence_line ,
					gdouble  offence_line ,
					S_Side_LR  side )
{
	switch( side )
	{
	case S_Side_LR::Left_Side:
		{
			double	left_line  = (+ deffence_line);
			double	right_line = (+ offence_line);

			pixmap.draw_line_absolute
				( right_field_player_gc ,
				  right_line , - FIELD_HALF_WIDTH ,
				  right_line , + FIELD_HALF_WIDTH );

			pixmap.draw_line_absolute
				( left_field_player_gc ,
				  left_line , - FIELD_HALF_WIDTH ,
				  left_line , + FIELD_HALF_WIDTH );
		}
		break;

	case S_Side_LR::Right_Side:
		{
			double	left_line  = (- offence_line);
			double	right_line = (- deffence_line);

			pixmap.draw_line_absolute
				( left_field_player_gc ,
				  left_line , - FIELD_HALF_WIDTH ,
				  left_line , + FIELD_HALF_WIDTH );

			pixmap.draw_line_absolute
				( right_field_player_gc ,
				  right_line , - FIELD_HALF_WIDTH ,
				  right_line , + FIELD_HALF_WIDTH );

		}
		break;

	case S_Side_LR::Unknown:
	default:
		return;
		break;
	}
}

void   Field_Canvas::draw_offside_line
			( const Monitor_View_Data &  monitor_view_data )
{
	std::multiset<gdouble>	left_x_list;
	std::multiset<gdouble>	right_x_list;

	for ( int i = 0  ;  i < SServer_Monitor_Log_Format::MAX_PLAYER * 2  ;
	      i ++ )
	{
		const Monitor_View_Data::Player_State &
			player = monitor_view_data.player[i];

		if ( player.valid() )
		{
			S_Side_LR	s = player.side();

			if ( s == S_Side_LR::Left_Side )
			{
				left_x_list.insert( player.x() );
			}
			else if ( s == S_Side_LR::Right_Side )
			{
				right_x_list.insert( player.x() );
			}
		}
	}


	//
	// Left Team Offside Line
	//
	gdouble	left_defencive_offside_line = 0.0;

	if ( left_x_list.size() >= 2 )
	{
		std::multiset<gdouble>::iterator	it;
		it = left_x_list.begin();
		it ++;

		if ( (*it) < left_defencive_offside_line )
		{
			left_defencive_offside_line = (*it);
		}
	}

	if ( monitor_view_data.ball.x() < left_defencive_offside_line )
	{
		left_defencive_offside_line = monitor_view_data.ball.x();
	}

	pixmap.draw_line_absolute
		       ( left_field_player_gc ,
			 left_defencive_offside_line , - FIELD_HALF_WIDTH ,
			 left_defencive_offside_line , + FIELD_HALF_WIDTH );


	//
	// Right Team Offside Line
	//
	gdouble	right_defencive_offside_line = 0.0;

	if ( right_x_list.size() >= 2 )
	{
		std::multiset<gdouble>::iterator	it;
		it = right_x_list.end();
		it --;
		it --;

		if ( (*it) > right_defencive_offside_line )
		{
			right_defencive_offside_line = (*it);
		}
	}

	if ( monitor_view_data.ball.x() > right_defencive_offside_line )
	{
		right_defencive_offside_line = monitor_view_data.ball.x();
	}

	pixmap.draw_line_absolute
		       ( right_field_player_gc ,
			 right_defencive_offside_line , - FIELD_HALF_WIDTH ,
			 right_defencive_offside_line , + FIELD_HALF_WIDTH );
}

void   Field_Canvas::draw_popup_to_pixmap
	 ( Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
	   bool  use_teamname ,
	   const Monitor_View_Data::Monitor_Play_Mode &  mode ,
	   const string &  left_teamname ,
	   const string &  right_teamname )
{
	if ( mode == Monitor_View_Data::Monitor_Play_Mode::Offside_Left
	  || mode == Monitor_View_Data::Monitor_Play_Mode::Offside_Right
	  || mode == Monitor_View_Data::Monitor_Play_Mode::Goal_Left
	  || mode == Monitor_View_Data::Monitor_Play_Mode::Goal_Right )
	{
		//
		// draw message
		//
		string	popup_message;

		if ( mode == Monitor_View_Data::Monitor_Play_Mode
			     ::Offside_Left )
		{
			if ( use_teamname
			     && left_teamname.length() != 0 )
			{
#if 0
				popup_message = string("Offside \"")
						+ left_teamname + "\" !!";
#else
				popup_message = string("Offside ")
						+ left_teamname + " !!";
#endif
			}
			else
			{
				popup_message = "Offside Left !!";
			}
		}
		else if( mode == Monitor_View_Data::Monitor_Play_Mode
				 ::Offside_Right )
		{
			if ( use_teamname
			     && right_teamname.length() != 0 )
			{
#if 0
				popup_message = string("Offside \"")
						+ right_teamname + "\" !!";
#else
				popup_message = string("Offside ")
						+ right_teamname + " !!";
#endif
			}
			else
			{
				popup_message = "Offside Right !!";
			}
		}
		else if ( mode == Monitor_View_Data::Monitor_Play_Mode
				  ::Goal_Left
		       || mode == Monitor_View_Data::Monitor_Play_Mode
				  ::Goal_Right )
		{
			popup_message = "GOAL !!";
		}

		gint	lbearing;
		gint	rbearing;
		gint	text_width;
		gint	ascent;
		gint	descent;

		popup_font.string_extents( popup_message ,
					   lbearing , rbearing ,
					   text_width ,
					   ascent , descent );

		gint	message_width  = lbearing + rbearing;
		gint	message_height = ascent + descent;

		pix.entity().draw_string(
		      popup_font ,
		      popup_string_gc ,
		      (pix.width() - message_width) / 2 ,
		      (pix.height() - message_height) / 2 + ascent ,
		      popup_message );
	}
}

void   Field_Canvas::draw_auto_drop_count_down_to_pixmap(
			  Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
			  long  auto_drop_rest_count )
{
	//
	// draw "Auto Kick Off: "
	//
	const string	auto_drop_string = "Auto Drop: ";

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		this -> auto_drop_message_x ,
		this -> auto_drop_message_y ,
		auto_drop_string );


	//
	// draw count down number
	//
	gchar	num[ sizeof(gint) * CHAR_BIT + strlen("-") + 1 ];
	sprintf( num , "%ld" ,
		 static_cast<long>(auto_drop_rest_count) );

	gint	padding	= std::max( MAX_AUTO_DROP_COLUMN
				    - static_cast<gint>( strlen( num ) ) - 1 ,
				    0 );

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		(this -> auto_drop_message_x)
		  + (this -> auto_drop_message_width)
		  + (this -> max_number_char_width) * padding ,
		this -> auto_drop_message_y ,
		num );
}

void   Field_Canvas::draw_auto_kick_off_count_down_to_pixmap(
			  Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
			  long  auto_kick_off_rest_count )
{
	//
	// draw "Auto Kick Off: "
	//
	const string	auto_kick_off_string = "Auto Kick Off: ";

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		this -> auto_kick_off_message_x ,
		this -> auto_kick_off_message_y ,
		auto_kick_off_string );


	//
	// draw count down number
	//
	gchar	num[ sizeof(gint) * CHAR_BIT + strlen("-") + 1 ];
	sprintf( num , "%ld" ,
		 static_cast<long>(auto_kick_off_rest_count) );

	gint	padding	= std::max( MAX_AUTO_KICK_OFF_COLUMN
				    - static_cast<gint>( strlen( num ) ) - 1 ,
				    0 );

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		(this -> auto_kick_off_message_x)
		  + (this -> auto_kick_off_message_width)
		  + (this -> max_number_char_width) * padding ,
		this -> auto_kick_off_message_y ,
		num );
}

void   Field_Canvas::draw_auto_quit_count_down_to_pixmap(
			  Coordinate_View_Pixmap<Transparent_Pixmap> &  pix ,
			  long  auto_quit_rest_count )
{
	//
	// draw "Auto Quit: "
	//
	const string	auto_quit_string = "Auto Quit: ";

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		this -> auto_quit_message_x ,
		this -> auto_quit_message_y ,
		auto_quit_string );


	//
	// draw count down number
	//
	gchar	num[ sizeof(gint) * CHAR_BIT + strlen("-") + 1 ];
	sprintf( num , "%ld" ,
		 static_cast<long>(auto_quit_rest_count) );

	gint	padding	= std::max( MAX_AUTO_QUIT_COLUMN
				    - static_cast<gint>( strlen( num ) ) - 1 ,
				    0 );

	pix.entity().draw_string(
		popup_font ,
		popup_string_gc ,
		(this -> auto_quit_message_x)
		  + (this -> auto_quit_message_width)
		  + (this -> max_number_char_width) * padding ,
		this -> auto_quit_message_y ,
		num );
}

void   Field_Canvas::draw_field_environment_to_pixmap
	 ( Coordinate_View_Pixmap<Gdk_Pixmap> &  pix )
{
	// ground
	pix.entity().draw_rectangle( ground_gc ,
				     true ,
				     0 , 0 ,
				     width() , height() );

	// field
	pix.draw_rectangle_absolute_another_interface(
				 field_gc ,
				 true ,
				 - FIELD_HALF_LENGTH , - FIELD_HALF_WIDTH ,
				 + FIELD_HALF_LENGTH , + FIELD_HALF_WIDTH );

	// field line
	pix.draw_rectangle_absolute_another_interface(
		line_gc ,
		false ,
		- FIELD_HALF_LENGTH , - FIELD_HALF_WIDTH ,
		+ FIELD_HALF_LENGTH , + FIELD_HALF_WIDTH );

	// goal(Left)
	pix.draw_rectangle_absolute_another_interface(
		goal_gc ,
		true ,
		- FIELD_HALF_LENGTH - 3.0 , - param.GOAL_WIDTH() / 2.0 ,
		- FIELD_HALF_LENGTH       , + param.GOAL_WIDTH() / 2.0 );

	// goal(Right)
	pix.draw_rectangle_absolute_another_interface(
		goal_gc ,
		true ,
		+ FIELD_HALF_LENGTH ,       - param.GOAL_WIDTH() / 2.0 ,
		+ FIELD_HALF_LENGTH + 3.0 , + param.GOAL_WIDTH() / 2.0 );

	// center circle
	pix.draw_arc_absolute_another_interface(
		line_gc ,
		false ,
		0.0 , 0.0 ,
		CENTER_CIRCLE_R , CENTER_CIRCLE_R ,
		0.0 , 360.0 );

	// center line
	pix.draw_line_absolute( line_gc ,
				0.0 , - FIELD_HALF_WIDTH ,
				0.0 , + FIELD_HALF_WIDTH );

	// penalty area(Left)
	pix.draw_rectangle_absolute_another_interface(
		line_gc ,
		false ,
		- FIELD_HALF_LENGTH ,
		- PENALTY_AREA_HALF_WIDTH ,
		- (FIELD_HALF_LENGTH - PENALTY_AREA_LENGTH) ,
		+ PENALTY_AREA_HALF_WIDTH );

	// penalty area(Right)
	pix.draw_rectangle_absolute_another_interface(
		line_gc ,
		false ,
		+ (FIELD_HALF_LENGTH -  PENALTY_AREA_LENGTH) ,
		- PENALTY_AREA_HALF_WIDTH ,
		+ FIELD_HALF_LENGTH ,
		+ PENALTY_AREA_HALF_WIDTH );

	// goal area(Left)
	pix.draw_rectangle_absolute_another_interface(
		line_gc ,
		false ,
		- FIELD_HALF_LENGTH ,
		- GOAL_AREA_HALF_WIDTH ,
		- (FIELD_HALF_LENGTH - GOAL_AREA_LENGTH) ,
		+ GOAL_AREA_HALF_WIDTH );

	// goal area(Right)
	pix.draw_rectangle_absolute_another_interface(
		line_gc ,
		false ,
		+ (FIELD_HALF_LENGTH -  GOAL_AREA_LENGTH) ,
		- GOAL_AREA_HALF_WIDTH ,
		+ FIELD_HALF_LENGTH ,
		+ GOAL_AREA_HALF_WIDTH );

	// penalty area arc(Left)
	pix.draw_arc_absolute_another_interface(
		line_gc ,
		false ,
		- (FIELD_HALF_LENGTH - PENALTY_SPOT_DIST) , 0.0 ,
		CENTER_CIRCLE_R , CENTER_CIRCLE_R ,
		- PENALTY_ARC_DIGREE , PENALTY_ARC_DIGREE * 2.0 );

	// penalty area arc(Right)
	pix.draw_arc_absolute_another_interface(
		line_gc ,
		false ,
		+ (FIELD_HALF_LENGTH - PENALTY_SPOT_DIST) , 0.0 ,
		CENTER_CIRCLE_R , CENTER_CIRCLE_R ,
		- PENALTY_ARC_DIGREE + 180.0 , PENALTY_ARC_DIGREE * 2.0 );

	// corner arc(Left Top)
	pix.draw_arc_absolute_another_interface
		( line_gc ,
		  false ,
		  - FIELD_HALF_LENGTH , FIELD_HALF_WIDTH ,
		  CORNER_CIRCLE_R , CORNER_CIRCLE_R ,
		  270.0 , 90.0 );

	// corner arc(Left Bottom)
	pix.draw_arc_absolute_another_interface
		( line_gc ,
		  false ,
		  - FIELD_HALF_LENGTH , - FIELD_HALF_WIDTH ,
		  CORNER_CIRCLE_R , CORNER_CIRCLE_R ,
		  0.0 , 90.0 );

	// corner arc(Right Top)
	pix.draw_arc_absolute_another_interface
		( line_gc ,
		  false ,
		  + FIELD_HALF_LENGTH , + FIELD_HALF_WIDTH ,
		  CORNER_CIRCLE_R , CORNER_CIRCLE_R ,
		  180.0 , 90.0 );

	// corner arc(Right Bottom)
	pix.draw_arc_absolute_another_interface
		( line_gc ,
		  false ,
		  + FIELD_HALF_LENGTH , - FIELD_HALF_WIDTH ,
		  CORNER_CIRCLE_R , CORNER_CIRCLE_R ,
		  90.0 , 90.0 );

	// corner arc(Right Bottom)
	pix.entity().draw_rectangle
		( frame_gc ,
		  false ,
		  0 ,
		  0 ,
		  std::min<gint>( pix.width()  - 1 , 0 ) ,
		  std::min<gint>( pix.height() - 1 , 0 ) );
}




void   Field_Canvas::draw_target_point( const D2_Vector &  point ,
					S_Side_LR  self_side )
{
	const	double	x_size = 1.0;

	D2_Vector	p;

	if ( self_side == S_Side_LR::Right_Side )
	{
		p = - point;
	}
	else
	{
		p = + point;
	}

	//
	// draw 'X' mark
	//
	pixmap.draw_line_absolute( target_point_gc ,
				   p.x() - x_size ,  p.y() - x_size ,
				   p.x() + x_size ,  p.y() + x_size );

	pixmap.draw_line_absolute( target_point_gc ,
				   p.x() - x_size ,  p.y() + x_size ,
				   p.x() + x_size ,  p.y() - x_size );
}
