package org.util.html.render;


import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import org.util.html.objects.*;
import org.util.html.event.*;

public class HTMLDocumentRenderer {

    private HTMLDocument doc_;
    private int offset_x_ = 0;
    private int offset_y_ = 0;
    private boolean finished_layout_;
    private int screen_w_ = 1;
    private int screen_h_ = 1;
    private int width_;
    private int height_;
    private JPanel panel_;
    private JScrollPane sp_;

    private ArrayList<HTMLDocumentRendererListener> listener_list_ = new ArrayList<HTMLDocumentRendererListener>();
    
    public HTMLDocumentRenderer(HTMLDocument doc) {
	doc_ = doc;
	doc_.addHTMLListener(new HTMLRenderListener());
	panel_ = new JPanel(new HTMLLayoutManager());
    }

    private volatile boolean do_layout_running_;
    private volatile boolean do_layout_cancel_;

    public void setScrollPane(JScrollPane sp) {
	sp_ = sp;
    }

    public boolean doLayout(int w, int h) {
	
	if(do_layout_running_) {
	    do_layout_cancel_ = true;
	    return false;
	}
	do_layout_running_ = true;
	do_layout_cancel_ = false;


	HTMLObject[] list = doc_.getObjectList();
	int now_x = 0;
	int now_y = 0;
	int line_height = 0;
	HTMLObject last = null;
	for(HTMLObject obj : list) {
	    
	    if(!obj.isLayouted()) {
		if(last != null) {
		    now_x = last.getNextX();
		    now_y = last.getNextY();
		}

		JComponent c = obj.getPanel();
		Dimension cd = c.getPreferredSize();
		int cw = (int)cd.getWidth();
		int ch = (int)cd.getHeight();

		if(now_x + cw > w) {
		    now_x = 0;
		    now_y += line_height;
		    line_height = ch;
		} else {
		    line_height = Math.max(ch, line_height);
		}
		c.setLocation(now_x, now_y);
		c.setSize(cw, ch);
		c.setVisible(true);
		obj.setLayouted(true);

		now_x += cw;
		obj.setNextXY(now_x, now_y);
	    }
	    last = obj;
	}
	width_ = w;
	if(list.length>0) {
	    JComponent lc = list[list.length-1].getPanel();
	    height_ = lc.getY()+lc.getHeight();
	} else {
	    height_ = 0;
	}
	finished_layout_ = true;
	screen_w_ = w;
	screen_h_ = h;

	do_layout_running_ = false;
	if(do_layout_cancel_) {
	    doLayout(w, h);
	}
	return true;
    }

    public int getOffsetX() {
	return offset_x_;
    }
    public int getOffsetY() {
	return offset_y_;
    }
    public int getWidth() {
	return width_;
    }
    public int getHeight() {
	return height_;
    }


    public void resized(int w, int h) {
	HTMLObject[] list = doc_.getObjectList();
	if(list.length>1)
	    for(int i=0; i<list.length; i++)
		list[i].setLayouted(false);
	doLayout(w, h);
	requestRepaint();
    }

    public void addHTMLDocumentRendererListener(HTMLDocumentRendererListener listener){
	listener_list_.add(listener);
    }
    public void removeHTMLDocumentRendererListener(HTMLDocumentRendererListener listener){
	listener_list_.remove(listener);
    }
    
    public JComponent getPanel() {
	return panel_;
    }

    private class HTMLRenderListener implements HTMLListener {
	public void cleared(HTMLDocument doc) {
	    finished_layout_ = false;
	    requestRepaint();
	    panel_.removeAll();
	    panel_.revalidate();
	}
	public void changed(HTMLDocument doc, HTMLObject obj) {
	    finished_layout_ = false;
	    for(HTMLObject o : doc_.getObjectList()) {
		o.setLayouted(false);
	    }
	    panel_.revalidate();
	    requestRepaint();
	}
	public void added(HTMLDocument doc, HTMLObject obj) {
	    finished_layout_ = false;
	    panel_.add(obj.getPanel());
	    panel_.revalidate();
	    requestRepaint();
	}
	public void removed(HTMLDocument doc, HTMLObject obj) {
	}
    }
    private void requestRepaint() {
	for(HTMLDocumentRendererListener listener : listener_list_)
	    listener.repaint();
    }



    private class HTMLLayoutManager implements LayoutManager {
	public void addLayoutComponent(String name, Component comp) {
	    
	}
	public void layoutContainer(Container parent) {
	    int w = parent.getWidth();
	    int h = parent.getHeight();
	    doLayout(w, h);
	}
	public Dimension minimumLayoutSize(Container parent) {
	    return new Dimension(100, 100);
	}
	public Dimension preferredLayoutSize(Container parent) {
	    Dimension screen_size = sp_.getSize();
	    int w = sp_.getWidth()-sp_.getVerticalScrollBar().getWidth()-3;
	    doLayout(w, 1);
	    return new Dimension(w, height_);
	}
        public void removeLayoutComponent(Component comp) {
	    
	}
    }
}
