package libsvm.wrapper;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Properties;

import libsvm.svm_parameter;

/**
 * <pre>
 * SVMパラメータ
 * これによってSVMの特性が大きく変化する．
 * (svm_parameter)
 * </pre>
 *
 * @author hirai
 */
public class SvmSettingParameter {

	/**
	 * SVMの種類
	 */
	public SvmType svmType;

	/**
	 * カーネルの種類
	 */
	public KernelType kernelType;

	/**
	 * for POLY
	 */
	public int degree;

	/**
	 * for POLY / RGF / SIGMOID
	 */
	public double gamma;

	/**
	 * for POLY / SIGMOID
	 */
	public double coef0;

	/**
	 * カーネルのキャッシュのサイズ(メガバイト単位)
	 */
	public double cacheSize;

	/**
	 * 停止基準(通常0.00001 nu-SVC，その他の場合0.001)
	 */
	public double eps;

	/**
	 * 制約違反の値(通常1から1000)
	 */
	public double cost;

	/**
	 * <pre>
	 * 複数のクラスのペナルティを変更するために使われる．
	 * (クラスのための重みが変更されないならば，それらは1にセットされる)
	 * 片寄った入力データを用いたトレーニング分類するとき，または，非対称の誤った分類コストをしようしたときに役にたつ．
	 *
	 * リストweightLabelとweightの要素の数
	 * 各々のweight[i]はweightLabel[i]と一致する．
	 *
	 * weightLabel[i]のペナルティがweight[i]倍にスケールされることを意味する．
	 *
	 * もしペナルティを他のクラスと交換したくないならば，
	 * nuWeightを0にセットする．
	 * </pre>
	 */
	public int nrWeight;
	public int[] weightLabel;
	public double[] weight;

	/**
	 * NU_SVM，NU_SVR，ONE_CLASSのパラメータ
	 */
	public double nu;

	/**
	 * EPSILON_SVM回帰のepsilon-insensitive損失関数のepsilon
	 */
	public double p;

	/**
	 * シュリンク(圧縮)する場合1，そうでない場合0
	 */
	public int shrinking;

	/**
	 * モデルと可能性の情報を得る場合は1，そうでない場合0
	 */
	public int probability;

	/**
	 * <pre>
	 * コンストラクタ
	 *
	 * 適当に各パラメータを初期化
	 * </pre>
	 */
	public SvmSettingParameter() {

		svmType = SvmType.C_SVC;
		kernelType = KernelType.LINEAR;
		degree = 2;
		gamma = 0.2;
		coef0 = 1;
		nu = 0.2;
		cacheSize = 100;
		cost = 100;
		eps = 1e-3;
		p = 0.1;
		shrinking = 1;
		probability = 0;
		nrWeight = 0;
		weightLabel = null;
		weight = null;
	}

	/**
	 * <pre>
	 * コンストラクタ
	 *
	 * 適当に各パラメータを初期化
	 * </pre>
	 *
	 * @param propertyFilePath プロパティファイルパス
	 */
	public SvmSettingParameter(String propertyFilePath) {

		this();

		Properties prop = new Properties();
		FileInputStream fis = null;

		try {

			fis = new FileInputStream(propertyFilePath);
			prop.load(fis);

			svmType = SvmType.toEnum(prop.getProperty("svmType", "0"));
			kernelType = KernelType.toEnum(prop.getProperty("kernelType", "0"));
			degree = Integer.valueOf(prop.getProperty("degree", "2"));
			gamma = Double.valueOf(prop.getProperty("gamma", "0.2"));
			coef0 = Double.valueOf(prop.getProperty("coef0", "1"));
			nu = Double.valueOf(prop.getProperty("nu", "0.2"));
			cacheSize = Double.valueOf(prop.getProperty("cacheSize", "100"));
			cost = Double.valueOf(prop.getProperty("cost", "100"));
			eps = Double.valueOf(prop.getProperty("eps", "1e-3"));
			p = Double.valueOf(prop.getProperty("p", "0.1"));
			shrinking = Integer.valueOf(prop.getProperty("shrinking", "1"));
			probability = Integer.valueOf(prop.getProperty("probability", "0"));
			nrWeight = Integer.valueOf(prop.getProperty("nrWeight", "0"));

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {

			try {
				fis.close();
			} catch (IOException e) {
				// TODO なんか処理する
				e.printStackTrace();
			}
		}
	}

	/**
	 * 利用可能な状態に変換します．
	 *
	 * @return
	 */
	public svm_parameter toUsable() {

		svm_parameter res = new svm_parameter();

		res.svm_type = this.svmType.toInt();
		res.kernel_type = this.kernelType.toInt();
		res.degree = this.degree;
		res.gamma = this.gamma;
		res.coef0 = this.coef0;
		res.nu = this.nu;
		res.cache_size = this.cacheSize;
		res.C = this.cost;
		res.eps = this.eps;
		res.p = this.p;
		res.shrinking = this.shrinking;
		res.probability = this.probability;
		res.nr_weight = this.nrWeight;
		res.weight_label = this.weightLabel;
		res.weight = this.weight;

		return res;
	}
}
