/*
 * Decompiled with CFR 0.152.
 */
package maps.osm;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import maps.osm.OSMBuilding;
import maps.osm.OSMException;
import maps.osm.OSMNode;
import maps.osm.OSMRoad;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

public class OSMMap {
    private static final Collection<String> ROAD_MARKERS = new HashSet<String>();
    private Map<Long, OSMNode> nodes;
    private Map<Long, OSMRoad> roads;
    private Map<Long, OSMBuilding> buildings;
    private boolean boundsCalculated;
    private double minLat;
    private double maxLat;
    private double minLon;
    private double maxLon;

    public OSMMap() {
        this.boundsCalculated = false;
        this.nodes = new HashMap<Long, OSMNode>();
        this.roads = new HashMap<Long, OSMRoad>();
        this.buildings = new HashMap<Long, OSMBuilding>();
    }

    public OSMMap(Document doc) throws OSMException {
        this();
        this.read(doc);
    }

    public OSMMap(File file) throws OSMException, DocumentException, IOException {
        this();
        SAXReader reader = new SAXReader();
        Document doc = reader.read(file);
        this.read(doc);
    }

    public OSMMap(OSMMap other, double minLat, double minLon, double maxLat, double maxLon) {
        this.minLat = minLat;
        this.minLon = minLon;
        this.maxLat = maxLat;
        this.maxLon = maxLon;
        this.boundsCalculated = true;
        this.nodes = new HashMap<Long, OSMNode>();
        this.roads = new HashMap<Long, OSMRoad>();
        this.buildings = new HashMap<Long, OSMBuilding>();
        for (OSMNode oSMNode : other.nodes.values()) {
            double lat = oSMNode.getLatitude();
            double lon = oSMNode.getLongitude();
            long id = oSMNode.getID();
            if (!(lat >= minLat) || !(lat <= maxLat) || !(lon >= minLon) || !(lon <= maxLon)) continue;
            this.nodes.put(id, new OSMNode(id, lat, lon));
        }
        for (OSMRoad oSMRoad : other.roads.values()) {
            ArrayList<Long> ids = new ArrayList<Long>(oSMRoad.getNodeIDs());
            Iterator it = ids.iterator();
            while (it.hasNext()) {
                Long nextID = (Long)it.next();
                if (this.nodes.containsKey(nextID)) continue;
                it.remove();
            }
            if (ids.isEmpty()) continue;
            this.roads.put(oSMRoad.getID(), new OSMRoad(oSMRoad.getID(), ids));
        }
        for (OSMBuilding oSMBuilding : other.buildings.values()) {
            boolean allFound = true;
            for (Long nextID : oSMBuilding.getNodeIDs()) {
                if (this.nodes.containsKey(nextID)) continue;
                allFound = false;
            }
            if (!allFound) continue;
            this.buildings.put(oSMBuilding.getID(), new OSMBuilding(oSMBuilding.getID(), new ArrayList<Long>(oSMBuilding.getNodeIDs())));
        }
    }

    public void read(Document doc) throws OSMException {
        Element e;
        this.boundsCalculated = false;
        this.nodes = new HashMap<Long, OSMNode>();
        this.roads = new HashMap<Long, OSMRoad>();
        this.buildings = new HashMap<Long, OSMBuilding>();
        Element root = doc.getRootElement();
        if (!"osm".equals(root.getName())) {
            throw new OSMException("Invalid map file: root element must be 'osm', not " + root.getName());
        }
        for (Object next : root.elements("node")) {
            e = (Element)next;
            OSMNode node = this.processNode(e);
        }
        for (Object next : root.elements("way")) {
            e = (Element)next;
            this.processWay(e);
        }
    }

    public Document toXML() {
        Element node;
        Element root = DocumentHelper.createElement((String)"osm");
        Element bounds = root.addElement("bounds");
        this.calculateBounds();
        bounds.addAttribute("minlat", String.valueOf(this.minLat));
        bounds.addAttribute("maxlat", String.valueOf(this.maxLat));
        bounds.addAttribute("minlon", String.valueOf(this.minLon));
        bounds.addAttribute("maxlon", String.valueOf(this.maxLon));
        for (OSMNode oSMNode : this.nodes.values()) {
            node = root.addElement("node");
            node.addAttribute("id", String.valueOf(oSMNode.getID()));
            node.addAttribute("lat", String.valueOf(oSMNode.getLatitude()));
            node.addAttribute("lon", String.valueOf(oSMNode.getLongitude()));
        }
        for (OSMRoad oSMRoad : this.roads.values()) {
            node = root.addElement("way");
            node.addAttribute("id", String.valueOf(oSMRoad.getID()));
            for (Long nextID : oSMRoad.getNodeIDs()) {
                node.addElement("nd").addAttribute("ref", String.valueOf(nextID));
            }
            node.addElement("tag").addAttribute("k", "highway").addAttribute("v", "primary");
        }
        for (OSMBuilding oSMBuilding : this.buildings.values()) {
            node = root.addElement("way");
            node.addAttribute("id", String.valueOf(oSMBuilding.getID()));
            for (Long nextID : oSMBuilding.getNodeIDs()) {
                node.addElement("nd").addAttribute("ref", String.valueOf(nextID));
            }
            node.addElement("tag").addAttribute("k", "building").addAttribute("v", "yes");
        }
        return DocumentHelper.createDocument((Element)root);
    }

    public double getMinLongitude() {
        this.calculateBounds();
        return this.minLon;
    }

    public double getMaxLongitude() {
        this.calculateBounds();
        return this.maxLon;
    }

    public double getCentreLongitude() {
        this.calculateBounds();
        return (this.maxLon + this.minLon) / 2.0;
    }

    public double getMinLatitude() {
        this.calculateBounds();
        return this.minLat;
    }

    public double getMaxLatitude() {
        this.calculateBounds();
        return this.maxLat;
    }

    public double getCentreLatitude() {
        this.calculateBounds();
        return (this.maxLat + this.minLat) / 2.0;
    }

    public Collection<OSMNode> getNodes() {
        return new HashSet<OSMNode>(this.nodes.values());
    }

    public void removeNode(OSMNode node) {
        this.nodes.remove(node.getID());
    }

    public OSMNode getNode(Long id) {
        return this.nodes.get(id);
    }

    public OSMNode getNearestNode(double lat, double lon) {
        double smallest = Double.MAX_VALUE;
        OSMNode best = null;
        for (OSMNode next : this.nodes.values()) {
            double d2;
            double d1 = next.getLatitude() - lat;
            double d = d1 * d1 + (d2 = next.getLongitude() - lon) * d2;
            if (!(d < smallest)) continue;
            best = next;
            smallest = d;
        }
        return best;
    }

    public void replaceNode(OSMNode old, OSMNode replacement) {
        for (OSMRoad r : this.roads.values()) {
            r.replace(old.getID(), replacement.getID());
        }
        for (OSMBuilding b : this.buildings.values()) {
            b.replace(old.getID(), replacement.getID());
        }
        this.removeNode(old);
    }

    public Collection<OSMRoad> getRoads() {
        return new HashSet<OSMRoad>(this.roads.values());
    }

    public void removeRoad(OSMRoad road) {
        this.roads.remove(road.getID());
    }

    public Collection<OSMBuilding> getBuildings() {
        return new HashSet<OSMBuilding>(this.buildings.values());
    }

    public void removeBuilding(OSMBuilding building) {
        this.buildings.remove(building.getID());
    }

    private void calculateBounds() {
        if (this.boundsCalculated) {
            return;
        }
        this.minLat = Double.POSITIVE_INFINITY;
        this.maxLat = Double.NEGATIVE_INFINITY;
        this.minLon = Double.POSITIVE_INFINITY;
        this.maxLon = Double.NEGATIVE_INFINITY;
        for (OSMNode node : this.nodes.values()) {
            this.minLat = Math.min(this.minLat, node.getLatitude());
            this.maxLat = Math.max(this.maxLat, node.getLatitude());
            this.minLon = Math.min(this.minLon, node.getLongitude());
            this.maxLon = Math.max(this.maxLon, node.getLongitude());
        }
        this.boundsCalculated = true;
    }

    private OSMNode processNode(Element e) {
        long id = Long.parseLong(e.attributeValue("id"));
        double lat = Double.parseDouble(e.attributeValue("lat"));
        double lon = Double.parseDouble(e.attributeValue("lon"));
        OSMNode node = new OSMNode(id, lat, lon);
        this.nodes.put(id, node);
        return node;
    }

    private void processWay(Element e) {
        long id = Long.parseLong(e.attributeValue("id"));
        ArrayList<Long> ids = new ArrayList<Long>();
        for (Object next : e.elements("nd")) {
            Element nd = (Element)next;
            Long nextID = Long.parseLong(nd.attributeValue("ref"));
            ids.add(nextID);
        }
        boolean road = false;
        boolean building = false;
        for (Object next : e.elements("tag")) {
            Element tag = (Element)next;
            building = building || this.tagSignifiesBuilding(tag);
            road = road || this.tagSignifiesRoad(tag);
        }
        if (building) {
            this.buildings.put(id, new OSMBuilding(id, ids));
        } else if (road) {
            this.roads.put(id, new OSMRoad(id, ids));
        }
    }

    private boolean tagSignifiesRoad(Element tag) {
        String key = tag.attributeValue("k");
        String value = tag.attributeValue("v");
        if (!"highway".equals(key)) {
            return false;
        }
        return ROAD_MARKERS.contains(value);
    }

    private boolean tagSignifiesBuilding(Element tag) {
        String key = tag.attributeValue("k");
        String value = tag.attributeValue("v");
        if ("building".equals(key)) {
            return "yes".equals(value);
        }
        if ("rcr:building".equals(key)) {
            return "1".equals(value);
        }
        return false;
    }

    static {
        ROAD_MARKERS.add("motorway");
        ROAD_MARKERS.add("motorway_link");
        ROAD_MARKERS.add("trunk");
        ROAD_MARKERS.add("trunk_link");
        ROAD_MARKERS.add("primary");
        ROAD_MARKERS.add("primary_link");
        ROAD_MARKERS.add("secondary");
        ROAD_MARKERS.add("secondary_link");
        ROAD_MARKERS.add("tertiary");
        ROAD_MARKERS.add("unclassified");
        ROAD_MARKERS.add("road");
        ROAD_MARKERS.add("residential");
        ROAD_MARKERS.add("living_street");
        ROAD_MARKERS.add("service");
        ROAD_MARKERS.add("track");
        ROAD_MARKERS.add("services");
        ROAD_MARKERS.add("pedestrian");
    }
}

