﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		CSFileStream.cs
 * @brief		FileStream ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2013-2014 Takazumi Shirayanagi\n
 * This software is released under the MIT License,
 * see LICENSE
*/
//-----------------------------------------------------------------------
//======================================================================

//======================================================================
// using
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Windows.Forms;
using System.IO;
using Common;

namespace rcsharp
{
	public class Form2Res
	{
		#region フィールド
		private CSAssembly m_Assembly = null;
		private List<Form> m_Forms = null;
		private Dictionary<string, ToolStrip> m_Menus = null;
        private Dictionary<Keys, ToolStripItem> m_AccKeys = null;
		#endregion

		#region enum
		#region MACROS
		private enum MACORS
		{
			APSTUDIO_READONLY_SYMBOLS,
			APSTUDIO_HIDDEN_SYMBOLS,
			APSTUDIO_INVOKED,
		}
		#endregion
		#endregion

		#region Convert
		/// <summary>
		/// 変換関数
		/// </summary>
		/// <param name="asm_path"></param>
        public bool Convert(string asm_path, CSResID resIDs, string directory, string resheader)
		{
			// フォームを収集
			if (!CollectForm(asm_path)) return false;

			// 出力
			string name = Path.GetFileNameWithoutExtension(asm_path);
			if (directory == null)
			{
				directory = Path.GetDirectoryName(Path.GetFullPath(asm_path));
			}
			string headerfile = Path.Combine(directory, name + "_resource.h");
            if (resheader != null)
            {
                headerfile = Path.Combine(directory, resheader);
            }
			string rcfile = name + ".rc";
			rcfile = Path.Combine(directory, rcfile);
            bool ret = Output(resIDs, headerfile, rcfile, directory);
			return ret;
		}
		#endregion

		#region CollectForm
		/// <summary>
		/// フォーム収集関数
		/// </summary>
		/// <param name="asm_path"></param>
		public bool CollectForm(string asm_path)
		{
			m_Assembly = new CSAssembly();
			if (!m_Assembly.LoadFrom(asm_path)) return false;

			Type[] types = m_Assembly.GetTypes();
			m_Forms = new List<Form>();
			m_Menus = new Dictionary<string, ToolStrip>();
            m_AccKeys = new Dictionary<Keys, ToolStripItem>();

			try
			{
				foreach (Type t in types)
				{
					if (t.IsClass)
					{
						if (t == typeof(Form) || CSTypeHelper.IsBaseType(t, typeof(Form)))
						{
							CSConsole.WriteLine(t.FullName);
							Form form = Activator.CreateInstance(t, true) as Form;
							if (form != null)
							{
								m_Forms.Add(form);

								FieldInfo[] fis = form.GetType().GetFields(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance);
								foreach (FieldInfo fi in fis)
								{
									object obj = null;
									string name = fi.Name;
									obj = fi.GetValue(form);
									{
										MenuStrip ms = obj as MenuStrip;
										if (ms != null)
										{
											m_Menus.Add(name, ms);
										}
									}
									{
										ContextMenuStrip ms = obj as ContextMenuStrip;
										if (ms != null)
										{
											m_Menus.Add(name, ms);
										}
									}
								}
							}
						}
					}
				}

                // shortcut keys
                foreach (ToolStrip strip in m_Menus.Values)
                {
                    foreach( ToolStripItem item in strip.Items )
                    {
                        CollectShortcutKeys(item);
                    }
                }
			}
			catch (Exception exp)
			{
                Console.WriteLine(exp.Message);
			}
			return true;
		}
		#endregion

        #region CollectShortcutKeys
        /// <summary>
		/// フォーム収集関数
		/// </summary>
        /// <param name="strip"></param>
        public void CollectShortcutKeys(ToolStripItem strip)
        {
			ToolStripMenuItem menu = strip as ToolStripMenuItem;
			if (menu == null) return;

            Keys keys = menu.ShortcutKeys;
            if (keys != Keys.None)
            {
                m_AccKeys.Add(keys, strip);
            }
            foreach (ToolStripItem item in menu.DropDownItems)
            {
                CollectShortcutKeys(item);
            }
        }
        #endregion

		#region Output
		/// <summary>
		/// ファイル出力
		/// </summary>
		/// <param name="head">ヘッダファイル名</param>
		/// <param name="rc">出力パス</param>
		/// <returns></returns>
        private bool Output(CSResID resIDs, string head, string rc, string directory)
		{
            string header_name = Path.GetFileName(head);
            string name = Path.GetFileNameWithoutExtension(rc);
			CSRCFileStream rcfile = new CSRCFileStream();
			if (!rcfile.Open(rc, System.IO.FileMode.Create))
			{
				return false;
			}

			// ヘッダの書き込み
			rcfile.Write(HEADER.Replace("resource.h", header_name), 0);

			rcfile.Write(APSTUDIO_FIELD, 0);

			string buf = "";

			#region 日本語リソース
			// TODO : とりあえず日本語リソースのみ
			buf = SEPARATE + "// 日本語 resources" + CRLF + CRLF;
			rcfile.Write(buf, 0);

			buf = "#if !defined(AFX_RESOURCE_DLL) || defined(AFX_TARG_JPN)" + CRLF;
			buf += "#ifdef _WIN32" + CRLF;
			buf += "LANGUAGE LANG_JAPANESE, SUBLANG_DEFAULT" + CRLF;
			buf += "#pragma code_page(932)" + CRLF;
			buf += "#endif //_WIN32" + CRLF;
			buf += CRLF;
			rcfile.Write(buf, 0);
			#endregion

			#region TEXTINCLUDE
			rcfile.Write(BEGIN_TEXTINCLUDE, 0);
			rcfile.Write(TEXINCLUDE1.Replace("resource.h", header_name), 0);
			rcfile.Write(TEXINCLUDE2, 0);
			rcfile.Write(TEXINCLUDE3, 0);
			rcfile.Write(END_TEXTINCLUDE, 0);
			#endregion

			// Forms
			#region Forms
			if (m_Forms.Count > 0)
			{
				buf = CSRCFileStream.RESCOMMENT.Replace("NAME", "Dialog");
				rcfile.Write(buf, 0);

				foreach (Form form in m_Forms)
				{
					form.Show();
					buf = "";
					if (CSRCDialog.Serialize(form, resIDs, out buf))
					{
						rcfile.Write(buf, 0);
					}
					form.Hide();
				}
			}
			#endregion

			// menu
			#region Menu
			{
				buf = CSRCFileStream.RESCOMMENT.Replace("NAME", "Menu");
				rcfile.Write(buf, 0);

				foreach (KeyValuePair<string, ToolStrip> pair in m_Menus)
				{
					buf = "";
					if (CSRCToolStrip.SerializeMenuStrip(pair.Value, pair.Key, resIDs, out buf))
					{
						rcfile.Write(buf, 0);
					}
				}
			}
			#endregion

            // アクセラレータ
            #region ACCELERATORS
            {
                buf = CSRCFileStream.RESCOMMENT.Replace("NAME", "アクセラレータ");
                rcfile.Write(buf, 0);

                buf = "";
                if (CSRCAccelerators.Serialize(m_AccKeys, name, out buf))
                {
                    rcfile.Write(buf, 0);
                }
            }
            #endregion

            // resource
			#region Resource
			{
				buf = "";
				if (CSRCResource.Serialize(m_Assembly.Assembly, resIDs, directory, out buf))
				{
					rcfile.Write(buf, 0);
				}
			}
			#endregion

			// version
			#region Version
			{
				System.Diagnostics.FileVersionInfo vi = System.Diagnostics.FileVersionInfo.GetVersionInfo(m_Assembly.GetAssemblyPath());
				buf = "";
				if (CSRCVersionInfo.Serialize(vi, out buf))
				{
					rcfile.Write(buf, 0);
				}
			}
			#endregion

			#region 日本語リソース
			buf = "#endif    // 日本語 resources" + CRLF;
			buf += SEPARATE;
			rcfile.Write(buf, 0);
			#endregion

			// ヘッダ出力
			resIDs.DumpFile(head, rc);

			return true;
		}
		#endregion

		#region private プロパティ
		private static string APSTUDIO_FIELD
		{
			get
			{
				string ret = "#define ";
				ret += MACORS.APSTUDIO_READONLY_SYMBOLS.ToString();
				ret += CRLF;
				ret += SEPARATE;
				ret += "//" + CRLF + "// Generated from the TEXTINCLUDE 2 resource.";
				ret += CRLF + "//" + CRLF;
                //if( false )
                //{
                //    ret += "#include \"afxres.h\"";
                //}
                //else
				{
					ret += "#ifndef " + MACORS.APSTUDIO_INVOKED.ToString();
					ret += CRLF;
					ret += "#include \"targetver.h\"";
					ret += CRLF;
					ret += "#endif ";
					ret += CRLF;
					ret += "#define " + MACORS.APSTUDIO_HIDDEN_SYMBOLS.ToString();
					ret += CRLF;
					ret += "#include \"windows.h\"";
					ret += CRLF;
					ret += "#undef " + MACORS.APSTUDIO_HIDDEN_SYMBOLS.ToString();
					ret += CRLF;
				}
				ret += CRLF + CRLF;
				ret += SEPARATE;
				ret += "#undef ";
				ret += MACORS.APSTUDIO_READONLY_SYMBOLS.ToString();
				ret += CRLF + CRLF;
				return ret;
			}
		}
		#endregion

		#region private field

		private const string CRLF = CSRCFileStream.CRLF;
		private const string SEPARATE = CSRCFileStream.SEPARATE;
		private string HEADER = "// " + Application.ProductName + " generated resource script." + CRLF + "//" + CRLF + "#include \"resource.h\"" + CRLF + CRLF;
		private const string BEGIN_TEXTINCLUDE = "#ifdef APSTUDIO_INVOKED" + CRLF + SEPARATE + "//" + CRLF + "// TEXTINCLUDE" + "//" + CRLF;
		private const string END_TEXTINCLUDE = "#endif    // APSTUDIO_INVOKED" + CRLF + CRLF + CRLF;
		private const string TEXINCLUDE1 = "1 TEXTINCLUDE " + CRLF + "BEGIN" + CRLF + "    \"resource.h\\0\"" + CRLF + "END" + CRLF + CRLF;
        //private const string TEXINCLUDE2 = "2 TEXTINCLUDE " + CRLF + "BEGIN" + CRLF + "    \"#include \"\"afxres.h\"\"\\r\\n\"" + CRLF + "    \"\\0\"" + CRLF + "END" + CRLF;
        private const string TEXINCLUDE2 = "2 TEXTINCLUDE " + CRLF + "BEGIN" + CRLF 
            + "    \"#ifndef APSTUDIO_INVOKED\\r\\n\"" + CRLF 
            + "    \"#include \"\"targetver.h\"\"\\r\\n\"" + CRLF 
            + "    \"#endif \\r\\n\"" + CRLF 
            + "    \"#define APSTUDIO_HIDDEN_SYMBOLS\\r\\n\"" + CRLF
            + "    \"#include \"\"windows.h\"\"\\r\\n\"" + CRLF
            + "    \"#undef APSTUDIO_HIDDEN_SYMBOLS\\r\\n\"" + CRLF
            + "    \"\\0\"" + CRLF + "END" + CRLF;
        private const string TEXINCLUDE3 = "3 TEXTINCLUDE " + CRLF + "BEGIN" + CRLF + "    \"\\r\\n\"" + CRLF + "    \"\\0\"" + CRLF + "END" + CRLF + CRLF;
		#endregion
	}

}	// end of namespace rcsharp
