﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		CSAssembly.cs
 * @brief		Assembly ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2013-2014 Takazumi Shirayanagi\n
 * This software is released under the MIT License,
 * see LICENSE
*/
//-----------------------------------------------------------------------
//======================================================================

//======================================================================
// using
using System;
using System.Reflection;
using System.Runtime.InteropServices;

namespace Common
{
	public class CSAssembly
	{
		#region フィールド
		private Assembly m_Assembly = null;
		#endregion

		#region コンストラクタ
		public CSAssembly()
		{
		}
		
		public CSAssembly(Assembly asm)
		{
			m_Assembly = asm;
		}
		#endregion

		#region GetTypes
		public Type[] GetTypes()
		{
			return GetTypes(m_Assembly);
		}
		#endregion

		#region GetAssemblyPath
		public string GetAssemblyPath()
		{
			return GetAssemblyPath(m_Assembly);
		}
		#endregion

		#region GetDirectoryName
		public string GetDirectoryName()
		{
			return GetDirectoryName(m_Assembly);
		}
		#endregion

		#region VersionInfo

		#region GetFileVersion
		public string GetFileVersion()
		{
			return GetFileVersion(m_Assembly);
		}
		#endregion

		#region GetAssemblyVersion
		public string GetAssemblyVersion()
		{
			return GetAssemblyVersion(m_Assembly);
		}
		#endregion

		#region GetCompanyName
		public string GetCompanyName()
		{
			return GetCompanyName(m_Assembly);
		}
		#endregion

		#region GetProductName
		public string GetProductName()
		{
			return GetProductName(m_Assembly);
		}
		#endregion

		#region GetCopyright
		public string GetCopyright()
		{
			return GetCopyright(m_Assembly);
		}
		#endregion

		#region GetDescription
		public string GetDescription()
		{
			return GetDescription(m_Assembly);
		}
		#endregion

		#endregion

		#region GetInstance
		public IntPtr GetInstanceHandle()
		{
			return GetInstanceHandle(m_Assembly);
		}
		#endregion

		#region LoadFrom
		public bool LoadFrom(string path)
		{
			try
			{
				m_Assembly = Assembly.LoadFrom(path);
			}
			catch (Exception e)
			{
				Console.WriteLine(e.ToString());
				return false;
			}
			return true;
		}
		#endregion

		#region LoadExecuting
		public void LoadExecuting()
		{
			m_Assembly = Assembly.GetExecutingAssembly();
		}
		#endregion

		#region プロパティ

		#region Assembly
		public Assembly Assembly
		{
			get { return m_Assembly; }
			set { m_Assembly = value; }
		}
		#endregion

		#region RuntimeVersion
		public string RuntimeVersion
		{
			get { return m_Assembly.ImageRuntimeVersion; }
		}
		#endregion

		#region EntryAssembly
		public static CSAssembly EntryAssembly
		{
			get { return new CSAssembly(Assembly.GetEntryAssembly()); }
		}
		#endregion

		#region ExecutingAssembly
		public static CSAssembly ExecutingAssembly
		{
			get { return new CSAssembly(Assembly.GetExecutingAssembly()); }
		}
		#endregion

		#endregion

		#region static

		#region GetTypes
		public static Type[] GetTypes(Assembly asm)
		{
			try
			{
				return asm.GetTypes();
			}
			catch (ReflectionTypeLoadException e)
			{
				return e.Types;
			}
			catch
			{
				return null;
			}
		}
		#endregion

		#region GetAssemblyPath
		public static string GetAssemblyPath(Assembly asm)
		{
			string path = asm.CodeBase;
			Uri uri = new Uri(path);

			if (!uri.IsFile)
			{
				return asm.Location;
			}

			if (uri.IsUnc)
			{
				return path.Substring(Uri.UriSchemeFile.Length + 1);
			}

			int s = Uri.UriSchemeFile.Length + Uri.SchemeDelimiter.Length;
			if (path[s] == '/' && path[s + 2] == ':')
				++s;

			return path.Substring(s);
		}
		#endregion

		#region GetDirectoryName
		public static string GetDirectoryName(Assembly asm)
		{
			return System.IO.Path.GetDirectoryName(GetAssemblyPath(asm));
		}
		#endregion

		#region VersionInfo

		#region GetFileVersion
		public static string GetFileVersion(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyFileVersionAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyFileVersionAttribute)attr[0]).Version;
		}
		#endregion

		#region GetAssemblyVersion
		public static string GetAssemblyVersion(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyVersionAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyVersionAttribute)attr[0]).Version;
		}
		#endregion

		#region GetCompanyName
		public static string GetCompanyName(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyCompanyAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyCompanyAttribute)attr[0]).Company;
		}
		#endregion

		#region GetProductName
		public static string GetProductName(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyProductAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyProductAttribute)attr[0]).Product;
		}
		#endregion

		#region GetCopyright
		public static string GetCopyright(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyCopyrightAttribute)attr[0]).Copyright;
		}
		#endregion

		#region GetDescription
		public string GetDescription(Assembly asm)
		{
			object[] attr = asm.GetCustomAttributes(typeof(AssemblyDescriptionAttribute), false);
			if (attr == null || attr.Length == 0) return null;
			return ((AssemblyDescriptionAttribute)attr[0]).Description;
		}
		#endregion

		#endregion

		#region GetInstanceHandle
		public static IntPtr GetInstanceHandle(Assembly asm)
		{
			if (asm == null) return IntPtr.Zero;
			Module[] ms = asm.GetModules();
			return Marshal.GetHINSTANCE(ms[0]);
		}
		#endregion

		#endregion


	}

}	// end of namespace Common

#region UNITTEST

#endregion