/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.action {

	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	/**
	 * <p>[概 要] </p>
	 * アクションの処理フローを実装する、アクション基底クラスです.
	 * 
	 * <p>[詳 細] </p>
	 * 継承クラスはexecute(ParameterMapping)メソッドを実装して処理を実装します。<br />
	 * AbstractActionが提供する基本的な機能＋機能モデルをコントローラ経由で呼び出す為のメソッドを持ちます。
	 * 
	 * FlexibleActionにはBaseAction#nextModelメソッドやBaseAction#successForwardメソッドのように、
	 * 予め用意された処理フローが有りません。<br />
	 * このクラスを基底アクションクラスとして使用する場合は、executeメソッド内で任意に処理フローを記述します。<br />
	 * 
	 * <p>[備 考] </p>
	 * FlexibleActionではBaseActionのような非同期処理のシーケンシャルな実行がサポートされません。<br />
	 * 非同期処理Aの実行後に処理Bを行うような機能の場合、Aの終了イベントリスナ内でBを実行するように実装します。
	 * 
	 * @example
	 * 入力値バリデーション、サーバロジック実行等を任意のタイミングで実行する
	 * 
	 * <listing version="3.0">
        public class AddNumberByRPCAction extends FlexibleAction {

            override public function execute(parameterMapping:ParameterMapping):void {
                
                var tiNum1:TextInput = TextInput(getComponentByName("panel.tiNum1"));
                var tiNum2:TextInput = TextInput(getComponentByName("panel.tiNum2"));
    
                var tiNum1Validator:NumberValidator = new NumberValidator();
                tiNum1Validator.source = tiNum1;
                tiNum1Validator.property = "text";
                tiNum1Validator.domain = "int";
                tiNum1Validator.required = true;
    
                var tiNum2Validator:NumberValidator = new NumberValidator();
                tiNum2Validator.source = tiNum2;
                tiNum2Validator.property = "text";
                tiNum2Validator.domain = "int";
                tiNum2Validator.required = true;
    
                var validators:Array = new Array();            
                validators.push(tiNum1Validator);
                validators.push(tiNum2Validator);
    
                var errors:ValidateErrors = validate(validators);
                if(errors.hasError()) {
                    return;
                }
                
                var addRpc:RemoteObjectCore = new RemoteObjectCore();
                addRpc.remoteDestination = "orderDestination";
                addRpc.methodName = "add";
                addRpc.addRpcParameter(tiNum1.text);
                addRpc.addRpcParameter(tiNum2.text);
                
                var successFunction:Function = function(evt:ModelProcessEvent):void {
                    var sum:int = ResultEvent(evt.cause).result as int;
                    var tiSum:TextInput = TextInput(getComponentByName("panel.tiSum"));
                    tiSum.text = sum.toString();
                };
    
                var failureFunction:Function = function(evt:ModelProcessEvent):void {
                    Alert.show("RPC呼び出し中にエラーが発生しました。\n " + evt.cause.toString());
                };
    
                runModel(addRpc, parameterMapping, successFunction, failureFunction);
            }        
         }
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class FlexibleAction extends AbstractAction {

		/**
		 * <p>[概 要]</p>
		 * コントローラにコールされるアクションの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * executeメソッドをテンプレートコールします。<br />
		 *  
		 * <p>[備 考]</p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @return null
		 */
        override public function run(parameterMapping:ParameterMapping):ParameterMapping{
        	super.run(parameterMapping);
        	
        	execute(parameterMapping);
        	
			return null;
        }
        
		/**
		 * <p>[概 要] </p>
		 * アクション処理を実装するメソッドです.
		 *  
		 * <p>[詳 細] </p>
		 * FlexibleAction継承アクションは、このメソッドをオーバーライドして、
		 * バリデーション、モデル実行、コンポーネント編集等の処理を任意に実装します。<br />
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 */
        protected function execute(parameterMapping:ParameterMapping):void {
        }
        
		/**
		 * <p>[概 要] </p>
		 * 機能モデルをコントローラ経由で呼び出します.
		 * 
		 * <p>[詳 細] </p>
		 * コントローラのモデル実行機能に引数modelの実行を委譲します。<br />
		 *  
		 * <p>[備 考]</p>
		 * 
		 * @param model コントローラに実行委譲するBaseModel継承オブジェクト
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @param successFunction 第一引数modelのModelProcessEvent.SUCCESSイベントハンドラ
		 * @param failureFunction 第一引数modelのModelProcessEvent.FAILUREイベントハンドラ
		 */
        protected function runModel(model:BaseModel, 
        						   parameterMapping:ParameterMapping, 
        						   successFunction:Function=null,
        						   failureFunction:Function=null):void {
        	if(successFunction != null) {
	        	model.addEventListener(ModelProcessEvent.SUCCESS, successFunction);
	        }
	        if(failureFunction != null) {
	        	model.addEventListener(ModelProcessEvent.FAILURE, failureFunction);
	        }
        	
        	controller.runModel(model, parameterMapping);
        }
	}
}
