/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.timer {
	import flash.events.TimerEvent;
	
	import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.manager.TimerProcessCoreManager;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;

	/**
	 * <p>[概 要]</p>
	 * 一定周期で任意の処理を行う為の機能モデルクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * 以下のプロパティを設定することで動作します。
	 * <ul>
	 *   <li>
	 *     timerId<br />
	 *     使用するTimerオブジェクトを識別する為のIDです。重複することは出来ません。
	 * 
	 *   </li>
	 *   <li>
	 *     interval<br />
	 *     Timer実行のインターバル（ミリ秒）です。3000を指定すると3秒間隔でタイマーが動作します。
	 * 
	 *   </li>
	 *   <li>
	 *     repeat<br />
	 *     インターバル実行の繰り返し回数です。3を指定すると、intervalミリ秒を3回繰り返してタイマーが終了します。
	 *   </li>
	 *   <li>
	 *     active<br />
	 *     タイマー開始、停止を指示します。デフォルトはtrue（開始）です。
	 *   </li>
	 * </ul>
	 * TimerProcessCoreオブジェクトは、上記の設定によって指定周期毎にModelProcessEvent.SUCCESSイベントを発行し、<br />
	 * コントローラによってTimerProcessCoreをリザーブしたアクションのsuccessForwardメソッドがコールバックされます。<br />
	 * 
	 * ・<b>5秒毎に3回successForwardを呼び出すアクション</b>
	 * <br />
	 * <listing version="3.0">
        public class SimpleTimerAction extends BaseAction {
            
            override protected function reserveModels(models:Array):void {
                models.push(Class(TimerProcessCore));
            }
            
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                if(index == 0) {
                    TimerProcessCore(next).timerId = "simpleTimer";
                    TimerProcessCore(next).interval = 5000;
                    TimerProcessCore(next).repeat = 3;
                }
                
                return true;
            }
            
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                trace("5秒毎に計3回コールバックされます");
            }
        }
	 * </listing>
	 * 
	 * 又、intervalActionプロパティが設定されていると、指定周期毎にコントローラに
	 * intervalAction（BaseAction継承クラス）の実行を委譲します。<br />
	 * この場合、TimerProcessCoreをリザーブしたアクションのsuccessForwardが呼ばれる契機は、intervalActionの終了、<br />
	 * つまりintervalAction#complete()が呼ばれるタイミングになります。<br />
	 * 以下のコードはサーバ時間を取得するアクションを定期的に実行します。
	 * 
	 * <ul>
	 * 	<li><b>サーバ時間を取得するアクション</b><br />
	 * 
	 * java.util.Dateを返却するリモートRPCメソッドを実行します。
	 * 
	 * <listing version="3.0">
        public class ServerDateFetchAction extends BaseAction {
            
            override protected function reserveModels(models:Array):void {
                models.push(Class(RemoteObjectCore));
            }
            
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                if(index == 0) {
                    RemoteObjectCore(next).remoteDestination = "serverInformation";
                    RemoteObjectCore(next).methodName = "getServerTime";
                }
                
                return true;
            }
            
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                trace("ServerDateFetchAction#successForward : サーバ時間 = " + ResultEvent(resultEvent).result);
            }
            
            override public function complete():void {
                trace("ServerDateFetchAction#complete : ServerDateFetchAction完了");
            }
        } 
	 * </listing>
	 * 	</li>
	 * 	<li><b>ServerDateFetchActionを定期的に実行するアクション</b><br />
	 * 
	 * regularityRemoteAccessというTimerIDで3秒間隔でServerDateFetchActionを実行します。<br />
	 * 3回実行した後、TimerProcessCoreManagerを介して明示的にタイマーを止めています。
	 * <listing version="3.0">
        public class ExecuteActionByTimerAction extends BaseAction {
            
            override protected function reserveModels(models:Array):void {
                models.push(Class(TimerProcessCore));
            }
            
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                if(index == 0) {
                    TimerProcessCore(next).timerId = "regularityRemoteAccess";
                    TimerProcessCore(next).interval = 3000;
                    TimerProcessCore(next).intervalAction = Class(ServerDateFetchAction);
                }
                
                return true;
            }
            
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                trace("ExecuteActionByTimerAction#successForward : ServerDateFetchAction実行成功");
                if(model.successCount == 3) {
                    TimerProcessCoreManager.getInstance().stop("regularityRemoteAccess");
                }
            }
            
            override public function complete():void {
                trace("ExecuteActionByTimerAction#complete :タイマー終了");
            }
        }
	 * </listing>
	 * 	</li>
	 * 	<li>実行結果<br />
	 * <pre>
	 *	ServerDateFetchAction#successForward : サーバ時間 = Fri Sep 25 21:47:24 GMT+0900 2009
	 *	ServerDateFetchAction#complete : ServerDateFetchAction完了
	 *	ExecuteActionByTimerAction#successForward : ServerDateFetchAction実行成功
	 *	ServerDateFetchAction#successForward : サーバ時間 = Fri Sep 25 21:47:27 GMT+0900 2009
	 *	ServerDateFetchAction#complete : ServerDateFetchAction完了
	 *	ExecuteActionByTimerAction#successForward : ServerDateFetchAction実行成功
	 *	ServerDateFetchAction#successForward : サーバ時間 = Fri Sep 25 21:47:30 GMT+0900 2009
	 *	ServerDateFetchAction#complete : ServerDateFetchAction完了
	 *	ExecuteActionByTimerAction#successForward : ServerDateFetchAction実行成功
	 *	ExecuteActionByTimerAction#complete :タイマー終了
	 * </pre>
	 * 	</li>
	 * </ul>
	 * 
	 * <p>[備 考]</p>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */ 
	public class TimerProcessCore extends BaseModel {
		
		// 識別子
		private var _timerId:String;
		
		// 実行間隔(ミリ秒)
		private var _interval:Number;
		
		// 繰り返す回数 デフォルトは無制限
		private var _repeat:int;
		
		// 開始フラグ
		private var _active:Boolean = true;
		
		// インターバル実行するアクションクラス型
		private var _intervalAction:Class;
		
		/**
		 * <p>[概 要]</p>
		 * 使用するタイマーの識別子です.
		 * 
		 * <p>[詳 細]</p>
		 * タイマーをストップさせる為に使用します。<br />
		 * 同時に同じtimerIdを持つTimerを実行しようとするとCoreLogicErrorがスローされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get timerId():String {
			return this._timerId;
		}
		public function set timerId(timerId:String):void {
			this._timerId = timerId;
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマー実行間隔です.
		 * 
		 * <p>[詳 細]</p>
		 * 実行間隔をミリ秒で指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get interval():Number {
			return this._interval;
		}
		public function set interval(interval:Number):void {
			this._interval = interval;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 繰り返す回数を指定します.
		 * 
		 * <p>[詳 細]</p>
		 * intervalを何回繰り返すかを指定します。<br />
		 * 指定が無い場合、無制限に実行されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get repeat():int {
			return this._repeat;
		}
		public function set repeat(repeat:int):void {
			this._repeat = repeat;
		}

		/**
		 * <p>[概 要]</p>
		 * インターバル実行するアクションクラスです.
		 * 
		 * <p>[詳 細]</p>
		 * このプロパティが設定されている場合は、タイマーイベントでこのアクションが
		 * コントローラに実行委譲されます。<br />
		 * 
		 * <p>[備 考]</p>
		 * 実行するアクションの処理が、intervalプロパティで指定した時間内に終わらない
		 * ものであった場合、負荷が累積します。<br />
		 * その場合、アクションの処理を軽減するか、intervalプロパティを長く設定するなどの対処が必要です。
		 * 
		 */ 
		public function get intervalAction():Class {
			return this._intervalAction
		}
		public function set intervalAction(intervalAction:Class):void {
			this._intervalAction = intervalAction;
		}

		/**
		 * <p>[概 要]</p>
		 * 開始状態を指定します.
		 * 
		 * <p>[詳 細]</p>
		 * true：tick開始<br />
		 * false : tick停止<br />
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default true
		 */ 
		public function get active():Boolean {
			return this._active;
		}
		public function set active(active:Boolean):void {
			this._active = active;
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマーの開始、終了を行います.
		 * 
		 * <p>[詳 細]</p>
		 * activeプロパティがtrueの場合はタイマー開始、falseの場合はタイマー停止します。<br />
		 * 具体的なタイマー制御処理はTimerProcessCoreManagerに委譲されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		override protected function mainProc():void {
			super.mainProc();
			
			if (this._active) {
				TimerProcessCoreManager.getInstance().start(this._timerId, this);
			} else {
				TimerProcessCoreManager.getInstance().stop(this._timerId);

				var modelSuccessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
				dispatchModelSuccess(modelSuccessEvent);

				var modelFinishedEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
				dispatchModelFinished(modelFinishedEvent);
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * タイマーイベントのハンドラです.
		 * 
		 * <p>[詳 細]</p>
		 * intervalAction指定が無い場合にTimerによってコールバックされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public final function resultHandler(event:TimerEvent):void {
			onSuccess(event);
			
			var modelProcessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
			modelProcessEvent.cause = event;
			dispatchModelSuccess(modelProcessEvent);
		}

		/**
		 * <p>[概 要]</p>
		 * オーバーライド用タイマーイベントハンドラです.
		 * 
		 * <p>[詳 細]</p>
		 * 継承クラスでタイマーイベントをハンドリングする為に使用します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		protected function onSuccess(event:TimerEvent):void{
		}
	}
}
