/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.listener;

import java.awt.Component;
import java.awt.Container;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;

import jp.co.fujitsu.reffi.client.swing.controller.AbstractController;

/**
 * <p>[概 要] </p>
 * コンテナコンポーネントの、内容コンポーネントの追加削除を監視するクラスです。
 * 
 * <p>[詳 細] </p>
 * コンテナに対するコンポーネント追加を監視して、EventBinderに紐付け予約されている
 * イベントリスナの付与をコントローラに委譲します。
 * <p>
 * 
 * AbstractControllerがウィンドウのオープンを検知すると、オープンされたウィンドウ
 * の内容を走査してコンテナコンポーネントに対してこのリスナを追加します。<br>
 * componentAddedイベントハンドラ実装では、追加されたコンポーネント子孫を再帰的に
 * 走査し、登録するべきイベントが予約されていないか、EventBinderにイベントリスナ付与
 * を依頼します。
 * 
 * 
 * <p>[備 考] </p>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ContainerListenerImpl implements ContainerListener {

	/** コントローラです。 */
	private AbstractController controller;
	
	
	/**
	 * <p>[概 要] </p>
	 * コントローラを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * controlerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コントローラ
	 */
	public AbstractController getController() {
		return controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * controlerフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラ
	 */
	public void setController(AbstractController controller) {
		this.controller = controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数controllerをフィールドに設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラ
	 */
	public ContainerListenerImpl(AbstractController controller) {
		this.controller = controller;
	}
	
	/**
	 * <p>[概 要] </p>
	 * コンポーネント追加イベントハンドラ実装です。
	 * 
	 * <p>[詳 細] </p>
	 * {@link ContainerListener#componentAdded(ContainerEvent)}を実装します。<br>
	 * コンポーネント追加イベントから追加されたコンポーネントを取り出し、<br>
	 * {@link #searchComponent(Component)}を呼び出して子孫コンポーネントを走査します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e コンポーネント追加イベント
	 */
	@Override
	public void componentAdded(ContainerEvent e) {
		// 追加されたコンポーネント
		Component addedComponent = e.getChild();
		// 追加されたコンポーネント内に含まれるコンポーネントを全検出
		searchComponent(addedComponent);
	}

	/**
	 * <p>[概 要] </p>
	 * 追加されたコンポーネントの子孫を走査します。
	 * 
	 * <p>[詳 細] </p>
	 * 追加されたコンポーネントの中にコンテナコンポーネントが有った場合、
	 * 再帰的にContainerListenerImplを追加します。
	 * <p>
	 * 
	 * 走査されたコンポーネント一つ一つに対して、
	 * {@link AbstractController#bindEvents(Component)}メソッドを呼び出し、
	 * イベントリスナ付与を委譲します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param addedComponent コンテナに追加されたコンポーネント
	 */
	private void searchComponent(Component addedComponent) {

		if(addedComponent instanceof Container) {
			Container container = ((Container)addedComponent);
			ContainerListener[] ls = container.getContainerListeners();

			// 既にContainerListenerImpleがリスナ登録されている場合は処理しない
			for(ContainerListener l : ls){
				if(l instanceof ContainerListenerImpl) {
					return;
				}
			}
			
			Component[] cs = container.getComponents();
			for(Component c : cs) {
				searchComponent(c);
			}
			
			container.addContainerListener(new ContainerListenerImpl(getController()));
		}

		getController().bindEvents(addedComponent);
	}
	
	/**
	 * <p>[概 要] </p>
	 * コンポーネント削除イベントハンドラ実装です。
	 * 
	 * <p>[詳 細] </p>
	 * 処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e コンポーネント削除イベント
	 */
	@Override
	public void componentRemoved(ContainerEvent e) {
	}
}
