/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.model;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Service;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.ws.handler.PortInfo;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import jp.co.fujitsu.reffi.client.swing.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.swing.util.TypeMaintainList;

/**
 * <p>[概 要] </p>
 * サーバとSOAP通信を行う為の機能モデルクラスです。
 * 
 * <p>[詳 細] </p>
 * wsimportで生成されたJAX-WS通信スタブを操作してSOAP通信を行います。<p/>
 * 
 * 
 * 必須設定メソッド
 * 
 * <ul>
 *   <li>{@link #setServiceClass(Class)} : JAX-WS Serviceクラスの設定</li>
 *   <li>{@link #setPortClass(Class)} : JAX-WS Portクラスの設定</li>
 *   <li>{@link #setMethodName(String)} : 呼び出すサービスメソッド名</li>
 *   <li>{@link #setParameters(TypeMaintainList)} : サービスメソッド引数</li>
 * </ul>
 * 
 * 通信スタブ内にハードコーディングされているWSDL以外のWSDLを取得する場合、
 * 別途以下のパラメータが必要です。
 * 
 * <ul>
 *   <li>{@link #setRequestTarget(String, QName)} : 接続先WSDLの設定</li>
 * </ul>
 * <pre class="samplecode">
 * 	// 送信先サービスを動的決定
 *	String wsdlLocation = "http://foo.bar/DummyReceptor/filerecieve?wsdl":
 *	QName serviceName = new QName("http://recept.ws.dummy/", "FileRecieveService");
 *	((WebServiceCore)next).setRequestTarget(wsdlLocation, serviceName);
 * </pre>
 *
 * 上記セッターを実行することで、Serviceクラスのインスタンス化時、実行するコンストラクタ
 * を切り替えます。<br>
 * QNameに設定するパラメータは生成されたServiceクラスのソースコードを参照してください。<p/>
 * 
 * 
 * 
 * <p>[備 考] </p>
 * BASIC認証、DIGEST認証、Cookieに関する設定は{@link AbstractHTTPCore}を参照して下さい。
 * 
 * 
 * <b>使用例）</b><br>
 * ローカルファイルをMTOM通信でサーバにアップロードする
 * <pre class="samplecode">
 * package demo.webservice.action;
 *
 *	import java.util.List;
 *	
 *	import javax.activation.DataHandler;
 *	import javax.activation.FileDataSource;
 *	
 *	import jp.co.fujitsu.reffi.client.swing.action.BaseAction;
 *	import jp.co.fujitsu.reffi.client.swing.event.ModelProcessEvent;
 *	import jp.co.fujitsu.reffi.client.swing.model.Model;
 *	import jp.co.fujitsu.reffi.client.swing.model.WebServiceCore;
 *	import jp.co.fujitsu.reffi.client.swing.util.MessageDialogUtil;
 *	import demo.server.ws.mtom.FileRecieve;
 *	import demo.server.ws.mtom.FileRecieveService;
 *	
 *	public class MtomSendAction extends BaseAction {
 *	
 *		&#064;Override
 *		protected void reserveModels(List<Class<? extends Model>> models) {
 *			models.add(WebServiceCore.class);
 *		}
 *	
 *		&#064;Override
 *		public boolean nextModel(int index, ModelProcessEvent prev, Model next) throws Exception {
 *			if(index == 0) {
 *				((WebServiceCore)next).setAsync(true);
 *				((WebServiceCore)next).setServiceClass(FileRecieveService.class);
 *				((WebServiceCore)next).setProxyClass(FileRecieve.class);
 *				((WebServiceCore)next).setMethodName("sendBinary");
 *	
 *				DataHandler uploadData = new DataHandler(new FileDataSource("D:\\bigSizeFile.zip"));
 *				((WebServiceCore)next).addParameter(uploadData);
 *			}
 *			
 *			return true;
 *		}
 *	
 *		&#064;Override
 *		public void successForward(int index, Model model, Object result)	throws Exception {
 *			showMessageDialog("ファイルアップロードが完了しました。");
 *		}
 *	
 *		&#064;Override
 *		public Exception failureForward(int index, Model model, Exception e) {
 *			showMessageDialog("ファイルアップロードが失敗しました", "", MessageDialogUtil.ERROR_MESSAGE);
 *			return super.failureForward(index, model, e);
 *		}
 *	}
 * </pre>
 * 
 * <b>使用例）</b><br>
 * SOAPヘッダにUsernameToken情報を入れる<p/>
 * 
 * SOAPリクエストハンドラを作成
 * <pre class="samplecode">
 * 	package demo.communication.action;
 *	
 *	import java.text.SimpleDateFormat;
 *	import java.util.Calendar;
 *	import java.util.Date;
 *	import java.util.Locale;
 *	import java.util.Set;
 *	
 *	import javax.xml.namespace.QName;
 *	import javax.xml.soap.SOAPElement;
 *	import javax.xml.soap.SOAPEnvelope;
 *	import javax.xml.soap.SOAPHeader;
 *	import javax.xml.ws.handler.MessageContext;
 *	import javax.xml.ws.handler.soap.SOAPHandler;
 *	import javax.xml.ws.handler.soap.SOAPMessageContext;
 *	
 *	public class UsernameTokenHandler implements SOAPHandler<SOAPMessageContext> {
 *	
 *		&#064;Override
 *		public boolean handleMessage(SOAPMessageContext context) {
 *			Boolean outboundProperty = (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);
 *	
 *	        if (outboundProperty.booleanValue()) {
 *	        	try {
 *	        		// SOAP情報
 *	                SOAPEnvelope envelope = context.getMessage().getSOAPPart().getEnvelope();
 *	                SOAPHeader header = envelope.getHeader();
 *	                
 *	                if (header == null){
 *	                	header = envelope.addHeader();
 *	                }
 *	
 *	                // セキュリティ
 *	                SOAPElement security =
 *	                    header.addChildElement("Security", "wsse", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");
 *	                
 *	                // セキュリティ（タイムスタンプ）
 *	                Calendar cal = Calendar.getInstance(Locale.ENGLISH);
 *	                cal.add(Calendar.HOUR, -9);
 *	                Date createdDate = cal.getTime();
 *	                cal.add(Calendar.MINUTE, 5);
 *	                Date expiresDate = cal.getTime();
 *	                
 *	                SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'.'S'Z'", Locale.ENGLISH);
 *	                
 *	                SOAPElement timestamp =
 *	                	security.addChildElement("Timestamp", "wsu", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
 *	                
 *	                SOAPElement created =
 *	                	timestamp.addChildElement("Created", "wsu");
 *	
 *	                created.addTextNode(sdf1.format(createdDate));
 *	                SOAPElement expires =
 *	                	timestamp.addChildElement("Expires", "wsu");
 *	                expires.addTextNode(sdf1.format(expiresDate));
 *	                
 *	                // セキュリティ（UsernameToken）
 *	                SOAPElement usernameToken =
 *	                        security.addChildElement("UsernameToken", "wsse");
 *	                
 *	                usernameToken.addAttribute(new QName("xmlns:wsu"), "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
 *	
 *	                SOAPElement username =
 *	                        usernameToken.addChildElement("Username", "wsse");
 *	                username.addTextNode("alice");
 *	
 *	                SOAPElement password =
 *	                        usernameToken.addChildElement("Password", "wsse");
 *	                password.setAttribute("Type", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
 *	                password.addTextNode("bobPW");
 *	            } catch (Exception e) {
 *	                e.printStackTrace();
 *	            }
 *	        }
 *			return true;
 *		}
 *	
 *		&#064;Override
 *		public Set<QName> getHeaders() {
 *			return null;
 *		}
 *	
 *		&#064;Override
 *		public boolean handleFault(SOAPMessageContext context) {
 *			return true;
 *		}
 *	
 *		&#064;Override
 *		public void close(MessageContext context) {
 *		}
 *	} 
 * </pre>
 * 
 * アクション
 * <pre class="samplecode">
 * 	&#064;Override
 *	public boolean nextModel(int index, ModelProcessEvent prev, Model next) throws Exception {
 *		if(index == 0) {
 *			boolean soapAsync = (Boolean)getComponentValue("communicationFrame.jrbSoapAsync");
 *			if(soapAsync) {
 *				((WebServiceCore)next).setAsync(true);
 *			}
 *			
 *			((WebServiceCore)next).setServiceClass(HeavyProcessService.class);
 *			((WebServiceCore)next).setProxyClass(HeavyProcess.class);
 *			((WebServiceCore)next).setMethodName("heavyProcess");
 *			((WebServiceCore)next).addParameter(5000L);
 *			((WebServiceCore)next).setAuthentication("user", "pass".toCharArray());
 *
 *			// UsernameToken情報をヘッダに挿入するハンドラを追加
 *			((WebServiceCore)next).addHandlerChain(new UsernameTokenHandler());
 *		}
 *		
 *		return true;
 *	}
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class WebServiceCore extends AbstractHTTPCore {
	
	/** JAX-WS 通信スタブのServiceクラスです。 */
	private Class<? extends Service> serviceClass;
	
	/** JAX-WS 通信スタブのProxyクラスです。 */
	private Class portClass;
	
	/** 呼び出すWebサービスメソッド名です。 */
	private String methodName;

	/** Webサービスメソッドの引数です。 */
	private TypeMaintainList parameters;
	
	/** 取得するWSDLのURL */
	private URL wsdlDocumentLocationUrl;
	
	/** サービス名QName */
	private QName serviceName;
	
	/** SOAPハンドラのリストです。 */
	private List<Handler> handlerChain; 
	
	
	/**
	 * <p>[概 要] </p>
	 * JAX-WS 通信スタブのServiceクラスを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * serviceClassフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return JAX-WS 通信スタブのServiceクラス
	 */
	public Class<? extends Service> getServiceClass() {
		return serviceClass;
	}

	/**
	 * <p>[概 要] </p>
	 * JAX-WS 通信スタブのServiceクラスを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * serviceClassフィールドを引数serviceClassで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param serviceClass JAX-WS 通信スタブのServiceクラス
	 */
	public void setServiceClass(Class<? extends Service> serviceClass) {
		this.serviceClass = serviceClass;
	}

	/**
	 * <p>[概 要] </p>
	 * JAX-WS 通信スタブのPortクラスを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * proxyClassフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return JAX-WS 通信スタブのProxyクラス
	 */
	public Class getPortClass() {
		return portClass;
	}

	/**
	 * <p>[概 要] </p>
	 * JAX-WS 通信スタブのPortクラスを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * proxyClassフィールドを引数proxyClassで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param portClass JAX-WS 通信スタブのProxyクラス
	 */
	public void setPortClass(Class portClass) {
		this.portClass = portClass;
	}

	/**
	 * <p>[概 要] </p>
	 * 呼び出すWebサービスメソッド名を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * methodNameフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 呼び出すWebサービスメソッド名
	 */
	public String getMethodName() {
		return methodName;
	}

	/**
	 * <p>[概 要] </p>
	 * 呼び出すWebサービスメソッド名を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * methodNameフィールドを引数methodNameで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param methodName 呼び出すWebサービスメソッド名
	 */
	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return Webサービスメソッドの引数
	 */
	public TypeMaintainList getParameters() {
		return parameters;
	}
	
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameters Webサービスメソッドの引数
	 */
	public void setParameters(TypeMaintainList parameters) {
		this.parameters = parameters;
	}

	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(Object parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(char parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(short parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(int parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(long parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(float parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(double parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * Webサービスメソッドの引数指定を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter Webサービスメソッドパラメータ
	 */
	public void addParameter(boolean parameter) {
		this.parameters.add(parameter);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 取得するWSDLを動的に変更します。
	 * 
	 * <p>[詳 細] </p>
	 * 通信スタブに記述されているWSDL以外を使用する場合はこのメソッドで
	 * ターゲットWSDLを変更します。<br>
	 * 
	 * <pre class="samplecode">
	 * 	String wsdlLocation = "http://foo.bar.com/Acceptance/filerecieve?wsdl"
	 *	QName serviceName = new QName("http://com.bar.foo/", "FileRecieveService");
	 *	((WebServiceCore)next).setRequestTarget(wsdlLocation, serviceName);
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param wsdlDocumentLocation WSDLロケーション
	 * @param serviceName Service QName
	 * @throws MalformedURLException 
	 */
	public void setRequestTarget(String wsdlDocumentLocation, QName serviceName) throws MalformedURLException {
		this.wsdlDocumentLocationUrl = new URL(wsdlDocumentLocation);
		this.serviceName = serviceName;
	}
	
	/**
	 * <p>[概 要] </p>
	 * SOAPハンドラのリストを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * handlerChainフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return SOAPハンドラのリスト
	 */
	public List<Handler> getHandlerChain() {
		return handlerChain;
	}

	/**
	 * <p>[概 要] </p>
	 * SOAPハンドラのリストを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * handlerChainフィールドに引数handlerChainを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param handlerChain SOAPハンドラのリスト
	 */
	public void setHandlerChain(List<Handler> handlerChain) {
		this.handlerChain = handlerChain;
	}

	/**
	 * <p>[概 要] </p>
	 * SOAPハンドラのリストにSOAPハンドラを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * handlerChainフィールドに引数handlerを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param handler SOAPハンドラ
	 */
	public void addHandlerChain(Handler<SOAPMessageContext> handler) {
		this.handlerChain.add(handler);
	}
	
	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * Webサービスメソッドパラメータリストインスタンスの初期化、
	 * Handlerチェインリストの初期化を行います。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @throws Exception
	 */
	public WebServiceCore() throws Exception {
		this.parameters = new TypeMaintainList();
		this.handlerChain = new ArrayList<Handler>();
	}
	
	/**
	 * <p>[概 要] </p>
	 * 設定された通信スタブを使用してサーバとSOAP通信を行います。
	 * 
	 * <p>[詳 細] </p>
	 * 以下の処理を行います。
	 * 
	 * <ol>
	 *   <li>
	 *   	Serviceクラスのインスタンス化。この際、setRequestTargetが実行されている
	 *   	場合は使用するコンストラクタを変更します。
	 *   </li>
	 *   <li>
	 *   	生成されたServiceオブジェクトにHandlerチェインを追加。
	 *   </li>
	 *   <li>
	 *   	ServiceオブジェクトからPortオブジェクトを生成。
	 *   </li>
	 *   <li>
	 *   	呼び出すWebサービスメソッドをリフレクション、パラメータ設定。
	 *   </li>
	 *   <li>
	 *   	Webサービスメソッド実行
	 *   </li>
	 * </ol>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @throws ClassNotFoundException 
	 * @throws IllegalAccessException 
	 * @throws InstantiationException 
	 * @throws NoSuchMethodException 
	 * @throws SecurityException 
	 * @throws InvocationTargetException 
	 * @throws IllegalArgumentException 
	 */
	@Override
	protected void mainproc() 	throws InstantiationException, 
										IllegalAccessException, 
										ClassNotFoundException, 
										SecurityException, 
										NoSuchMethodException, 
										IllegalArgumentException, 
										InvocationTargetException 
	{
		// wsdlロケーションが設定されている場合はオーバーロードコンストラクタを使用
		Object service = null;
		if(this.wsdlDocumentLocationUrl == null) {
			service = getServiceClass().newInstance();
		}else{
			Class[] constructerParametersType = {URL.class, QName.class};
			Object[] constructerParameters = {this.wsdlDocumentLocationUrl, this.serviceName};
			service = getServiceClass().
				getConstructor(constructerParametersType).newInstance(constructerParameters);
		}
		
		// Handlerチェイン追加
		if(getHandlerChain().size() > 0) {
			((Service)service).setHandlerResolver(new HandlerResolver(){
				@Override
				public List<Handler> getHandlerChain(PortInfo portInfo) {
					return WebServiceCore.this.getHandlerChain();
				}
			});
		}
		
		Method getPort = service.getClass().getMethod("get" + portClass.getSimpleName() + "Port", null);
		Object port = getPort.invoke(service, null);

		if(getAuthUser() != null) {
			BindingProvider bp = (BindingProvider)port;

			Map<String, Object> reqContext = bp.getRequestContext();
			reqContext.put(BindingProvider.USERNAME_PROPERTY, getAuthUser());
			reqContext.put(BindingProvider.PASSWORD_PROPERTY, String.valueOf(getAuthPassword()));
		}
		
		Class[] parameterTypes = getParameters().toTypeArray();
		Method method = port.getClass().getMethod(getMethodName(), parameterTypes);
		Object result = method.invoke(port, getParameters().toValueArray());
		
		setResult(result);
	}
	
	/**
	 * モデル処理成功イベント、モデル処理終了イベントを発行します。
	 * 
	 * <p>[詳 細] </p>
	 * fireModelSuccess、fireModelFinishedメソッドをコールします。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	@Override
	protected void postproc() throws Exception {
		ModelProcessEvent successEvent = new ModelProcessEvent(this);
		successEvent.setResult(getResult());
		fireModelSuccess(successEvent);
		
		fireModelFinished(new ModelProcessEvent(this));

	}
}