/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.model;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Hashtable;

import javax.naming.InitialContext;
import javax.naming.NamingException;

import jp.co.fujitsu.reffi.client.swing.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.swing.util.TypeMaintainList;

/**
 * <p>[概 要] </p>
 * サーバとEJB通信を行う機能モデルクラスです。
 * 
 * <p>[詳 細] </p>
 * EJBリモートオブジェクトを取得してRPCを行います。
 * <p>
 * 
 * 必須設定メソッド
 * <ul>
 *   <li>{@link #setLookupName(String)} : リモートEJBオブジェクトのJNDI名設定</li>
 *   <li>{@link #setMethodName(String)} : 呼び出すメソッド名の設定</li>
 * </ul>
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b><br>
 * EJBを使用してログイン認証を行う。
 * <pre class="samplecode">
 * 	package demo.login.action;
 *
 *	import java.util.EventListener;
 *	import java.util.List;
 *	
 *	import javax.swing.JPasswordField;
 *	import javax.swing.JTextField;
 *	
 *	import jp.co.fujitsu.reffi.client.swing.action.BaseAction;
 *	import jp.co.fujitsu.reffi.client.swing.event.ModelProcessEvent;
 *	import jp.co.fujitsu.reffi.client.swing.model.EJBProxyCore;
 *	import jp.co.fujitsu.reffi.client.swing.model.Model;
 *	
 *	public class LoginAction extends BaseAction {
 *	
 *		&#064;Override
 *		protected void reserveModels(List<Class<? extends Model>> models) {
 *			models.add(EJBProxyCore.class);
 *		}
 *	
 *		&#064;Override
 *		public boolean nextModel(int index, ModelProcessEvent prev, Model next) throws Exception {
 *			if(index == 0) {
 *				// 使用するEJBオブジェクトのJNDI名を設定。
 *				((EJBProxyCore)next).setLookupName("ReffiFunctionalDemoForSwingServer/LoginBean/remote");
 *				// 呼び出すメソッド名を設定。
 *				((EJBProxyCore)next).setMethodName("login");
 *				// パラメータ設定
 *				((EJBProxyCore)next).addParameter(getComponentValueAsString("loginFrame.jtfUserId"));
 *				((EJBProxyCore)next).addParameter(String.valueOf((char[])getComponentValue("loginFrame.jpwPassword")));
 *			}
 *			
 *			return true;
 *		}
 *	
 *		&#064;Override
 *		public void successForward(int index, Model model, Object result)	throws Exception {
 *			showMessageDialog("ログインしました", "成功", MessageDialogUtil.INFORMATION_MESSAGE, MessageDialogUtil.DEFAULT_OPTION);			}
 *		
 *		&#064;Override
 *		public Exception failureForward(int index, Model model, Exception e) {
 *			showMessageDialog("ログインに失敗しました", "エラー", MessageDialogUtil.ERROR_MESSAGE, MessageDialogUtil.DEFAULT_OPTION);
 *			return null;
 *		}
 *	}
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class EJBProxyCore extends BaseModel {

	/** 接続先環境設定プロパティです。 */
	private Hashtable<String, String> environment; 
	
	/** 使用するEJBコンポーネントのJNDI名です。 */
	private String lookupName;
	
	/** 呼び出すEJBメソッド名です。 */
	private String methodName;
	
	/** EJBメソッドパラメータです。 */
	private TypeMaintainList parameters;

	/**
	 * <p>[概 要] </p>
	 * 接続先環境設定プロパティを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * environmentフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 接続先環境設定プロパティ
	 */
	public Hashtable<String, String> getEnvironment() {
		if(environment == null) {
			environment = getController().getClientConfig().getDefaultEjbEnvironment();
		}
		return environment;
	}

	/**
	 * <p>[概 要] </p>
	 * 接続先環境設定プロパティを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * environmentフィールドに引数environmentを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param environment 接続先環境設定プロパティ
	 */
	public void setEnvironment(Hashtable<String, String> environment) {
		this.environment = environment;
	}

	/**
	 * <p>[概 要] </p>
	 * 使用するEJBコンポーネントのJNDI名を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * lookupNameフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 使用するEJBコンポーネントのJNDI名
	 */
	public String getLookupName() {
		return lookupName;
	}

	/**
	 * <p>[概 要] </p>
	 * 使用するEJBコンポーネントのJNDI名を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * lookupNameフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param lookupName 使用するEJBコンポーネントのJNDI名
	 */
	public void setLookupName(String lookupName) {
		this.lookupName = lookupName;
	}

	/**
	 * <p>[概 要] </p>
	 * 呼び出すEJBメソッド名を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * methodNameフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 呼び出すEJBメソッド名
	 */
	public String getMethodName() {
		return methodName;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 呼び出すEJBメソッド名を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * methodNameフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param methodName 呼び出すEJBメソッド名
	 */
	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return EJBメソッドパラメータ
	 */
	public TypeMaintainList getParameters() {
		return parameters;
	}

	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameters EJBメソッドパラメータ
	 */	
	public void setParameters(TypeMaintainList parameters) {
		this.parameters = parameters;
	}
	
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(Object parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(char parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(short parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(int parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(long parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(float parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(double parameter) {
		this.parameters.add(parameter);
	}
	/**
	 * <p>[概 要] </p>
	 * EJBメソッドパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドに引数parameterを追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameter EJBメソッドパラメータ
	 */
	public void addParameter(boolean parameter) {
		this.parameters.add(parameter);
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * フィールドの初期化を行います。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public EJBProxyCore() {
		this.parameters = new TypeMaintainList();
	}
	
	/**
	 * <p>[概 要] </p>
	 * EJBオブジェクトをルックアップしてサーバと通信を行います。
	 * 
	 * <p>[詳 細] </p>
	 * 以下の処理を行います。
	 * 
	 * <ol>
	 *   <li>environmentフィールド値を元にInitialContextを生成</li>
	 *   <li>lookupNameのルックアップ</li>
	 *   <li>呼び出すメソッドのリフレクション</li>
	 *   <li>EJBメソッド実行</li>
	 * </ol>
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	@Override
	protected void mainproc() throws NamingException, 
										SecurityException, 
										NoSuchMethodException, 
										IllegalArgumentException, 
										IllegalAccessException, 
										InvocationTargetException  
	{
		InitialContext ctx = new InitialContext(getEnvironment());
		Object ejb = ctx.lookup(getLookupName());
		
		Class[] parameterTypes = getParameters().toTypeArray();
		Method method = ejb.getClass().getMethod(getMethodName(), parameterTypes);
		Object result = method.invoke(ejb, getParameters().toValueArray());

		setResult(result);
	}

	/**
	 * <p>[概 要] </p>
	 * モデル処理成功イベント、モデル処理終了イベントを発行します。
	 * 
	 * <p>[詳 細] </p>
	 * fireModelSuccess、fireModelFinishedメソッドをコールします。
	 *  
	 * <p>[備 考] </p>
	 *  
	 */
	@Override
	protected void postproc() throws Exception {
		ModelProcessEvent successEvent = new ModelProcessEvent(this);
		successEvent.setResult(getResult());
		fireModelSuccess(successEvent);
		
		fireModelFinished(new ModelProcessEvent(this));
	}
}

