/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.validator;

import java.awt.Component;
import java.util.Properties;

import jp.co.fujitsu.reffi.client.swing.parser.ComponentValue;
import jp.co.fujitsu.reffi.client.swing.parser.ComponentValues;
import jp.co.fujitsu.reffi.common.util.ResourceUtil;

/**
 * <p>[概 要] </p>
 * 文字列長範囲バリデータクラスです。
 * 
 * <p>[詳 細] </p>
 * コンポーネントの文字列値を桁数判定して範囲内であるかどうか調べます。
 * 
 * <p>[備 考] </p>
 *
 * <b>使用例）</b><br>
 * BaseAction継承クラス#validatorsメソッドで以下のように使用します。
 * <pre class="samplecode">
 *	&#064;Override
 *	protected void validators(List<CustomValidator> validators) {
 *		// バリデーション対象コンポーネントを引数にしてインスタンス生成
 *		RangeValidator range_channelName = 
 *			new RangeValidator(getComponent("channelName"));
 *		// 独自のエラーメッセージを使う場合 
 *		range_channelName.setErrorMessage("チャンネル名は1～20文字で指定して下さい");
 *		// 文字数範囲（1文字～20文字）を設定
 *		range_channelName.setRange(1,20);
 *
 *		// 引数validatorsに追加
 *		validators.add(range_channelName);
 *	}
 * </pre>
 *
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class RangeValidator extends Validator {

	/** エラーメッセージ取得ソースです。 */
	private final String MESSAGE_RESOURCE = "jp.co.fujitsu.reffi.common.exception.corelogic_message";

	/** 最大文字長です。 */
	private int max;

	/** 最小文字長です。 */
	private int min;

	/**
	 * <p>[概 要] </p>
	 * 許容範囲の文字列長を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * minフィールドに引数minを、maxフィールドに引数maxを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param min 最小文字長数
	 * @param max 最大文字長数
	 */
	public void setRange(int min, int max) {
		this.min = min;
		this.max = max;
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 *   
	 * <p>[詳 細] </p>
	 * 引数element付きでsuper()を呼び出します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param component 
	 */
	public RangeValidator(Component component) {
		super(component);
	}

	/**
	 * <p>[概 要] </p>
	 * エラー文言登録メソッドです。
	 *  
	 * <p>[詳 細] </p>
	 * jp.co.fujitsu.reffi.shared.exception.corelogic_message.properties
	 * から対応するエラーメッセージを取得して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return エラー時メッセージ
	 */
	@Override
	public String registerErrorMessage() {
		Properties messages = ResourceUtil.instance.asProperties(MESSAGE_RESOURCE);
		return messages.getProperty("EFC2003");
	}

	/**
	 * <p>[概 要] </p>
	 * 桁数バリデーションを行います。
	 *  
	 * <p>[詳 細] </p>
	 * コンポーネントの文字列値を桁数判定して範囲内でなければfalseを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param values コンポーネント値
	 * @return エラー有：false、エラー無：true
	 */
	@Override
	protected boolean validate(ComponentValues values) {

		int max = this.max;
		int min = this.min;
		for (int i = 0; i < values.size(); i++) {
			ComponentValue elementValue = values.getComponentValue(i);
			//TODO trim() をオプションでできるように
			String v = elementValue.getValue().toString();
			if (v.length() < min || v.length() > max) {
				return false;
			}
		}
		return true;
	}
}
