/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.util {
	import flash.errors.EOFError;
	import flash.utils.ByteArray;

	/**
	 * <p>[概 要]</p>
	 * 入力文字の文字種をチェックするユーティリティクラスです.
	 *
	 * <p>[詳 細]</p>
	 *
	 * <p>[備 考]</p>
	 *
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */
	public class CharTypeUtil
	{

		/** '0'～'9'の半角数字 */
		public static const TYPE_DIGIT:int=0x2;

		/** 'A'～'Z','a'～'z'の半角英字 */
		public static const TYPE_ALPHABET:int=0x3;

		/** 半角記号 */
		public static const TYPE_OTHER:int=0x4;

		/** 半角スペース(0x20) */
		public static const TYPE_SPACE:int=0x5;

		/** 全角文字（JIS第一水準、JIS第二水準） */
		public static const TYPE_WIDE:int=0x20;

		/** 外字（JIS第一水準、JIS第二水準以外の漢字） */
		public static const TYPE_ECS:int=0x40;

		/** 異常な文字（外字、半角カナ）が含まれている */
		public static const TYPE_ILLEGAL:int=TYPE_ECS;

		/** 検査結果OKを示す定数 */
		public static const OK:int=-1;

		/** エラーのあったマルチバイトを含む文字列インデックス */
		private static var errorIndex:int = 0;

		/** エラーのあったキャラクタタイプ */
		private static var causeType:int=0;

		private static function convertUnicodeToShiftJis(value:String):ByteArray
		{
			// Unicode→Shift_JISコードへ変換
			var inputByteArray:ByteArray=new ByteArray();
			inputByteArray.writeMultiByte(value.toString(), "shift_jis");

			return inputByteArray;
		}

		private static function convertShiftJisToUnicode(value:String, inputByteArray:ByteArray):void
		{
			//Shif_JISコード→Unicodeに逆変換
			inputByteArray.position=0;
			try
			{
				var unicode:String=inputByteArray.readMultiByte(inputByteArray.length, "utf-16");
				// 逆変換不能の場合、入力unicode文字コードに対応するShift-JISコードが存在しない(?に化ける)。
				if (unicode != value)
				{
					var index:int=0;
					if ((index=unicode.indexOf("?")) != -1)
					{
						errorIndex=index;
					}
					causeType=TYPE_ILLEGAL;
				}
			}
			catch (error:EOFError)
			{
				trace("<Error> " + error.message);
				throw error;
			}
		}

		/**
		 * <p>[概 要]</p>
		 * 文字列を構成する要素を調査します.
		 *
		 * <p>[詳 細]</p>
		 * 入力文字が半角数字、半角英字、半角記号、半角スペース、全角文字、
		 * それ以外の文字の各領域のどこに属するかを判定します。
		 * 全角文字を判定するにあたり、ActionScript内部であつかうUnicodeとShift_JISでは、
		 * キャラクタマッピングの関連性が無いため、入力文字をShift_JISコードのバイト配列へ変換してから、
		 * バイト単位で第一水準、第二水準を識別します。
		 *
		 * <p>[備 考]</p>
		 * http://ash.jp/ash/src/code/index.htm" キャラクタタイプ検出アルゴリズム参照元
		 * <p/>
		 *
		 * @param value 調査対象の文字列
		 * @param expect 調査の期待値
		 * @return TYPE_ASCIIなどの論理和
		 * @see #TYPE_CONTROL
		 * @see #TYPE_DIGIT
		 * @see #TYPE_ASCII
		 * @see #TYPE_WIDE
		 * @see #TYPE_ECS
		 * @see #TYPE_ILLEGAL
		 */
		public static function getType(value:String, expect:int):Boolean
		{
			var ret:Boolean=false;

			// Unicode→Shift_JISコードへ変換
			var inputByteArray:ByteArray=convertUnicodeToShiftJis(value);

			//Shif_JISコード→Unicodeに逆変換できるかを確認
			convertShiftJisToUnicode(value, inputByteArray);

			var buf:Array=byteArray2Array(inputByteArray);
			var multiIndex:int=1;
			var k:int=0;
			var b:int;
			var b1:int=0;
			for (var i:int=0; i < buf.length; i++)
			{
				b=buf[i];
				if (b < 0)
				{
					b+=256;
				}

				if (k == 0)
				{
					// 文字の1バイト目チェック
					if (b == 0x20)
					{
						// 半角スペース
						if (!(TYPE_SPACE == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x21 <= b) && (b <= 0x2F))
					{
						// 半角記号
						if (!(TYPE_OTHER == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x30 <= b) && (b <= 0x39))
					{
						// 数字('0'～'9')
						if (!(TYPE_DIGIT == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x3A <= b) && (b <= 0x40))
					{
						// 半角記号
						if (!(TYPE_OTHER == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x41 <= b) && (b <= 0x5A))
					{
						// 半角英字(A-Z)
						if (!(TYPE_ALPHABET == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x5B <= b) && (b <= 0x60))
					{
						// 半角記号
						if (!(TYPE_OTHER == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x61 <= b) && (b <= 0x7A))
					{
						// 半角英字(a-z)
						if (!(TYPE_ALPHABET == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x7B <= b) && (b <= 0x7E))
					{
						// 半角記号
						if (!(TYPE_OTHER == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
					}
					else if ((0x81 <= b) && (b <= 0x9F))
					{
						// 全角文字
						k=1;
						b1=b;
						if (!(TYPE_WIDE == expect))
						{
							errorIndex=multiIndex;
							break;
						}
					}
					else if ((0xE0 <= b) && (b <= 0xEF))
					{
						// 全角文字（漢字）
						k=2;
						b1=b;
						if (!(TYPE_WIDE == expect))
						{
							errorIndex=multiIndex;
							break;
						}

					}
					else
					{
						// 異常キャラクタ
						if (!(TYPE_ILLEGAL == expect))
						{
							errorIndex=multiIndex;
							break;
						}
						multiIndex++;
						break;
					}
				}
				else if (k == 1)
				{
					// 漢字の2バイト目チェック
					if (((0x40 <= b) && (b <= 0x7E)) || ((0x80 <= b) && (b <= 0xFC)))
					{
						k=0;
						if ((b1 == 0x87) && (b < 0x9E))
						{
							// NEC特殊文字
							if (!(TYPE_ILLEGAL == (expect)))
							{
								errorIndex=multiIndex;
								break;
							}

						}
						else if ((0x85 <= b1) && (b1 <= 0x87))
						{
							// 機種依存文字
							if (!(TYPE_ILLEGAL == (expect)))
							{
								errorIndex=multiIndex;
								break;
							}

						}
						else if ((b1 == 0x88) && (b < 0x9E))
						{
							// 機種依存文字
							if (!(TYPE_ILLEGAL == (expect)))
							{
								errorIndex=multiIndex;
								break;
							}

						}

					}
					else
					{
						// 異常キャラクタ
						if (!(TYPE_ILLEGAL == (expect)))
						{
							errorIndex=multiIndex;
							break;
						}

						break;
					}
					multiIndex++;
				}
				else if (k == 2)
				{
					// 漢字の2バイト目チェック
					if (((0x40 <= b) && (b <= 0x7E)) || ((0x80 <= b) && (b <= 0xFC)))
					{
						k=0;

						if (!(TYPE_ILLEGAL == expect))
						{
							errorIndex=multiIndex;
							break;
						}

					}
					else
					{
						// 異常キャラクタ
						if (!(TYPE_ILLEGAL == expect))
						{
							errorIndex=multiIndex;
							break;
						}

						break;
					}
					multiIndex++;
				}

				if (i == (buf.length - 1))
				{
					ret=true;
				}
			}


			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字が半角スペースだけで構成されているかどうか検証します.
		 *
		 * <p>[詳 細]</p>
		 *
		 *
		 * <p>[備 考]</p>
		 *
		 * @param value 入力文字
		 * @return CharTypeUtil.OK=半角スペースだけで構成されている, CharTypeUtil.errorIndex=それ以外の文字が含まれている。
		 */
		public static function isSpace(value:String):int
		{
			var ret:int;
			if (getType(value, TYPE_SPACE))
			{
				ret=CharTypeUtil.OK;
			}
			else
			{
				ret=CharTypeUtil.errorIndex;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字が半角記号(半角英数字を除く)だけで構成されているかどうか検証します.
		 *
		 * <p>[詳 細]</p>
		 *
		 *
		 * <p>[備 考]</p>
		 *
		 * @param value 入力文字
		 * @return CharTypeUtil.OK=半角記号だけで構成されている, CharTypeUtil.errorIndex=それ以外の文字が含まれている
		 */
		public static function isOther(value:String):int
		{
			var ret:int;
			if (getType(value, TYPE_OTHER))
			{
				ret=CharTypeUtil.OK;
			}
			else
			{
				ret=CharTypeUtil.errorIndex;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字が全角文字だけで構成されているかどうか検証します.
		 *
		 * <p>[詳 細]</p>
		 *
		 *
		 * <p>[備 考]</p>
		 * ここで定義する「全角文字」に当てはまらない全角文字は以下の通りです。<BR>
		 * ・NEC特殊文字<BR>
		 * ・IBM拡張漢字<BR>
		 * ・SJISコードの0xED40以降
		 * ・SJIS上未定義の文字コード(0xF0A0-0xF0FF,0xF1A0-0xF1FF,0xF2A0-0xF2FF,0xFEA0-0xFEFF)
		 * <p/>
		 *
		 * @param value 入力文字
		 * @return CharTypeUtil.OK=指定文字だけで構成されている, CharTypeUtil.errorIndex=それ以外の文字が含まれている
		 */
		public static function isWide(value:String):int
		{
			var ret:int;
			if (getType(value, TYPE_WIDE))
			{
				ret=CharTypeUtil.OK;
			}
			else
			{
				ret=CharTypeUtil.errorIndex;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字が半角数字だけで構成されているかどうか検証します.
		 *
		 * <p>[詳 細]</p>
		 *
		 *
		 * <p>[備 考]</p>
		 *
		 * @param value 入力文字
		 * @return CharTypeUtil.OK=半角数字だけで構成されている, CharTypeUtil.errorIndex=それ以外の文字が含まれている
		 */
		public static function isDigit(value:String):int
		{
			var ret:int;
			if (getType(value, TYPE_DIGIT))
			{
				ret=CharTypeUtil.OK;
			}
			else
			{
				ret=CharTypeUtil.errorIndex;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字が半角英字だけで構成されているかどうか検証します.
		 *
		 * <p>[詳 細]</p>
		 *
		 *
		 * <p>[備 考]</p>
		 *
		 * @param value 入力文字
		 * @return CharTypeUtil.OK=半角英字だけで構成されている, CharTypeUtil.errorIndex=それ以外の文字が含まれている
		 */
		public static function isAlphabet(value:String):int
		{
			var ret:int;
			if (getType(value, TYPE_ALPHABET))
			{
				ret=CharTypeUtil.OK;
			}
			else
			{
				ret=CharTypeUtil.errorIndex;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * ByteArray型をArray型に変換します.
		 *
		 * <p>[詳 細]</p>
		 * 入力文字を配列単位で検証するため、ByteArray型をArray型に変換します。
		 *
		 * <p>[備 考]</p>
		 * readByteで値を抽出した場合、上位３バイトがFFFFFFで埋められているため、
		 * 000000FFでANDをとります。
		 * <p/>
		 *
		 * @param byteArray ByteArray型変数
		 * @return Array型変数
		 *
		 */
		public static function byteArray2Array(byteArray:ByteArray):Array
		{
			var array:Array=new Array();
			byteArray.position=0;
			while (byteArray.position <= byteArray.length - 1)
			{
				array.push(byteArray.readByte() & 0x000000FF);
			}
			return array;
		}
	}
}
