/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.rpc {
	
	import flash.events.IEventDispatcher;
	import flash.utils.Dictionary;
	
	import mx.messaging.ChannelSet;
	import mx.messaging.config.ServerConfig;
	import mx.rpc.remoting.Operation;
	import mx.rpc.remoting.RemoteObject;
	
	/**
	 * <p>[概 要]</p>
	 * RemoteObjectによるリモートモジュール呼び出しをサポートする機能モデルクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * 呼び出されるサーバモジュールがjp.co.fujitsu.reffi.server.flex.web.controller.RPCController、
	 * 若しくはRPCController継承クラスであることを想定したRPCクライアントクラスです。<br />
	 * このクラスをRPC呼び出しクライアントとして選択すると、呼び出し予定のモジュールが
	 * RPCController一クラスに限定される為、サーバ側RPCモジュール定義を簡略化することが出来ます。
	 * <br /> 
	 * 接続するデフォルトdestination識別子は「remote-controller」です。<br />
	 * サーバ側のremoting-config.xmlに以下の定義を追加することでRPCControllerを起動することが出来ます。
	 * <listing version="3.0">
        &lt;!-- RPCControllerClientCoreが期待するdestination名で登録 --&gt;
        &lt;destination id="remote-controller"&gt;
            &lt;properties&gt;
                &lt;!-- RPCController継承クラスでも可 --&gt;
                &lt;source&gt;jp.co.fujitsu.reffi.server.flex.web.controller.RPCController&lt;/source&gt;
                &lt;scope&gt;application&lt;/scope&gt;
            &lt;/properties&gt;
        &lt;/destination&gt;
	 * </listing>
	 *
	 * このクラスによって呼び出されたサーバ側RPCControllerオブジェクトは、
	 * modelFQCNプロパティを元に業務ロジックが記述されたサーバ側モジュールをインスタンス化、
	 * 実行し、結果をRPCControllerClientCoreに返却します。<br />
	 * 
	 * <p>[備 考]</p>
	 * RPCControllerの詳細な説明はreff-server-flexのJavaDocを参照して下さい。
	 * 
	 * @example 都道府県コンボボックスを選択してサーバからデータを取得する
	 * 
	 * <listing version="3.0">
        public class WardListChangeAction extends BaseAction {
            
            override protected function reserveModels(models:Array):void {
                models.push(Class(RPCControllerClientCore));
            }
    
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch (index) {
                    case 0:
                        var postalCombo:ComboBox = getComponentByName("wardList") as ComboBox;
    
                        RPCControllerClientCore(next).modelFQCN = "demo.rpc.model.FetchPostalDataModel";
                        RPCControllerClientCore(next).addRpcParameter("selectedWard", postalCombo.selectedLabel);
                        break;                
                }
                return true;
            }
            
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                switch (index) {
                    case 0:
                        var postalTable:DataGrid = getComponentByName("postaltable") as DataGrid;
                        postalTable.dataProvider = ResultEvent(resultEvent).result;
                        break;                    
                }
            }
        }
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */ 
	public class RPCControllerClientCore extends RPCCore {
		
		public static const DEFAULT_DESTINATION:String = "remote-controller";
		
		// RPCコントローラの宛先
		private var _remoteDestination:String = DEFAULT_DESTINATION;
		
		// リモートでの実行モデルFQCN
		private var _modelFQCN:String;
		
		// RPC引数 
		private var _rpcParameter:Object = new Object();
		
		// チャンネルセット
		private static var _channelSets:Dictionary = new Dictionary();

		/**
		 * <p>[概 要]</p>
		 * RPCコントローラの宛先です.
		 * 
		 * <p>[詳 細]</p>
		 * remoting-config.xmlで設定されているRPCコントローラの宛先を文字列で設定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default remote-controller
		 */ 
		public function get remoteDestination():String{
			return this._remoteDestination;
		}
		public function set remoteDestination(remoteDestination:String):void{
			this._remoteDestination = remoteDestination;
		}

		/**
		 * <p>[概 要]</p>
		 * サーバリモートの実行モデルFQCNです.
		 * 
		 * <p>[詳 細]</p>
		 * RPCControllerに実行される、業務ロジックが記述されたモデルクラスです。<br />
		 * 業務ロジッククラスはjp.co.fujitsu.reffi.server.model.AbstractModelを継承したクラスです。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get modelFQCN():String{
			return this._modelFQCN;
		}
		public function set modelFQCN(modelFQCN:String):void{
			this._modelFQCN = modelFQCN;
		}
		
		/**
		 * <p>[概 要]</p>
		 * RPCメソッド呼び出し時のパラメータです.
		 * 
		 * <p>[詳 細]</p>
		 * サーバ側ではMap&lt;Object, Object&gt;に型変換されます。<br />
		 * 設定された値は、サーバ側AbstractModel継承業務ロジッククラス内で
		 * getRequest()、getRequestParameter(Object key)メソッドによって取得出来ます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get rpcParameter():Object{
			return this._rpcParameter;
		}
		public function set rpcParameter(rpcParameter:Object):void{
			this._rpcParameter = rpcParameter;
		}
		
		/**
		 * <p>[概 要]</p>
		 * RPCメソッド呼び出し時のパラメータに新規パラメータを追加します.
		 * 
		 * <p>[詳 細]</p>
		 * 設定された値は、サーバ側AbstractModel継承業務ロジッククラス内で
		 * getRequest()、getRequestParameter(Object key)メソッドによって取得出来ます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param key パラメータキー
		 * @param value パラメータ値
		 */ 
		public function addRpcParameter(key:Object, value:Object):Object {
			rpcParameter[key] = value;
			
			return rpcParameter;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 具象通信オブジェクトを生成します.
		 * 
		 * <p>[詳 細]</p>
		 * RemoteObjectオブジェクトを、destination = "remote-controller"で生成します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return RemoteObjectオブジェクト
		 */ 
		override protected function createConcreteService() : IEventDispatcher {
			return new RemoteObject(DEFAULT_DESTINATION);
		}
		
		/**
		 * <p>[概 要]</p>
		 * キャッシュモード実行時、通信オブジェクトを識別する為のIDを返却します.
		 * 
		 * <p>[詳 細]</p>
		 * remoteDestinationプロパティを返却します。<br/>
		 * cacheIdはcacheModeプロパティがtrueの場合のみ使用されます。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return 通信オブジェクトを識別する為のキャッシュID
		 */
		override protected function cacheId() : String {
			return this.remoteDestination;
		}

		/**
		 * <p>[概 要]</p>
		 * RPCリクエストを行います.
		 * 
		 * <p>[詳 細]</p>
		 * 指定されたChannelSet、サーバ側業務ロジッククラス、パラメータマップで、
		 * サーバ側で待機しているRPCController(デフォルトdestination：remote-controller)を呼び出します。
		 * 
		 * <p>[備 考]</p>
		 */ 
		override protected function send(dispatcher:IEventDispatcher) : Object {			
			
			// remoting-config.xmlで設定されているRPCコントローラ参照を取得
			var service:RemoteObject = dispatcher as RemoteObject;
			
			// チャンネル情報を取得
			var channelSetProc:ChannelSet;
			if (_channelSets.hasOwnProperty(this.remoteDestination)) {
				channelSetProc = _channelSets[this.remoteDestination];
			} else {
				try {
					var channelSetNew:ChannelSet = ServerConfig.getChannelSet(this.remoteDestination);
					channelSetProc = channelSetNew
					_channelSets[this.remoteDestination] = channelSetNew;
				} catch (e:Error) {
					throw e;
				}
			}
			
			// リモートオブジェクトを更新
			service.destination = this.remoteDestination;
			service.channelSet = channelSetProc;
			
			// 実行するリモートオブジェクトのFQCNをオーバーライドメソッドから取得
			var modelFQCN:String = this.modelFQCN;
			// パラメータ
			var rpcParameter:Object = this.rpcParameter;
			if (!rpcParameter.hasOwnProperty("model.fqcn")) {
				rpcParameter["model.fqcn"] = modelFQCN;
			}
			
			// メソッド名からオペレーションを取得			
			var operation:Operation = service.getOperation("invoke") as Operation;
			operation.arguments = new Array(this.rpcParameter);
			
			// RPCコントローラに指定したリモートオブジェクト＆メソッドを実行依頼
			return operation.send();
		}
		
		/**
		 * <p>[概 要]</p>
		 * 通信成功、失敗に関わらずコールされます.
		 *  
		 * <p>[詳 細]</p>
		 * 処理は有りません。
		 * 
		 * <p>[備 考]</p>
		 * 通信終了処理実装はonFinishをオーバーライドしてください。
		 */ 
		override protected final function finishConnect() : void {
			super.finishConnect();
		}
	}
}
