/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import net.morilib.db.expr.RelationExpression;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.SqlResponse;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.Relations;
import net.morilib.db.relations.TableRelation;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.relations.UnionAllRelation;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.ddl.SqlAlterTableAdd;
import net.morilib.db.sqlcs.ddl.SqlAlterTableDrop;
import net.morilib.db.sqlcs.ddl.SqlAlterTableModify;
import net.morilib.db.sqlcs.ddl.SqlAlterTableRenameColumn;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlDropTable;
import net.morilib.db.sqlcs.ddl.SqlTruncateTable;
import net.morilib.db.sqlcs.dml.SqlDelete;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlInsertSelect;
import net.morilib.db.sqlcs.dml.SqlInsertValues;
import net.morilib.db.sqlcs.dml.SqlJoin;
import net.morilib.db.sqlcs.dml.SqlRelation;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSetBinaryOperation;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import net.morilib.db.sqlcs.dml.SqlSubqueryRelation;
import net.morilib.db.sqlcs.dml.SqlTable;
import net.morilib.db.sqlcs.dml.SqlUpdate;
import net.morilib.db.sqlcs.dml.SqlWith;

public abstract class SqlEngine {

	/**
	 * 
	 */
	protected SqlSchema schema;

	/**
	 * 
	 * @param schema
	 */
	public SqlEngine(SqlSchema schema) {
		this.schema = schema;
	}

	public SqlSchema getSchema() {
		return schema;
	}

	public abstract Relation visit(SqlSelect t,
			RelationTuple p,
			List<Object> h) throws IOException, SQLException;

	public abstract RelationExpression visit(SqlExpression e,
			List<Object> h) throws SQLException;

	public abstract int visit(SqlInsertValues t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlInsertSelect t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlUpdate t,
			List<Object> h) throws IOException, SQLException;

	public abstract int visit(SqlDelete t,
			List<Object> h) throws IOException, SQLException;

	public abstract Object visit(SqlCreateTable c
			) throws IOException, SQLException;

	public abstract Object visit(SqlDropTable c
			) throws IOException, SQLException;

	public abstract Object visit(SqlTruncateTable c
			) throws IOException, SQLException;

	public abstract Object visit(SqlAlterTableAdd c
			) throws IOException, SQLException;

	public abstract Object visit(SqlAlterTableModify c
			) throws IOException, SQLException;

	public abstract Object visit(SqlAlterTableDrop c
			) throws IOException, SQLException;

	public abstract Object visit(SqlAlterTableRenameColumn c
			) throws IOException, SQLException;

	public abstract void commit() throws IOException, SQLException;

	public abstract void rollback() throws IOException, SQLException;

	public Relation visit(SqlRelation t,
			List<Object> h) throws IOException, SQLException {
		SqlSubqueryRelation ts;
		Relation a, b;
		SqlTable tt;
		SqlJoin tj;

		if(t instanceof SqlTable) {
			tt = (SqlTable)t;
			return schema.readRelation(tt.getName(), tt.getAs());
		} else if(t instanceof SqlJoin) {
			tj = (SqlJoin)t;
			a  = visit(tj.getJoinee(), h);
			b  = visit(tj.getJoiner(), h);
			return Relations.join(
					this, schema, visit(tj.getOn(), h), a, b, null,
					null, tj.getType(), h);
		} else if(t instanceof SqlSubqueryRelation) {
			ts = (SqlSubqueryRelation)t;
			return new TableRenameRelation(
					visit(ts.getSubquery(), Relations.NULLTUPLE, h),
					ts.getAs());
		} else {
			throw new RuntimeException();
		}
	}

	Relation _union(Relation a,
			Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l;
		Map<String, Object> m, n;
		RelationCursor c;
		List<SqlColumnDefinition> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet()) {
				n.put(p.get(i++).getName(), m.get(s));
			}
			l.add(new DefaultRelationTuple(n));
		}
		return new TableRelation(p, l);
	}

	Relation _intersect(Relation a,
			Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l, q;
		Map<String, Object> m, n;
		RelationCursor c;
		List<SqlColumnDefinition> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		q = new LinkedHashSet<RelationTuple>();
		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet()) {
				n.put(p.get(i++).getName(), m.get(s));
			}
			q.add(new DefaultRelationTuple(n));
		}

		l.retainAll(q);
		return new TableRelation(p, l);
	}

	Relation _except(Relation a,
			Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l;
		Map<String, Object> m, n;
		RelationCursor c;
		RelationTuple t;
		List<SqlColumnDefinition> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet()) {
				n.put(p.get(i++).getName(), m.get(s));
			}
			t = new DefaultRelationTuple(n);
			if(l.contains(t))  l.remove(t);
		}
		return new TableRelation(p, l);
	}

	private boolean _in(RelationTuple t, RelationTuple v,
			Relation b) throws SQLException {
		for(SqlColumnDefinition s : b.getColumnNames()) {
			if(!t.get(s.getName()).equals(v.get(s.getName()))) {
				return false;
			}
		}
		return true;
	}

	Relation _divide(Relation a,
			Relation b) throws IOException, SQLException {
		Map<String, Object> n = new LinkedHashMap<String, Object>();
		Collection<RelationTuple> l = null, m;
		RelationCursor c, j;
		RelationTuple t, v;
		List<SqlColumnDefinition> p;
		Iterator<SqlColumnDefinition> i;
		SqlColumnDefinition d;

		p = new ArrayList<SqlColumnDefinition>(a.getColumnNames());
		i = p.iterator();
		while(i.hasNext()) {
			d = i.next();
			for(SqlColumnDefinition g : b.getColumnNames()) {
				if(d.getName().equals(g.getName())) {
					i.remove();
				}
			}
		}

		c = b.iterator();
		for(; c.hasNext(); ) {
			t = c.next();
			j = a.iterator();
			m = new LinkedHashSet<RelationTuple>();
			while(j.hasNext()) {
				v = j.next();
				if(_in(v, t, b)) {
					for(SqlColumnDefinition s : p) {
						n.put(s.getName(), v.get(s.getName()));
					}
					m.add(new DefaultRelationTuple(n));
				}
			}

			if(l == null) {
				l = m;
			} else {
				l.retainAll(m);
			}
		}

		if(l == null) {
			throw ErrorBundle.getDefault(10003);
		}
		return new TableRelation(p, l);
	}

	public Relation visit(SqlSetExpression s,
			List<Object> h) throws IOException, SQLException {
		SqlSetBinaryOperation b;
		Relation x, y;

		if(s instanceof SqlSetBinaryOperation) {
			b = (SqlSetBinaryOperation)s;
			x = visit(b.getExpression1(), h);
			y = visit(b.getExpression2(), h);
			switch(b.getOpeator()) {
			case UNION:      return _union(x, y);
			case UNION_ALL:  return new UnionAllRelation(x, y);
			case INTERSECT:  return _intersect(x, y);
			case MINUS:      return _except(x, y);
			case DIVIDE:     return _divide(x, y);
			}
		} else if(s instanceof SqlSelect) {
			return visit((SqlSelect)s, Relations.NULLTUPLE, h);
		} else {
			throw new NullPointerException();
		}
		return null;
	}

	public Object visit(Object o,
			List<Object> h) throws IOException, SQLException {
		if(o instanceof SqlSetExpression) {
			return visit((SqlSetExpression)o, h);
		} else if(o instanceof SqlInsertValues) {
			return visit((SqlInsertValues)o, h);
		} else if(o instanceof SqlInsertSelect) {
			return visit((SqlInsertSelect)o, h);
		} else if(o instanceof SqlUpdate) {
			return visit((SqlUpdate)o, h);
		} else if(o instanceof SqlDelete) {
			return visit((SqlDelete)o, h);
		} else if(o instanceof SqlCreateTable) {
			return visit((SqlCreateTable)o);
		} else if(o instanceof SqlDropTable) {
			return visit((SqlDropTable)o);
		} else if(o instanceof SqlTruncateTable) {
			return visit((SqlTruncateTable)o);
		} else if(o instanceof SqlAlterTableAdd) {
			return visit((SqlAlterTableAdd)o);
		} else if(o instanceof SqlAlterTableModify) {
			return visit((SqlAlterTableModify)o);
		} else if(o instanceof SqlAlterTableDrop) {
			return visit((SqlAlterTableDrop)o);
		} else if(o instanceof SqlAlterTableRenameColumn) {
			return visit((SqlAlterTableRenameColumn)o);
		} else if(o instanceof SqlWith) {
			return visit((SqlWith)o);
		} else {
			throw ErrorBundle.getDefault(10039);
		}
	}

	public Object execute(String t,
			List<Object> h) throws IOException, SQLException {
		DbSqlLexer l;

		l = new DbSqlLexer(new StringReader(t));
		return visit(new DbSqlParser().parse(l),
				Collections.emptyList());
	}

	public Object execute(String t
			) throws IOException, SQLException {
		return execute(t, Collections.emptyList());
	}

	public Object visit(SqlWith c) throws IOException, SQLException {
		Map<String, SqlSetExpression> m;

		m = c.getWith();
		for(String x : m.keySet()) {
			schema.bindSchema(x,
					visit(m.get(x), Collections.emptyList()));
		}
		return new SqlResponse(true, "With statement processed");
	}

}
