/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.expr;

import java.sql.SQLException;
import java.util.EnumMap;

import net.morilib.db.misc.DbBool;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.LikeWildcard;
import net.morilib.db.misc.Rational;
import net.morilib.db.sqlcs.dml.SqlBinaryOperator;

public abstract class RelationBinaryOperator {

	public static final RelationBinaryOperator
	ADD = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			if(a instanceof Rational && b instanceof Rational) {
				return ((Rational)a).add((Rational)b);
			} else if(a.equals("")) {
				return b;
			} else if(b.equals("")) {
				return a;
			} else {
				return a.toString() + b.toString();
			}
		}

	};

	public static final RelationBinaryOperator
	SUB = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			if(a instanceof Rational && b instanceof Rational) {
				return ((Rational)a).subtract((Rational)b);
			} else if(a.equals("")) {
				return b;
			} else if(b.equals("")) {
				return a;
			} else {
				throw ErrorBundle.getDefault(10040, "-");
			}
		}

	};

	public static final RelationBinaryOperator
	MUL = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			if(a instanceof Rational && b instanceof Rational) {
				return ((Rational)a).multiply((Rational)b);
			} else if(a.equals("")) {
				return b;
			} else if(b.equals("")) {
				return a;
			} else {
				throw ErrorBundle.getDefault(10040, "*");
			}
		}

	};

	public static final RelationBinaryOperator
	DIV = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			if(a instanceof Rational && b instanceof Rational) {
				try {
					return ((Rational)a).divide((Rational)b);
				} catch(ArithmeticException e) {
					throw ErrorBundle.getDefault(10010, "/");
				}
			} else if(a.equals("")) {
				return b;
			} else if(b.equals("")) {
				return a;
			} else {
				throw ErrorBundle.getDefault(10040, "/");
			}
		}

	};

	public static final RelationBinaryOperator
	CONCAT = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return a.toString() + b.toString();
		}

	};

	public static final RelationBinaryOperator
	EQ = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) == 0);
		}

	};

	public static final RelationBinaryOperator
	NE = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) != 0);
		}

	};

	public static final RelationBinaryOperator
	LT = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) < 0);
		}

	};

	public static final RelationBinaryOperator
	LE = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) <= 0);
		}

	};

	public static final RelationBinaryOperator
	GT = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) > 0);
		}

	};

	public static final RelationBinaryOperator
	GE = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					RelationExpression.cmp(a, b) >= 0);
		}

	};

	public static final RelationBinaryOperator
	AND = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(bool(a).and(bool(b)));
		}

	};

	public static final RelationBinaryOperator
	OR = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(bool(a).or(bool(b)));
		}

	};

	public static final RelationBinaryOperator
	LIKE = new RelationBinaryOperator() {

		@Override
		public Object op(Object a, Object b) throws SQLException {
			return RelationExpression.value(
					LikeWildcard.matches(b.toString(), a.toString()));
		}

	};

	private static
	EnumMap<SqlBinaryOperator, RelationBinaryOperator> map;

	static {
		map = new EnumMap<SqlBinaryOperator, RelationBinaryOperator>(
				SqlBinaryOperator.class);
		map.put(SqlBinaryOperator.ADD, ADD);
		map.put(SqlBinaryOperator.SUB, SUB);
		map.put(SqlBinaryOperator.MUL, MUL);
		map.put(SqlBinaryOperator.DIV, DIV);
		map.put(SqlBinaryOperator.CONCAT, CONCAT);
		map.put(SqlBinaryOperator.EQ, EQ);
		map.put(SqlBinaryOperator.NE, NE);
		map.put(SqlBinaryOperator.LT, LT);
		map.put(SqlBinaryOperator.LE, LE);
		map.put(SqlBinaryOperator.GT, GT);
		map.put(SqlBinaryOperator.GE, GE);
		map.put(SqlBinaryOperator.AND, AND);
		map.put(SqlBinaryOperator.OR, OR);
		map.put(SqlBinaryOperator.LIKE, LIKE);
	}

	static DbBool bool(Object o) {
		return RelationExpression.bool(o);
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static RelationBinaryOperator get(SqlBinaryOperator s) {
		return map.get(s);
	}

	/**
	 * 
	 * @param a
	 * @param b
	 * @return
	 */
	public abstract Object op(Object a, Object b) throws SQLException;

}
