// -*- Mode: java -*-
//Header:
//File: BoardBase.java
//Author: NODA, Itsuki
//Date: 2002/01/10
//Copyright (C) 2002 by Itsuki NODA, AIST
//EndHeader:

import java.io.* ;
import java.net.* ;
import java.awt.* ;
import javax.swing.* ;
import java.awt.event.* ;
import java.util.* ;

import Side ;
import CellBase ;
import Sexp ;

//======================================================================
// class BoardBase

public class BoardBase {

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // constants

    static final int size = 8 ;
    static final int pieceN = 12 ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // cell

    public CellBase cell[][] ;

    boolean isBlack(int x, int y) { return cell[x][y].isBlack() ; } ;
    boolean isWhite(int x, int y) { return cell[x][y].isWhite() ; } ;
    boolean isEmpty(int x, int y) { return cell[x][y].isEmpty() ; } ;

    void turnBlack(int x, int y) { cell[x][y].turnBlack() ; } ;
    void turnWhite(int x, int y) { cell[x][y].turnWhite() ; } ;
    void turnEmpty(int x, int y) { cell[x][y].turnEmpty() ; } ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // next side

    public Side myside ;

    public Side nextSide ;
    public int turn ;

    public void turnSide() {
	turn++ ;
	nextSide = nextSide.otherside() ;
    } ;

    public boolean isMyTurn() {
	return myside.isSame(nextSide) ;
    } ;

    public boolean gameoverp ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // marking

    public CellBase markedCell ;

    public boolean mark(int x, int y, boolean forcep) {
	if(markedCell != null) {
	    if(!forcep) return false ;
	} 

	clearMark() ;

	CellBase c = cell[x][y] ;

	if(c.isEmpty()) return false ;

	markedCell = c ;
	c.mark() ;

	return true ;
    } ;

    public void clearMark() {
	if(markedCell != null) {
	    markedCell.unmark() ;
	    markedCell = null ;
	}
    } ;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // logging

    public boolean loggingp_ ;
    public PrintStream logstr ;
    public Sexp logsexp ;
    public Sexp lastlog ;

    public boolean loggingp() { return loggingp_ ; } ;

    public void logging(Sexp sexp) {
	//logsexp.addLast(sexp) ;
	logsexp = logsexp.push(sexp) ;
	lastlog = sexp ;
	if(logstr != null) {
	    logstr.println(sexp) ;
	} ;
    } ;

    public boolean setLogstr(PrintStream str) {
	logstr = str ;
	return true ;
    } ;

    public void loggingOn() { loggingOn(System.out) ; } ;

    public void loggingOn(PrintStream str) {
	logstr = str ;
	loggingp_ = true ;
    } ;

    public void loggingOff() { 
	logstr = null ;
	loggingp_ = false ;
    } ;

    public void clearLog() {
	loggingOff() ;
	logsexp = Sexp.Nil ;
    } ;

    public void saveLog(String filename) {
	try {
	    saveLog(new FileOutputStream(filename)) ;
	} catch(java.io.FileNotFoundException ex) {
	} ;
    } ;

    public void saveLog(OutputStream str) {
	saveLog(new PrintWriter(str)) ;
    } ;

    public void saveLog(PrintWriter writer) {
	for(Sexp s = logsexp.reverse() ; s.isCons() ; s = s.cdr()) {
	    writer.println(s.car()) ;
	} 
    } ;
    
    //------------------------------------------------------------
    // initialize

    public BoardBase() {
	init(Side.none) ;
    } ;

    public BoardBase(Side side) {
	init(side) ;
    } ;

    //------------------------------
    // init

    public void init(Side side) {
	initCell(size) ;
	markedCell = null ;
	nextSide = Side.black ;
	myside = side ;
	turn = 0 ;
	gameoverp = false ;

	clearLog() ;
    } ;

    //------------------------------
    // init cells

    void initCell(int s) {
	allocCell(s) ;
	setup() ;
    } ;

    //------------------------------
    // allocate cells

    void allocCell(int s) {
	cell = new CellBase[s][] ;
	for(int i = 0 ; i < s ; i++) {
	    cell[i] = new CellBase[s] ;
	    for(int j = 0 ; j < s ; j++) {
		cell[i][j] = new CellBase(this,i,j) ;
	    }
	}
    } ;

    //------------------------------
    // setup stones on the initial position

    void setup() {
	for(int k = 1 ; k < size-1 ; k++) {
	    turnWhite(0,k) ;
	    turnWhite(size-1,k) ;
	    turnBlack(k,0) ;
	    turnBlack(k,size-1) ;
	}
    } ;


    //------------------------------------------------------------
    // find movable cell

    //----------------------------------------
    // check the position (x,y) is on board

    boolean onBoard(int x, int y) {
	return (0 <= x && x < size && 0 <= y && y < size) ;
    } ;
    
    //----------------------------------------
    // 4 line direction 

    static final int dirX = 1 ;		// N-S
    static final int dirY = 2 ;		// E-W
    static final int dirXY = 3 ;	// NE-SW
    static final int dirYX = 4 ;	// NW-SE

    //----------------------------------------
    // count stone in a line

    int countStone(int x, int y, int dir) {
	int dx = 0 ;
	int dy = 0 ;
	switch(dir) {
	case dirX: dx = 1 ; dy = 0 ; break ;
	case dirY: dx = 0 ; dy = 1 ; break ;
	case dirXY: dx = 1 ; dy = 1 ; break ;
	case dirYX: dx = 1 ; dy = -1 ; break ;
	default: return 0 ;
	}
	
	int c = 0 ;
	for(int i = -size ; i < size ; i++) {
	    int xx = x + dx * i ;
	    int yy = y + dy * i ;
	    if(onBoard(xx,yy)) {
		if(!cell[xx][yy].isEmpty()) c++ ;
	    } 
	} 
	
	return c ;
    } ;

    //----------------------------------------
    // 8 move direction

    static final int dirXp = 5 ;	// N
    static final int dirXn = 6 ;	// S
    static final int dirYp = 7 ;	// E
    static final int dirYn = 8 ;	// W
    static final int dirXYp = 9 ;	// NE
    static final int dirXYn = 10 ;	// SW
    static final int dirYXp = 11 ;	// NW
    static final int dirYXn = 12 ;	// SE

    //----------------------------------------
    // check movable or not

    public CellBase movableTo(int x, int y, int dir, int l) {
	int dx = 0 ; 
	int dy = 0 ;
	switch(dir) {
	case dirXp:  dx =  1 ; dy =  0 ; break ;
	case dirXn:  dx = -1 ; dy =  0 ; break ;
	case dirYp:  dx =  0 ; dy =  1 ; break ;
	case dirYn:  dx =  0 ; dy = -1 ; break ;
	case dirXYp: dx =  1 ; dy =  1 ; break ;
	case dirXYn: dx = -1 ; dy = -1 ; break ;
	case dirYXp: dx =  1 ; dy = -1 ; break ;
	case dirYXn: dx = -1 ; dy =  1 ; break ;
	default: return null ;
	}

	CellBase home = cell[x][y] ;
	Side side = home.state ;

	CellBase r = null ;

	for(int i = 1 ; i < l+1 ; i++) {
	    int xx = x + dx * i ;
	    int yy = y + dy * i ;

	    if(!onBoard(xx,yy)) return null ;

	    CellBase c = cell[xx][yy] ;

	    if(i == l) { // stop position
		if(c.isStoppable(side)) r = c ;
	    } else {
		if(!c.isPassable(side)) break ;
	    }
	}
	return r ;
    } ;

    //----------------------------------------
    // check and mark all movable cell

    public void markMovable(int x, int y) {
	clearMovable() ;
	int lx = countStone(x,y,dirX) ;
	int ly = countStone(x,y,dirY) ;
	int lxy = countStone(x,y,dirXY) ;
	int lyx = countStone(x,y,dirYX) ;

	CellBase c ;
	c = movableTo(x,y,dirXp,lx)   ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirXn,lx)   ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirYp,ly)   ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirYn,ly)   ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirXYp,lxy) ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirXYn,lxy) ; if(c != null) c.setMovable() ;	
	c = movableTo(x,y,dirYXp,lyx) ; if(c != null) c.setMovable() ;
	c = movableTo(x,y,dirYXn,lyx) ; if(c != null) c.setMovable() ;

    } ;

    //----------------------------------------
    // clear movable flag

    public void clearMovable() {
	for(int x = 0 ; x < size ; x++) {
	    for(int y = 0 ; y < size ; y++) {
		cell[x][y].unsetMovable() ;
	    }
	}
    } ;

    //------------------------------------------------------------
    // move

    public boolean move(int fx, int fy, int tx, int ty) {
	return move(fx,fy,tx,ty,true) ;
    } ;

    public boolean move(int fx, int fy, int tx, int ty, boolean checksidep) {

	if(gameoverp) return false ;

	CellBase from = cell[fx][fy] ;

	/*
	 * check the from-cell.
	 * if the cell is empty, return immediately with false.
	 * if checksidep is true and the from-cell is not the nextSide,
	 * then return false.
	 */
	if(checksidep) {
	    if(!from.isSame(nextSide)) return false ;
	} else {
	    if(from.isEmpty()) return false ;
	}
	
	/* check and mark movable cells */
	markMovable(fx,fy) ;

	CellBase to = cell[tx][ty] ;

	/* if the to-cell is not movable, then return false. */
	if(!to.isMovable()) return false ;

	boolean pickedp = !(to.isEmpty()) ; /* if non-empty, existing stone
					     * is picked. 
					     */
	/* move the stone */
	to.turnSame(from.state) ;
	from.turnEmpty() ;

	/* clear flags */
	from.unmark() ;
	clearMovable() ;

	/* logging */
	if(loggingp()) {
	    logging(moveSexp(nextSide,fx,fy,tx,ty,pickedp)) ;
	} 

	/* turn next side */
	turnSide() ;

	return true ;
    } ;

    //------------------------------------------------------------
    // check game over

    public Side isGameOver() {
	Side s = Side.none ;
	if(isWin(Side.black)) s = Side.black ;
	if(isWin(Side.white)) s = Side.white ;

	/* logging */
	if(loggingp() && !s.isNone()) {
	    logging(gameoverSexp(s)) ;
	} 

	return s ;
    } ;

    public boolean isWin(Side side) {
	CellBase c = null;

	/* clear check flag */
	for(int x = 0 ; x < size ; x++) {
	    for(int y = 0 ; y < size ; y++) {
		CellBase c0 = cell[x][y] ;
		c0.check = false ;
		if(c0.isSame(side) && c == null) c = c0 ;
	    } 
	}

	/* propagate check flag from a cell */
	propagateCheck(c) ;

	/* check all colored cell is checked */
	for(int x = 0 ; x < size ; x++) {
	    for(int y = 0 ; y < size ; y++) {
		if(cell[x][y].isSame(side) && !cell[x][y].check)
		    return false ;
	    }
	}
	return true ;
    } ;

    //------------------------------------------------------------
    // check and mark connectivity

    public void propagateCheck(CellBase c) {
	c.check = true ;
	for(int dx = -1 ; dx <= 1 ; dx++) {
	    for(int dy = -1 ; dy <= 1 ; dy++) {
		int xx = c.x + dx ;
		int yy = c.y + dy ;
		if(onBoard(xx,yy)) {
		    CellBase c1 = cell[xx][yy] ;
		    if(c != c1 && c1.isSame(c.state) && !c1.check) {
			propagateCheck(c1) ;
		    }
		}
	    }
	}
    } ;

    //------------------------------------------------------------
    // generate sexp & string

    //----------------------------------------
    // move sexp/string

    public Sexp moveSexp(Side side, int fx, int fy, int tx, int ty, 
			 boolean pickedp) {
	Sexp from = Sexp.factory().list(new Long(fx), new Long(fy)) ;
	Sexp to = Sexp.factory().list(new Long(tx), new Long(ty)) ;
	Sexp s = Sexp.factory().list("move", side.toString(),
				     from, to) ;
	if(pickedp) {
	    s.addLast(Sexp.factory().list("pick")) ;
	} ;
	return s ;
    } ;

    public String moveString(Side side, int fx, int fy, int tx, int ty,
			     boolean pickedp) {
	return moveSexp(side,fx,fy,tx,ty,pickedp).toString() ;
    } ;

    //----------------------------------------
    // gameover sexp/string

    public Sexp gameoverSexp(Side side) {
	return Sexp.factory().list("gameover",side.toString()) ;
    } ;

    public String gameoverString(Side side) {
	return gameoverSexp(side).toString() ;
    } ;

    //----------------------------------------
    // state sexp

    public Sexp stateSexp() {
	Sexp black = Side.blackCharSexp ;
	Sexp white = Side.whiteCharSexp ;
	Sexp empty = Side.noneCharSexp ;
	
	Sexp s = Sexp.factory().list(Sexp.Nil) ;
	for(int i = 0 ; i < size ; i++) {
	    Sexp l = Sexp.factory().list(Sexp.Nil) ;
	    for(int j = 0 ; j < size ; j++) {
		Sexp c ;
		if      (cell[i][j].isBlack()) c = black ;
		else if (cell[i][j].isWhite()) c = white ;
		else                           c = empty ;
		l.addLast(c) ;
	    }
	    s.addLast(l.cdr()) ;
	}

	return Sexp.factory().list("state",new Long(turn), 
				   nextSide.toString(), s.cdr()) ;
    } ;

    //------------------------------------------------------------
    // show state

    public void showState() {
	System.out.println("[turn=" + turn + "]") ;
	for(int i = 0 ; i < size ; i++) {
	    for(int j = 0 ; j < size ; j++) {
		String s = "" ;
		if      (cell[i][j].isBlack()) s = "+" ;
		else if (cell[i][j].isWhite()) s = "-" ;
		else                           s = "." ;
		System.out.print(s) ;
	    }
	    System.out.println("") ;
	} ;
    } ;

    //------------------------------------------------------------
    // command interpreter

    public boolean execSexpCommand(Sexp s) throws Exception {
	if(!s.isCons())
	    throw new Exception("Illegal message:"
				+ "[not a list]:" 
				+ s) ;

	if        (s.car().equals("move")) {
	    execSexpMove(s) ;
	} else if (s.car().equals("state")) {
	    execSexpState(s) ;
	} else if (s.car().equals("setSide")) {
	    execSexpSetSide(s) ;
	} else if (s.car().equals("gameover")) {
	    execSexpGameOver(s) ;
	} else if (s.car().equals("quit")) {
	    execSexpQuit(s) ;
	} else {
	    throw new Exception("Illegal message:"
				+ "[unknown command]:"
				+ s) ;
	}
	return true ;
    } ;

    //------------------------------
    // move command

    public boolean execSexpMove(Sexp s) throws Exception {
	Sexp sideSexp = s.nth(1) ;
	Sexp fromSexp = s.nth(2) ;
	Sexp toSexp = s.nth(3) ;

	Side side = Side.scan(sideSexp.strVal()) ;
	if(side.isError()) {
	    throw new Exception("Illegal message:"
				+ "[unknown color]:"
				+ sideSexp + " in " + s) ;
	}
			
	if(!fromSexp.isCons()) 
	    throw new Exception("Illegal message:"
				+ "[wrong position format]:"
				+ fromSexp + " in " + s) ;

	int fx = (int)fromSexp.nth(0).intVal() ;
	int fy = (int)fromSexp.nth(1).intVal() ;
			
	if(!toSexp.isCons())
	    throw new Exception("Illegal message:"
				+ "[wrong position format]:"
				+ toSexp + " in " + s) ;
			
	int tx = (int)toSexp.nth(0).intVal() ;
	int ty = (int)toSexp.nth(1).intVal() ;

	move(fx,fy,tx,ty) ;
			
	return true ;
    } ;

    //------------------------------
    // state command

    public boolean execSexpState(Sexp s) throws Exception {
	Sexp turnSexp = s.nth(1) ;
	Sexp sideSexp = s.nth(2) ;
	Sexp stateSexp = s.nth(3) ;

	turn = (int)turnSexp.intVal() ;
	nextSide = Side.scan(sideSexp.strVal()) ;
	
	for(int i = 0 ; i < size ; i++) {
	    for(int j = 0 ; j < size ; j++) {
		Sexp c = stateSexp.nth(i).nth(j) ;
		cell[i][j].turnSame(Side.scan(c.strVal())) ;
	    }
	}

	return true ;
    } ;
	
    //------------------------------
    // set side

    public boolean execSexpSetSide(Sexp s) throws Exception {
	Sexp sideSexp = s.nth(1) ;

	myside = Side.scan(sideSexp.strVal()) ;

	return true ;
    } ;
	
    //------------------------------
    // gameover

    public boolean execSexpGameOver(Sexp s) throws Exception {
	gameoverp = true ;
	return true ;
    } ;

    //------------------------------
    // quit

    public boolean execSexpQuit(Sexp s) throws Exception {
	quit(false) ;
	return true ;
    } ;

    public void quit(boolean directp) { System.exit(0) ; } ;
	
    //============================================================
    // main

    //----------------------------------------
    // body

    static void run(String argv[]) {
	BoardBase b = new BoardBase(Side.none) ;
	System.out.println(b.stateSexp()) ;

	try {
	    while(true) {
		b.showState() ;
		Sexp com = Sexp.factory().scan(System.in) ;
		b.execSexpCommand(com) ;
	    }
	} catch (Exception ex) { } ; 
    } ;

    //----------------------------------------
    // test

    //--------------------
    // test1
    static void test1() {
	BoardBase b = new BoardBase() ;
    } ;

    //----------------------------------------
    // main

    public static void main(String argv[]) {
	//test1() ;
	run(argv) ;
    } ;

} ;

