// -*- Mode: java -*-
//Header:
//File: BoardWindow.java
//Author: NODA, Itsuki
//Date: 2002/01/13
//Copyright (C) 2002 by Itsuki NODA, AIST
//EndHeader:

import java.awt.* ;
import javax.swing.* ;
import java.awt.event.* ;

import Side ;
import Sexp ;
import CellBase ;
import SynchronizedBoard ;

//======================================================================
// class BoardWindow

public class BoardWindow implements ActionListener {

    //========================================
    // repaint thread

    static class RepaintThread extends Thread {
	SynchronizedBoard board ;
	BoardWindow window ;

	RepaintThread(SynchronizedBoard b, BoardWindow w) {
	    board = b ; 
	    window = w ;
	} ;

	public void run() {
	    try {
		synchronized(board) {
		    while(true) {
			board.wait() ;
			window.myPaintAll() ;
		    }
		}
	    } catch (java.lang.InterruptedException ex) {
		System.err.println(ex.getMessage()) ;
	    }
	} ;
	
    } ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // parent

    public RepaintThread repaintThread ;

    //============================================================
    // class CellWin

    class CellWin extends Button {

	BoardWindow frame ;

	int x ;
	int y ;

	CellBase cell ;
	
	public CellWin(BoardWindow f, int i, int j) { 
	    frame = f ;
	    x = i ; y = j ; 
	    cell = f.board.cell[x][y] ;
	} ;

	public void repaint(Graphics g) { paint(g) ; } ;
	
	public void paint(Graphics g) {

	    if (cell.isWhite()) {
		g.setColor(Color.white) ;
	    } else if (cell.isBlack()) {
		g.setColor(Color.black) ;
	    } else if ((x + y) % 2 == 0) {
		g.setColor(frame.grey0) ;
	    } else {
		g.setColor(frame.grey1) ;
	    }

	    g.fillArc(5,5,35,35,0,360) ;

	    if(cell.isMarked()) { g.setColor(Color.red) ; } ;

	    if(cell.isMovable()) { g.setColor(Color.blue) ; } ;

	    g.fillArc(15,15,15,15,0,360) ;

	} ;
	    
    } ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // parent

    public SynchronizedBoard board ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // constant
    //

    public int size() { return board.size ; } ;

    public final int cellWinSize = 50 ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Window Objects
    //

    public Frame mainWin ;

    public Panel infoWin ;
    public Label titleLabel ;
    public Label turnLabel ;
    public Label sideLabel ;
    public Button quitButton ;

    public Panel boardWin ;
    public Button cell_[] ;

    public Button cell(int i, int j) {
	return cell_[i * size() + j] ;
    } ;
    public Button setCell(int i, int j, Button c) {
	return cell_[i * size() + j] = c ;
    } ;

    static final float g0 = (float)0.4 ;
    static final float g1 = (float)0.6 ;
    Color grey0 ;
    Color grey1 ;

    //------------------------------------------------------------
    // initialize
 
    //----------------------------------------
    // constructor

    public BoardWindow(SynchronizedBoard b) { init(b) ; } ;

    //----------------------------------------
    // initialize

    public void init(SynchronizedBoard b) {
	board = b ;

	initWindow() ;
	
	repaintThread = new RepaintThread(b,this) ;
	repaintThread.start() ;
    } ;

    public void initWindow() {

	//System.err.println(b) ;

	boardWin = new Panel() ;
	boardWin.setLayout(new GridLayout(size(),size())) ;
	boardWin.setSize(50 * size(), 50 * size()) ;

	grey0 = new Color(g0,g0,g0) ;
	grey1 = new Color(g1,g1,g1) ;

	cell_ = new Button[size() * size()] ;
	for(int i = 0 ; i < size() ; i++) {
	    for(int j = 0 ; j < size() ; j++) {
		setCell(i,j,new CellWin(this,i,j)) ;
		cell(i,j).setSize(cellWinSize,cellWinSize) ;
		cell(i,j).addActionListener(this) ;
		if((i+j) % 2 == 0) {
		    cell(i,j).setBackground(grey0) ;
		} else {
		    cell(i,j).setBackground(grey1) ;
		} 
		boardWin.add(cell(i,j)) ;
	    } ;
	} ;

	titleLabel = new Label("Lines Of Action",Label.CENTER) ;
	turnLabel = new Label("next turn",Label.CENTER) ;

	if(board.myside.isBlack()) {
	    sideLabel = new Label("black",Label.CENTER) ;
	    sideLabel.setBackground(Color.black) ;
	    sideLabel.setForeground(Color.white) ;
	} else if (board.myside.isWhite()) {
	    sideLabel = new Label("white",Label.CENTER) ;
	    sideLabel.setBackground(Color.white) ;
	    sideLabel.setForeground(Color.black) ;
	} else {
	    sideLabel = new Label("black/white",Label.CENTER) ;
	    sideLabel.setBackground(grey0) ;
	}

	quitButton = new Button("quit") ;
	quitButton.addActionListener(this) ;

	titleLabel.setSize(20,100) ;
	turnLabel.setSize(20,100) ;
	sideLabel.setSize(20,100) ;
	quitButton.setSize(20,100) ;
	

	infoWin = new Panel() ;
	infoWin.setLayout(new BoxLayout(infoWin,BoxLayout.X_AXIS)) ;
	infoWin.add(titleLabel) ;
	infoWin.add(turnLabel) ;
	infoWin.add(sideLabel) ;
	infoWin.add(quitButton) ;
	paintTurnLabel() ;

	mainWin = new Frame() ;
	mainWin.setSize(boardWin.getWidth(),
			boardWin.getHeight() + infoWin.getHeight() + 20) ;
	mainWin.setLayout(new BorderLayout()) ;
	mainWin.add("North",infoWin) ;
	mainWin.add("Center",boardWin) ;

	mainWin.setVisible(true) ;

    } ;

    //------------------------------------------------------------
    // Action Listener method

    public void actionPerformed(ActionEvent e) {
	String arg = e.getActionCommand();

	Object s = e.getSource() ;

	if(s instanceof CellWin) {
	    pushCell((CellWin)s) ;
	} 
	else if ("quit".equals(arg)) {
	    board.quit(true) ;
	}

    }
    
    //----------------------------------------
    // action for push cell button

    public void pushCell(CellWin c) {

	if(!(!board.myside.isNone() && !board.isMyTurn())) {
	
	    CellBase mark = board.markedCell ;

	    if(mark == null) { // choose stone
		chooseAction(c) ;
	    } else { // move stone
		moveAction(mark.x,mark.y,c.x,c.y) ;
	    }
	}

	myPaintAll() ;

    } ;

    //----------------------------------------
    // choose action

    public void chooseAction(CellWin c) {
	if(c.cell.state == board.nextSide) {
	    boolean markp = board.mark(c.x,c.y,true) ;
	    board.clearMovable() ;
	    if(markp) {
		//c.frame.board.test_countStone(c.x,c.y) ;
		//c.frame.board.test_movableTo(c.x,c.y) ;
		board.markMovable(c.x,c.y) ;
	    }
	} else {
	    // different side. no action.
	}
    } ;

    //----------------------------------------
    // move action

    public void moveAction(int fx, int fy, int tx, int ty) {
	board.move(fx,fy,tx,ty) ;
	board.clearMark() ;
	board.clearMovable() ;
	board.isGameOver() ;
    } ;

    //----------------------------------------
    // paint turn label

    public void paintTurnLabel() {
	if(board.nextSide.isBlack()) {
	    turnLabel.setBackground(Color.black) ;
	    turnLabel.setForeground(Color.white) ;
	    turnLabel.setText("black") ;
	} else {
	    turnLabel.setBackground(Color.white) ;
	    turnLabel.setForeground(Color.black) ;
	    turnLabel.setText("white") ;
	}
    } ;

    //----------------------------------------
    // paintAll

    public void myPaintAll() {
	for(int i = 0 ; i < size() ; i++) {
	    for(int j = 0 ; j < size() ; j++) {
		cell(i,j).paintAll(cell(i,j).getGraphics()) ;
	    }
	}

	paintTurnLabel() ;

	//mainWin.paintAll(mainWin.getGraphics()) ;
	
    } ;

    //============================================================
    // main

    //----------------------------------------
    // run

    public static void run(String argv[]) {
	SynchronizedBoard b = new SynchronizedBoard() ;
	b.loggingOn(System.out) ;

	BoardWindow w = new BoardWindow(b) ;
    } ;

    //----------------------------------------
    // test

    //--------------------
    // test 1

    public static void test1() {} ;

    //----------------------------------------
    // main

    public static void main(String argv[]) {
	//test1() ;
	run(argv) ;
    } ;

} ;




