// -*- Mode: java -*-
//Header:
//File: ServerBoard.java
//Author: NODA, Itsuki
//Date: 2002/01/15
//Copyright (C) 2002 by Itsuki NODA, AIST
//EndHeader:

import java.io.* ;
import java.net.* ;
import java.awt.* ;
import javax.swing.* ;
import java.awt.event.* ;
import java.util.* ;

import Side ;
import CellBase ;
import SynchronizedBoard ;
import Sexp ;

//============================================================
// ServerBoard

public class ServerBoard extends SynchronizedBoard {

    static final int defaultPort = 16000 ;
    static final String defaultLogFile = "loa.log" ;

    //========================================
    // class Server Thread

    class ServerThread extends Thread {
	
	public ServerBoard board ;
	public int port ;
	public ServerSocket socket ;

	public ServerThread(ServerBoard b) { init(b) ; } ;
	public ServerThread(ServerBoard b, int p) { init(b,p) ; } ;

	public void init(ServerBoard b) {
	    init(b,b.defaultPort) ;
	} ;

	public void init(ServerBoard b, int p) {
	    board = b ;
	    port = p ;
	} ;

	public void run() {
	    try {
		socket = new ServerSocket(port) ;
		
		while(true) {
		    Socket client = socket.accept() ;
		    board.forkSocketThread(client) ;
		}
	    } catch (Exception ex) { 
		System.err.println("Error:" + ex.getMessage()) ;
		return ;
	    }
	} ;

    } ;

    //========================================
    // class Socket Thread

    class SocketThread extends Thread {

	public ServerBoard board ;

	public Side side ;

	public Socket socket ;

	public Reader reader ; 
	public Writer writer ; 

	public SocketThread() { init(null,null) ; } ;
	public SocketThread(ServerBoard b, Socket s) { init(b,s) ; } ;
	public void init(ServerBoard b, Socket s) {
	    board = b ; 
	    socket = s ;
	} ;

	public void run() {
	    try {
		reader = new InputStreamReader(socket.getInputStream()) ;
		writer = new OutputStreamWriter(socket.getOutputStream()) ;

		writer.write("(setSide " + side.toString() + ")") ;
		writer.flush() ;

		while(true) {
		    cycle() ;
		}
	    } catch (Exception ex) { 
		System.err.println("Error:" + ex.getMessage()) ;
		return ;
	    }
	} ;

	public void cycle() {
	    try {
		char buf[] = new char[1024] ;
		reader.read(buf) ;
		
		String str = new String(buf) ;

		Sexp s = Sexp.factory().scan(str) ;
		if(s.equals(Sexp.Eof)) {
		    throw new Exception("EOF is deteced.") ;
		}

		if(s.isCons() && s.car().equals("ok")) { return ; }
		if(s.isCons() && s.car().equals("error")) { return ; }

		boolean successp = board.execSexpCommand(s) ;

		Sexp ret ;
		if(successp) {
		    ret = Sexp.factory().cons("ok",s) ;
		} else {
		    ret = Sexp.factory().cons("error",s) ;
		}
		writer.write(ret.toString()) ;
		writer.flush() ;

	    } catch (Exception ex) { 
		System.err.println("Error:" + ex.getMessage()) ;
		return ;
	    }
	} ;
    } ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // socket

    public ServerThread serverThread ;
    public SocketThread socketThread[] ;
    public int socketN ;

    public SocketThread black ;
    public SocketThread white ;

    public boolean publicModep ;

    public static final int defaultSocketN = 32 ;

    //------------------------------------------------------------
    // constructor

    public ServerBoard() { init() ; } ;

    public void init() {
	super.init(Side.none) ;
	socketThread = new SocketThread[defaultSocketN] ;
	socketN = 0 ;
	black = null ;
	white = null ;

	publicModep = false ;

	try { 
	    FileOutputStream fstr = new FileOutputStream(defaultLogFile) ;
	    
	    //loggingOn(System.out) ;
	    loggingOn(new PrintStream(fstr)) ;
	} catch (Exception ex) {
	    System.err.println(ex.getMessage()) ;
	    return ;
	}
	
	runServerThread() ;
	    
    } ;

    //------------------------------------------------------------
    // start threads

    //----------------------------------------
    // server thread

    public void runServerThread() {
	serverThread = new ServerThread(this,defaultPort) ;
	serverThread.start() ;
    } ;

    //----------------------------------------
    // client thread
    
    public void forkSocketThread(Socket socket) {
	if(socketN >= defaultSocketN) {
	    return ;
	}
	SocketThread sock = new SocketThread(this,socket) ;

	socketThread[socketN] = sock ;
	socketN++ ;
	
	if(publicModep) {
	    sock.side = Side.none ;
	} else if (black == null) {
	    black = sock ;
	    sock.side = Side.black ;
	} else if (white == null) {
	    white = sock ;
	    sock.side = Side.white ;
	} else {
	    sock.side = Side.error ;
	} ;

	sock.start() ;
    } ;

    //------------------------------------------------------------
    // move 

    public boolean move(int fx, int fy, int tx, int ty, boolean checksidep) {
	boolean r = false ;
	r = super.move(fx,fy,tx,ty,checksidep) ;
	if(r) {
	    Sexp state = stateSexp() ;
	    String statestr = state.toString() ;

	    Side win = isGameOver() ;
	    boolean winp = (!win.isNone()) ;
	    String winstr = "";
	    if(winp) { 
		winstr = gameoverSexp(win).toString() ;
	    }

	    try {

		for(int i = 0 ; i < socketN ; i++) {

		    socketThread[i].writer.write(statestr) ;
		    socketThread[i].writer.flush() ;

		    if(winp) {
			socketThread[i].writer.write(winstr) ;
			socketThread[i].writer.flush() ;
		    }
		    
		} ;
	    } catch (Exception ex) { 
		System.err.println("Error:" + ex.getMessage()) ;
		return false;
	    }
	}
	return r ;
    } ;

    //------------------------------------------------------------
    // quit

    public void quit(boolean directp) {
	try {
	    Sexp quitSexp = Sexp.factory().list("quit") ;
	    String quitstr = quitSexp.toString() ;

	    for(int i = 0 ; i < socketN ; i++) {
		socketThread[i].writer.write(quitstr) ;
		socketThread[i].writer.flush() ;
	    } ;
	} catch (Exception ex) { 
	    System.err.println("Error:" + ex.getMessage()) ;
	}
	super.quit(directp) ;
    } ;

} ;

