// -*- Mode: java -*-
//Header:
//File: Sexp.java
//Author: NODA, Itsuki
//Date: 2001/12/24
//Copyright (C) 2001 by Itsuki NODA, AIST
//EndHeader:

import java.io.Reader ;
import java.io.StringReader ;
import java.io.StreamTokenizer ;
import java.io.* ;

//======================================================================

public class Sexp {
    
    //============================================================
    // static Factory

    static class Factory {

	//------------------------------------------------------------
	// return constant

	public Sexp Nil() { return Sexp.Nil ; } ;
	public Sexp Eof() { return Sexp.Eof ; } ;

	//------------------------------------------------------------
	// new data

	public Sexp newCell() { return new Sexp() ; } ;

	public Sexp newCell(Object v) { 
	    Sexp r = newCell() ;
	    r.setValue(v) ;
	    return r ;
	} ;

	public Sexp cell(Object v) {
	    if(v instanceof Sexp) return (Sexp)v ;
	    else return newCell(v) ;
	} ;

	public Sexp newCons(Sexp car, Sexp cdr) {
	    Cons c = new Cons(car,cdr) ;
	    return newCell(c) ;
	} ;

	//public Sexp cons(Sexp car, Sexp cdr) { return newCons(car,cdr) ; } ;

	public Sexp cons(Object car, Object cdr) {
	    return newCons(cell(car), cell(cdr)) ;
	} ;

	public Sexp newAtom(Object v) { return newCell(v) ; } ;

	public Sexp newInt(long ival) { return newInt(new Long(ival)) ; } ;

	public Sexp newInt(Long ival) { return newAtom(ival) ; } ;

	public Sexp newFlt(double fval) { return newFlt(new Double(fval)) ; } ;
	
	public Sexp newFlt(Double fval) { return newAtom(fval) ; } ;

	public Sexp newSymbol(String sym) { return newAtom(sym) ; } ;

	public Sexp newString(String str) { return newAtom(str) ; } ;

	//----------------------------------------
	// new list

	public Sexp list() { return Nil() ; } ;
	public Sexp list(Object s0) { return cons(s0,list()) ; } ;
	public Sexp list(Object s0, Object s1) { return cons(s0,list(s1)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2) { 
	    return cons(s0,list(s1,s2)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3) { 
	    return cons(s0,list(s1,s2,s3)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4) { 
	    return cons(s0,list(s1,s2,s3,s4)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4, Object s5){
	    return cons(s0,list(s1,s2,s3,s4,s5)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4, Object s5, Object s6){
	    return cons(s0,list(s1,s2,s3,s4,s5,s6)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4, Object s5, Object s6, Object s7){
	    return cons(s0,list(s1,s2,s3,s4,s5,s6,s7)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4, Object s5, Object s6, Object s7, 
			 Object s8){
	    return cons(s0,list(s1,s2,s3,s4,s5,s6,s7,s8)) ; } ;
	public Sexp list(Object s0, Object s1, Object s2, Object s3, 
			 Object s4, Object s5, Object s6, Object s7, 
			 Object s8, Object s9){
	    return cons(s0,list(s1,s2,s3,s4,s5,s6,s7,s8,s9)) ; } ;

	//------------------------------------------------------------
	// more list operation

	//------------------------------
	// reverse

	public Sexp reverse(Sexp list) {
	    Sexp r = Nil() ;
	    for(Sexp l = list ; l.isCons() ; l = l.cdr()) {
		r = cons(l.car(),l) ;
	    }
	    return r ;
	} ;

	//------------------------------------------------------------
	// scan

	//------------------------------
	// scan top entry

	public Sexp scanFile(String filename) {
	    try {
		return scan(new FileReader(filename)) ;
	    } catch (java.io.FileNotFoundException ex) {
		return Nil ; 
	    }
	} ;

	public Sexp scan(String str) {
	    return scan(new StringReader(str)) ;
	} ;

	public Sexp scan(InputStream stream) {
	    return scan(new InputStreamReader(stream)) ;
	} ;

	public Sexp scan(Reader reader) {
	    return scan(new StreamTokenizer(reader)) ;
	} ;

	public Sexp scan(StreamTokenizer tokenizer) {
	    //tokenizer.resetSyntax() ;
	    tokenizer.eolIsSignificant(false) ;
	    tokenizer.slashSlashComments(false) ;
	    tokenizer.slashStarComments(false) ;
	    tokenizer.commentChar(';') ;
	    tokenizer.quoteChar('"') ;
	    tokenizer.wordChars(':',':') ;
	    return scanTop(tokenizer) ;
	} ;

	//------------------------------
	// scan top

	public Sexp scanTop(StreamTokenizer tokenizer) {
	    try {
		int token = tokenizer.nextToken() ;
		switch(token) {
		case StreamTokenizer.TT_EOF : 
		    return Eof ;
		case StreamTokenizer.TT_EOL : 
		    return scanTop(tokenizer) ;
		case StreamTokenizer.TT_NUMBER : 
		    return newFlt(tokenizer.nval) ;
		case StreamTokenizer.TT_WORD : 
		    return newString(tokenizer.sval) ;
		case '(' : 
		    return scanCons(tokenizer) ;
		default : 
		    scanError(tokenizer,"illegal token in scanTop().") ;
		}
	    } catch (java.io.IOException ex) {
		scanError(tokenizer,"catch exception in scanTop().") ;
	    }
	    return Nil ;
	} ;

	//------------------------------
	// scan Cons

	public Sexp scanCons(StreamTokenizer tokenizer) {
	    Sexp car = scanTop(tokenizer) ;
	    Sexp cdr = scanCdr(tokenizer) ;
	    return cons(car,cdr) ;
	} ;

	public Sexp scanCdr(StreamTokenizer tokenizer) {
	    try {
		int token = tokenizer.nextToken() ;
		switch(token) {
		case ')' : return Nil ;
		case '.' : return scanTop(tokenizer) ;
		case StreamTokenizer.TT_EOL : 
		    return scanCdr(tokenizer) ;
		case StreamTokenizer.TT_EOF : 
		    scanError(tokenizer,"illegal EOF in scanCdr().")  ;
		    break ;
		default :
		    tokenizer.pushBack() ;
		    return scanCons(tokenizer) ;
		}
	    } catch (java.io.IOException ex) {
		scanError(tokenizer,"catch exception in scanCdr().") ;
	    }
	    return Nil ;
	} ;

	//------------------------------
	// scan error
    
	public void scanError(StreamTokenizer tokenizer,
			      String message) {
	    System.err.println("Error: scan: " + message) ;
	    System.err.println("\t at line " + tokenizer.lineno()) ;
	    System.err.println("\t token=" + tokenizer.toString()) ;

	    System.exit(1) ;
	} ;

    } ;

    static final Factory factory_ = new Factory() ;
    
    public Factory myFactory() { 
	//return (Factory)getClass().getField("factory").get(this) ; 
	return Sexp.factory() ;
    } ;

    public static Factory factory() { return factory_ ; } ;

    //============================================================
    // class NIL

    static class NilClass {
	public boolean equals(NilClass nil) {  return true ; } ;
    } ;

    //============================================================
    // class Cons

    static class Cons {
	Sexp car_ ;
	Sexp cdr_ ;

	public Sexp car() { return car_ ; } ;
	public Sexp cdr() { return cdr_ ; } ;

	public Sexp setCar(Sexp v) { return car_ = v ; } ;
	public Sexp setCdr(Sexp v) { return cdr_ = v ; } ;

	public Sexp rplaca(Sexp v) { return setCar(v) ; } ;
	public Sexp rplacd(Sexp v) { return setCdr(v) ; } ;

	public Cons() { 
	    car_ = cdr_ = Nil ;
	} ;

	public Cons(Sexp carval, Sexp cdrval) {
	    car_ = carval ; 
	    cdr_ = cdrval ;
	} ;

	public boolean equals(Cons cons) {
	    return car().equals(cons.car()) && cdr().equals(cons.cdr()) ;
	} ;

    } ;

    //============================================================
    // body

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // final static variable

    final static Sexp Nil = new Sexp(new NilClass()) ;
    final static Sexp Eof = new Sexp("_eof_") ;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // variable

    Object value_ ;

    public Object value() { return value_ ; } ;
    public Object setValue(Object v) { return value_ = v ; } ;

    //------------------------------------------------------------
    // constructor

    Sexp() { 
	setValue(Nil.value()) ;
    } ;

    Sexp(Object v) {
	setValue(v) ;
    } ;

    //------------------------------------------------------------
    // type check

    public boolean isNil() { return value() instanceof NilClass ; } ;

    public boolean isCons() { return value() instanceof Cons ; } ;

    public boolean isAtom() { return !isCons() ; } ;

    public boolean isInt() { return value() instanceof Long ; } ;

    public boolean isFlt() { return value() instanceof Double ; } ;

    public boolean isNumber() { return value() instanceof Number ; } ;

    public boolean isSymbol() { return value() instanceof String ; } ;

    //------------------------------------------------------------
    // get list item

    //----------------------------------------
    // car / cdr

    public Sexp car() { 
	if(!isCons()) return Nil ;
	else          return ((Cons)value()).car() ; 
    } ;
    public Sexp cdr() { 
	if(!isCons()) return Nil ;
	return        ((Cons)value()).cdr() ; 
    } ;

    public Sexp rplaca(Sexp s) {
	if(!isCons()) return Nil ;
	else          return ((Cons)value()).rplaca(s) ;
    } ;

    public Sexp rplacd(Sexp s) {
	if(!isCons()) return Nil ;
	else          return ((Cons)value()).rplacd(s) ;
    } ;

    //----------------------------------------
    // nth

    public Sexp nthcdr(long n) {
	Sexp s = this ;
	for(long i = 0 ; i < n ; i++) {
	    if(!s.isCons()) return Nil ;
	    s = s.cdr() ;
	}
	return s ;
    } ;

    public Sexp nth(long n) {
	return nthcdr(n).car() ;
    }

    public Sexp first() { return car() ; } ;
    public Sexp rest() { return cdr() ; } ;

    public Sexp last() { 
	if(!isCons()) return this ;
	Sexp s = this ;
	for( ; s.cdr().isCons() ; s = s.cdr()) ;
	return s ;
    } ;

    //------------------------------------------------------------
    // comparison

    public boolean equals(Sexp sexp) {
	return value().equals(sexp.value()) ;
    } ;

    public boolean equals(String symbol) {
	return isSymbol() && value().equals(symbol) ;
    } ;

    public boolean equals(long ival) {
	return isNumber() && intVal() == ival ;
    } ;

    public boolean equals(Long ival) {
	return isNumber() && ival.longValue() == intVal() ;
    } ;

    public boolean equals(double fval) {
	return isNumber() && fltVal() == fval ;
    } ;

    public boolean equals(Double fval) {
	return isNumber() && fval.doubleValue() == fltVal() ;
    } ;

    //------------------------------------------------------------
    // basic list operation

    //----------------------------------------
    // length

    public long length() {
	long l = 0 ;
	for(Sexp s = this ; s.isCons() ; s = s.cdr()) {
	    l++ ;
	}
	return l ;
    } ;

    //----------------------------------------
    // append
    
    public static Sexp append(Sexp l0, Sexp l1) {
	return l0.append(l1) ;
    } ;

    public Sexp append(Sexp tail) {
	if(!isCons()) return tail ;
	else return myFactory().cons(car(),cdr().append(tail)) ;
    } ;

    public static Sexp concat(Sexp l0, Sexp l1) {
	Sexp t = l0.last() ;
	return t.rplacd(l1) ;
    } ;

    public Sexp concat(Sexp tail) {
	return Sexp.concat(this,tail) ;
    } ;

    //----------------------------------------
    // member

    public Sexp member(Sexp elm) {
	if      (!isCons()) { return Nil ; } 
	else if (car().equals(elm)) { return this ; } 
	else { return cdr().member(elm) ; } 
    } ;

    //----------------------------------------
    // addLast

    public Sexp addLast(Sexp item) {
	return concat(myFactory().list(item)) ;
    } ;

    //----------------------------------------
    // push/pop

    public Sexp push(Sexp item) {
	return myFactory().cons(item,this) ;
    } ;

    public Sexp pop() { return cdr() ; } ;

    //----------------------------------------
    // reverse

    public Sexp reverse() {
	return myFactory().reverse(this) ;
    } ;

    //------------------------------------------------------------
    // basic table operation

    //----------------------------------------
    // assoc

    public Sexp assoc(Sexp label) {
	if(label.isSymbol()) return assoc((String)(label.value())) ;
	else return Nil ; 
    }

    public Sexp assoc(String label) {
	if       (!isCons()) { return Nil ; } 
	else if (!car().isCons()) { return cdr().assoc(label) ; } 
	else if (car().car().equals(label)) { return car() ; } 
	else { return cdr().assoc(label) ; } 
    } ;

    //----------------------------------------
    // assocValue

    public Sexp assocValue(Sexp label) {
	return assoc(label).cdr() ;
    } ;

    public Sexp assocValue(String label) {
	return assoc(label).cdr() ;
    } ;

    //----------------------------------------
    // passoc

    public Sexp passoc(Sexp label) {
	if(label.isSymbol()) return passoc((String)(label.value())) ;
	return Nil ; 
    } ;

    public Sexp passoc(String label) {
       Sexp pos = passocPos(label) ;

       if(pos.isCons()) {
	  if(pos.cdr().isCons()) {
	     return pos.cdr().car() ;
	  } else {
	     return Nil ;
	  }
       }
       else { return Nil ; }
    } ;

    //----------------------------------------
    // passocPos

    public Sexp passocPos(Sexp label) {
	if(label.isSymbol()) return passoc((String)(label.value())) ;
	return Nil ;
    } ;

    public Sexp passocPos(String label) {
       if       (!isCons()) { return Nil ; } 
       else if (car().equals(label)) { return this ; }
       else if (cdr().isCons()) { return cdr().cdr().passocPos(label) ; } 
       else { return Nil ; }
    } ;

    //------------------------------------------------------------
    // convert

    //----------------------------------------
    // int/flt/sym Values

    public long intVal() {
	if (isInt()) { return ((Long)value()).longValue() ; }
	else if (isFlt()) { return (long)(((Double)value()).doubleValue()) ; }
	else { return 0 ; } 
    } ;

    public double fltVal() {
	if (isFlt()) { return ((Double)value()).doubleValue() ; } 
	else if (isInt()) { return (double)(((Long)value()).longValue()) ; } 
	else { return 0.0 ; }
    } ;

    public String symVal() {
	if (isSymbol()) return (String)value() ;
	else { return null ; } ;
    } ;

    public String strVal() {
	if (isSymbol()) return (String)value() ;
	else { return null ; } ;
    } ;

    //----------------------------------------
    // toString

    public String toString() {
	if(isNil()) {
	    return "()" ;
	}
	if(isAtom()) {
	    return value().toString() ;
	} else {
	    return toStringCons() ;
	}
    } ;

    public String toStringCons() {
	String r = "(" ;
	Sexp s = this ; 
	for(; s.isCons() ; s = s.cdr()) {
	    if(s != this) { r = r + " " ; }

	    r = r + s.car().toString() ;
	}
	if(!s.isNil()) { r = r + " . " + s.toString() ;}
	r = r + ")" ;

	return r ;
    } ;

    //============================================================
    // test

    //------------------------------
    // test 1

    public static void test1() {
	Sexp.Factory f = Sexp.factory() ;

	Cons c = new Cons() ;

	Sexp s1 = new Sexp() ;
	System.out.println(s1) ;

	Sexp s2 = Sexp.Nil ;
	System.out.println(s2) ;

	Sexp s3 = f.newInt(3) ;
	System.out.println(s3) ;

	Sexp s4 = f.newFlt(3.14) ;
	System.out.println(s4) ;

	Sexp s5 = f.cons(s3,s4) ;
	System.out.println(s5) ;

	Sexp s6 = f.newString("aho") ;

	Sexp s7 = f.list(s1,s2,s3,s4,s5,s6) ;
	System.out.println(s7) ;

	String a = "abc" ;
	String b = new String("abc") ;

	System.out.println(a + b + ":ans:" + (a.equals(b))) ;
    } ;
 
    //------------------------------
    // test 2

    public static void test2() {
	//	String str = "(this is a pen \n\t\t\t   1 2 3 4 55.234 1.2.3 \"aho\")" ;
	String str = "( aabc(a c d) :abc 1 2.0 . 2.4.5 2-1-3 \n \t \t ; aho \n def )" ;
	System.out.println(str) ;
	System.out.println(Sexp.factory().scan(str)) ;
    } ;

    //------------------------------
    // test 3

    public static void test3() {
	String str = "aabc (a c d) (:abc)" ;
	System.out.println(str) ;
	Reader reader = new StringReader(str) ;
	Sexp s ;
	while((s = Sexp.factory().scan(reader)) != Sexp.Eof) {
	    System.out.println("scan=" + s) ;
	} ;
    } ;

    //------------------------------
    // test 4

    public static void test4sub1(Object x) {
	System.out.println(x) ;
    } ;

    public static void test4() {
	test4sub1("aho") ;
	test4sub1(new Long(3)) ;
	//test4sub1(3) ;
	//test4sub1(null) ;
    } ;

    //------------------------------------------------------------
    // main for test

    public static void main(String argv[]) {
	//test1() ;
	test2() ;
	test3() ;
    } ;
    
} ;
    
