<?php
/* ========================================================================
 - [libs/rkt_calendar.php]
 -      カレンダー情報
 -      Copyright (c) 2003-2006 Yujiro Takahashi
 - ライセンス:
 -      This source file is subject to version 3.0 of the PHP license,
 -      that is available at http://www.php.net/license/3_0.txt
 -      If you did not receive a copy of the PHP license and are unable 
 -      to obtain it through the world-wide-web, please send a note to 
 -      license@php.net so we can mail you a copy immediately.  
 - 問い合わせ先：
 -      yujiro@rakuto.net
 -      http://rakuto.net/
 - 更新履歴：
 -      [2005/09/19] タイムスタンプの取得を追加
 -      [2005/07/11] splitDate()の不具合修正
 -      [2003/10/16] PEAR Codingに修正
 -      [2003/10/07] メンバー変数$data を$datasに変更 エラー解釈あり
 -      [2003/06/25] 作成
 - ======================================================================== */

/**
 * 年月日の格納番号
 *      $date[RKT_YEAR]
 *      $date[RKT_MONTH]
 *      $date[RKT_DAY]
 */
/**
 * 年月日の格納番号
 *      $date[RKT_YEAR]
 *      $date[RKT_MONTH]
 *      $date[RKT_DAY]
 * @const RKT_YEAR 年
 * @const RKT_MONTH 月
 * @const RKT_DAY 日
 */
define('RKT_YEAR',  0);
define('RKT_MONTH', 1);
define('RKT_DAY',   2);

/**
 * 曜日の格納番号
 *      $calendar[$week][RKT_SUN]
 * @const RKT_SUN 日
 * @const RKT_MON 月
 * @const RKT_TUE 火
 * @const RKT_WED 水
 * @const RKT_THU 木
 * @const RKT_FRI 金
 * @const RKT_SAT 土
 */
define('RKT_SUN', 0);
define('RKT_MON', 1);
define('RKT_TUE', 2);
define('RKT_WED', 3);
define('RKT_THU', 4);
define('RKT_FRI', 5);
define('RKT_SAT', 6);

/**
 * 平日・休日／祝日・土曜日フラグ
 *      $info[$week][RKT_MON] = RKT_HOLIDAY;
 * @const RKT_WEEKDAY 月～金曜
 * @const RKT_HOLIDAY 日曜祝日
 * @const RKT_SATURDAY 土曜日
 * @const RKT_TODAY 今日
 */
define('RKT_WEEKDAY',  0);
define('RKT_HOLIDAY',  1);
define('RKT_SATURDAY', 2);
define('RKT_TODAY',    3);

/**
 * カレンダー生成クラス
 *
 * @author 高橋 裕志郎 <yujiro@rakuto.net>
 * @package RKT_calendar
 * @access public
 * @version 1.1
 */
class RKT_calendar
{
    /**
     * 日付
     * @var array
     */
    var $dates = array();

    /**
     * タイムスタンプ
     * @var long
     */
    var $timestamp = 0;

    /**
     * 一日の曜日
     * @var integer
     */
    var $start = 0;

    /**
     * 月の最終日
     * @var integer
     */
    var $last  = 0;

    /**
     * 週の数
     * @var integer
     */
    var $num_weeks = 0;

    /**
     * カレンダー
     * @var array
     */
    var $calendar = array();

    /**
     * カレンダー情報
     * @var array
     */
    var $info = array();

    /**
     * 日付の指定
     *
     * @access public
     * @param string $date 取得対象の日付
     *        フォーマット:"year/month/day"
     *        例)          "2003/6/20"
     */
    function RKT_calendar($date='')
    {
        $this->timestamp = 0;
        $this->setDate($date);
    }

    /**
     * 値の初期化
     *
     * @access private
     * @param long $timestamp タイムスタンプ
     */
    function init_value()
    {
        $date = date('Y-n-j', $this->timestamp);
        $this->dates = explode('-', $date);
        $timestamp = mktime(0,0,0,$this->dates[RKT_MONTH],1,$this->dates[RKT_YEAR]);

        $this->start = intval(date('w',$timestamp));
        $this->last  = intval(date('t',$timestamp));
        
        $this->num_weeks = intval((($this->start + $this->last) / 7) + 0.9);
    }

    /**
     * 日付の指定
     *
     * @access public
     * @param string $date 取得対象の日付
     *        フォーマット:"year-month-day"
     *        例)          "2003/6/20"
     */
    function setDate($date='')
    {
        $date = trim($date,"'");
        if (preg_match("/\d{4}-\d{1,2}-\d{1,2}/", $date)){
            $dates = explode('-', $date);
            $this->timestamp = mktime(0,0,0,$dates[RKT_MONTH],$dates[RKT_DAY],$dates[RKT_YEAR]);
        } else {
            $this->timestamp = empty($date)?time():strtotime($date);
        }
        $this->init_value();
    }

    /**
     * 現在の月に値加えてカレンダーの日付を設定
     *
     * @access public
     * @param integer $val 月に加える値
     */
    function addMonth($val)
    {
        $month = intval($this->dates[RKT_MONTH]) + $val;
        
        $this->timestamp = mktime(0,0,0,$month,$this->dates[RKT_DAY],$this->dates[RKT_YEAR]);
        $this->init_value();
    }

    /**
     * 現在の日に値加えてカレンダーの日付を設定
     *
     * @access public
     * @param integer $val 日に加える値
     */
    function addDay($val)
    {
        $day = intval($this->dates[RKT_DAY]) + $val;

        $this->timestamp = mktime(0,0,0,$this->dates[RKT_MONTH],$day,$this->dates[RKT_YEAR]);
        $this->init_value();
    }

    /**
     * 指定月の日にちを2次元配列に格納しカレンダーにする
     *
     * @access public
     * @return array 2次元配列のカレンダー
     */
    function getCalendar()
    {
        $day = 1;
        $linecount = 0;
        $start = $this->start;
        
        for ($row=0; $row<$this->num_weeks; $row++){
            $this->calendar[$row] = array('','','','','','','');
            $this->info[$row] = array(
                            RKT_HOLIDAY,RKT_WEEKDAY,RKT_WEEKDAY,
                            RKT_WEEKDAY,RKT_WEEKDAY,RKT_WEEKDAY,
                            RKT_SATURDAY);
            for ($col=$start; $col<7 && $day<=$this->last; $col++){
                $this->calendar[$row][$col] = $day++;
            }
            $start = 0;
        }

        return ($this->calendar);
    }

    /**
     * 2次元配列のカレンダーでの今日の位置を取得
     *
     * @access public
     * @return array 今日の位置
     */
    function getTodayPos()
    {
        $pos = $this->start + $this->dates[RKT_DAY] - 1;
        $week = intval($pos/7);
        $yobi = $pos - ($week*7);
        
        return (array('week'=>$week,'yobi'=>$yobi));
    }

    /**
     * 春分／秋分の日を取得する
     *
     * @access private
     */
    function set_equinox(&$h_days)
    {
        $year = $this->dates[RKT_YEAR];
        $month = $this->dates[RKT_MONTH];
        if ($month != 3 && $month != 9){
            return;
        }
        $difference = intval(($year - 1980)/4);
        $const = array(
            3=>20.8431,
            9=>23.2488
        );

        $equinox = (int)($const[$month] + 0.242194*($year - 1980) - $difference);
        array_push($h_days[$month],$equinox);
    }
        
    /**
     * 指定月の休日／祝日などを2次元配列に格納する
     *
     * @access public
     * @return array 休日／祝日などの情報
     */
    function getInformation()
    {
        $month = $this->dates[RKT_MONTH];
        $h_days = array(
             1=> array(1),        // 元旦
             2=> array(11),        // 建国記念日
             3=> array(),
             4=> array(29),        // みどりの日
             5=> array(3,4,5),    // 憲法記念日,こどもの日
             6=> array(),
             7=> array(),
             8=> array(),
             9=> array(),
            10=> array(),
            11=> array(3,23),    // 文化の日,勤労感謝の日
            12=> array(23)        // 天皇誕生日
        );
        $this->set_equinox($h_days);
        
        // 固定祝日
        foreach ($h_days[$month] as $h_day){
            $pos = $this->start + ($h_day -1);
            $week = intval($pos/7);
            $yobi = $pos - ($week*7);
            $this->info[$week][($yobi ? $yobi:RKT_MON)] = RKT_HOLIDAY;
        }
        
        // ハッピーマンデー
        $mdct = ($this->start > 1)? 1:0;       // 最初の月曜日が第1週にあるか
        switch ($month){
        case 1:
            $mdct += 1;                        // 第2月曜 成人の日
            $this->info[$mdct][RKT_MON] = RKT_HOLIDAY;
            break;
        case 7:
            $mdct += 2;                        // 第3月曜 海の日
            $this->info[$mdct][RKT_MON] = RKT_HOLIDAY;
            break;
        case 9:
            $mdct += 2;                        // 第3月曜 敬老の日
            $this->info[$mdct][RKT_MON] = RKT_HOLIDAY;
            break;
        case 10:
            $mdct += 1;                        // 第2月曜 体育の日
            $this->info[$mdct][RKT_MON] = RKT_HOLIDAY;
            break;
        default:
            break;
        }

        return ($this->info);
    }

    /**
     * 指定された週の1週間分を取得する
     *
     * @access public
     * @param integer $weekly 対象となる週
     * @param string $format  date()関数のフォーマット
     *    注意：第一週は0
     * @return array 選択された週の日付
     *         $weeks[RKT_SUN] 日曜日の日付
     *         $weeks[RKT_MON] 月曜日の日付
     *         $weeks[RKT_TUE] 火曜日の日付
     *         $weeks[RKT_WED] 水曜日の日付
     *         $weeks[RKT_THU] 木曜日の日付
     *         $weeks[RKT_FRI] 金曜日の日付
     *         $weeks[RKT_SAT] 土曜日の日付
     */
    function getWeeks($weekly,$format='Y/m/d')
    {
        $weeks = array();
        $days = array(1,8,15,22,29,36);
        $timestamp = mktime(0,0,0,$this->dates[RKT_MONTH],$days[$weekly],$this->dates[RKT_YEAR]);
        $dis = date('w',$timestamp);

        for ($week=0;$week<7;$week++){
            $day = $days[$weekly] - $dis + $week;
            $timestamp = mktime(0,0,0,$this->dates[RKT_MONTH],$day,$this->dates[RKT_YEAR]);

            $weeks[$week] = date($format,$timestamp);
        }

        return ($weeks);
    }

    /**
     * 指定されている年月日が何週目に当たるかを取得
     *
     * @access public
     *    注意：第一週は0
     * @return integer 何週目
     */
    function getWeek()
    {
        $weeks = array();
        $days = array(1,8,15,22,29,36);
        $timestamp = mktime(0,0,0,$this->dates[RKT_MONTH],$this->dates[RKT_DAY],$this->dates[RKT_YEAR]);
        $day = intval($this->dates[RKT_DAY]);
        $day_w = intval(date('w',$timestamp));
        $day -= $day_w;

        foreach ($days as $week=> $master){
            if ($day <= $master){
                break;
            }
        }

        return ($week);
    }

    /**
     * 指定されている年月日を取得する
     *
     * @access public
     * @return array 年月日配列
     *         $date[RKT_YEAR]  年
     *         $date[RKT_MONTH] 月
     *         $date[RKT_DAY]   日
     */
    function getDates()
    {
        return ($this->dates);
    }

    /**
     * 指定されているタイムスタンプを取得する
     *
     * @access public
     * @return integer タイムスタンプ
     */
    function getTimestamp()
    {
        return ($this->timestamp);
    }

    /**
     * 指定されている月の日数を取得する
     *
     * @access public
     * @return integer 月の日数
     */
    function getLastday()
    {
        return $this->last;
    }

    /**
     * 指定されている月の週の数を取得
     *
     * @access public
     * @return integer 週の数を取得
     */
    function getNumberWeeks()
    {
        return $this->num_weeks;
    }

    /**
     * 日にちから曜日を取得
     *
     * @access public
     * @param integer $day 日にち
     * @return integer 曜日
     */
    function getDayToWeek($day)
    {
        return (int)(($this->start + $day)%7) - 1;
    }

    /**
     * 曜日と週目から日にちを取得
     *
     * @access public
     * @param integer $week 曜日
     * @param integer $num_weeks 何週目か
     *    注意：第一週は0
     * @return integer 日にち
     */
    function getWeekToDay($week,$num_weeks)
    {
        return ($num_weeks*7) - $this->start + $week +1;
    }

    /**
     * 配列の日付データを文字列に
     *
     * @access public
     * @parmas array $dates 日付データ
     * @return string
     */
    function joinDate($dates)
    {
        if (!is_array($dates)){
            return $dates;
        }
        $result  = empty($dates['year'])?0:$dates['year'];
        $result .= '-';
        $result .= empty($dates['month'])?0:$dates['month'];
        $result .= '-';
        $result .= empty($dates['day'])?0:$dates['day'];

        $needle = array('hour','minute','seconds');
        if (isSet($dates['hour']) && isSet($dates['minute']) && isSet($dates['seconds'])){
            $result .= ' ';
            $result .= empty($dates['hour'])?0:$dates['hour'];
            $result .= ':';
            $result .= empty($dates['minute'])?0:$dates['minute'];
            $result .= ':';
            $result .= empty($dates['seconds'])?0:$dates['seconds'];
        }

        if (ereg('0-0-0',$result)){
            return null;
        }
        if (ereg('0-0-0 0:0:0',$result)){
            return null;
        }
        
        return $result;
    }

    /**
     * 文字列の日付データを配列に
     *
     * @access public
     * @parmas string $date 日付データ
     * @return array
     */
    function splitDate($date)
    {
        if (!is_string($date)){
            return $date;
        }

        if (preg_match("/[\/: -]/i", $date)){
            $dates = split ('[/: -]', $date);
        } else {
            $dates = array (
                substr($date, 0, 4),
                substr($date, 4, 2),
                substr($date, 6, 2),
                substr($date, 8, 2),
                substr($date, 10, 2),
                substr($date, 12, 2),
            );
        }

        $result = array(
            'year'=>$dates[0],
            'month'=>$dates[1],
            'day'=>$dates[2],
        );
        
        if (!empty($dates[3])){
            $result['hour'] = $dates[3];
            $result['minute'] = $dates[4];
            $result['seconds'] = $dates[5];
        }
        return $result;
    }

    /**
     * w3cdtf形式の日付を返す
     *
     * @access public
     * @param integer $timestamp タイムスタンプ
     * @return string w3cdtf形式の日付
     */
    function w3cdtf($timestamp)
    {
        if (!is_int($timestamp)){
            $dates = RKT_calendar::splitDate($timestamp);
            $timestamp = mktime(
                            $dates['hour'],
                            $dates['minute'],
                            $dates['seconds'],
                            $dates['month'],
                            $dates['day'],
                            $dates['year']);
        }
        
        $w3cdtf = date('Y-m-d\TH:i:s' , $timestamp);
        $delay  = date('O' , $timestamp);
        $w3cdtf .= substr($delay, 0, 3);
        $w3cdtf .= ':';
        $w3cdtf .= substr($delay, 3, 2);

        return $w3cdtf;
    }
}
// RKT_calendarの終了
?>