/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.rpn_computer;

import jp.sourceforge.tokenizer.Token;
import jp.sourceforge.tokenizer.TokenInfo;

/**
 * <p>
 * 構文解析に失敗したことを現す例外です。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class ParseException extends RuntimeException {

    private Token       token;

    private TokenInfo[] tokenInfos;

    /**
     * <p>
     * インスタンスを初期化します。期待したトークンとは違うトークンが出現したことを表すメッセージを構築します。
     * </p>
     * 
     * @param token
     *            出現したトークン。
     * @param tokenInfos
     *            期待したトークンの情報の配列。
     */
    public ParseException(Token token, TokenInfo[] tokenInfos) {
        super(ParseException.buildMessage(token, tokenInfos));

        this.token = token;
        this.tokenInfos = tokenInfos;
    }

    private static String buildMessage(Token token, TokenInfo[] tokenInfos) {
        String msg;
        if (token != null) {
            msg = token.getLine() + "行目、" + token.getColumn() + "列目でトークン\"" + token.getToken() + "\"が出現しました。正しくは、次のパターンのトークンを期待しています。{";
        } else {
            msg = "期待したトークンが出現しませんでした。次のパターンのトークンを期待しています。{";
        }
        for (int i = 0; i < tokenInfos.length; i++) {
            if (i != 0) {
                msg += ", ";
            }
            msg += tokenInfos[i].getTokenPattern().pattern();
        }
        msg += "}";
        return msg;
    }

    /**
     * <p>
     * 出現したトークンを返します。
     * </p>
     * 
     * @return 出現したトークン。
     */
    public Token getToken() {
        return this.token;
    }

    /**
     * <p>
     * 期待したトークンの配列を返します。
     * </p>
     * 
     * @return 期待したトークンの配列。
     */
    public TokenInfo[] getTokenInfos() {
        return this.tokenInfos;
    }
}
