/**
 * @file nes_posix_pthread.c
 * @define implement thread core operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <basic.h>
#include <tkse/errno.h>
#include <tkse/extension/proctask.h>

#include "nes_posix_pthread.h"
#include "common.h"
#include "utility/hashtable_itr.h"
#include "manager.h"
#include "clean.h"

extern int __g_finish_now;		/* init_lib.c */
extern struct hashtable* __g_threadhash;	/* manager.c */	

/**********************************************************************/
/* Function name: nes_posix_pthread_create                                      */
/* Description: Create a thread with given attributes ATTR(or default */
/*		attributes if ATTR is NULL), and call function FP with*/
/*		given arguments ARG.                                  */
/*                                                                    */
/* Return type: On success, the identifier of the newly created thread*/
/*              is stored in the location pointed by the thread       */
/*              argument, and a 0 is returned. On  error,  a non-zero */
/*              error code is returned.                               */
/*     EINVAL : Param invalid                                         */
/*     ENOME : not  enough memory                                     */
/*     ERANGE : not  enough system resources                          */
/*                                                                    */
/* Argument - pthread_t * thread:  thread handle if create ok         */
/* Argument - const pthread_attr_t * attr:  thread init attribute     */
/* Argument - FP start_routine:  the thread excute function           */
/* Argument - void * arg:  thread excute function's param             */
/***********************************************************************/
int nes_posix_pthread_create(pthread_t * thread, const pthread_attr_t* attr, void (*start_routine)(void*), void * arg)
{
	W wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	int priority = PRIORITY_NORMAL;
	P_STATE PS;		/* Use for get main thread priority */

	// Check param
	if ( !thread || !start_routine ) 
	{
		nRet = EINVAL;
		goto exit;
	}

	/* Check thread attr */
	if( attr ) 
	{
		if( attr->__detachstate != PTHREAD_CREATE_JOINABLE && 
		    attr->__detachstate != PTHREAD_CREATE_DETACHED ) 
		{
			nRet = EINVAL;
			goto exit;
		}

		if( attr->__schedpolicy != SCHED_RR && attr->__schedpolicy != SCHED_FIFO ) 
		{
			nRet = EINVAL;
			goto exit;
		}

		priority = attr->__schedparam.sched_priority;
	}

	/* Default priority is Process's priority */
	if( priority == PRIORITY_NORMAL && tkse_prc_sts(0, &PS, NULL) > 0 ) 
	{
		priority = PS.priority;
	}

	/* Create thread descriptor struct */
	__pthread_desr* desr = (__pthread_desr*)malloc(sizeof(__pthread_desr));
	if( !desr ) 
	{
		nRet = ENOMEM;
		goto exit;
	}

	/*
	 * If __pthread_desr_init failed, most case is internal lock count is limited
	 */
	if( (nRet = __pthread_desr_init(desr, attr, 0)) != 0 ) 
	{
		free(desr);
		nRet = ERANGE;
		goto exit;
	}

	wCall = tkse_crs_tsk((FP)start_routine, priority, (W)arg); /* use tkse_crs_tsk, not tkse_cre_tsk */

	/* Create successfully */
	if (wCall > 0)
	{
		*thread = wCall;
		desr->__attr.__schedparam.sched_priority = priority;	/* Save current real priority */
		
		GLOBAL_THREADMGR_LOCK();
		__pthread_add_desr(wCall, desr);
		GLOBAL_THREADMGR_UNLOCK();
		nRet = 0;

		goto exit;
	}

	/* Error code mapping */
	switch ( wCall )
	{
	case E_MACV:
	case E_PAR:		/* Out priority range */
		nRet = EINVAL;
		break;
	case E_LIMIT:
		nRet = ERANGE;
		break;
	case E_NOMEM:
		nRet = ENOMEM;
		break;
	default:
		nRet = EINVAL;
		break;
	}

	/* Release memory if create failed */
	__pthread_desr_destroy(desr);
	free(desr);
exit:
	return nRet;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_equal                                       */
/* Description: Compare two thread identifiers                        */
/* Return type: return non zero value if equal, otherwise return 0    */
/* Argument - pthread_t thread1:  first thread                        */
/* Argument - pthread_t thread2:  second thread                       */
/**********************************************************************/
int nes_posix_pthread_equal(pthread_t thread1, pthread_t thread2)
{
	return (thread1 == thread2) ? 1 : 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_exit                                        */
/* Description: Terminate calling thread.                             */
/* Return type: void                                                  */
/* Argument - void *retval: to get the thread's return value at final */
/**********************************************************************/
void nes_posix_pthread_exit(void *retval)
{
	int thread_status = 0;
	__pthread_desr* desr = NULL;
	pthread_t self = nes_posix_pthread_self();

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(self);

	if( !desr )	/* Invalid thread id */
	{
		GLOBAL_THREADMGR_UNLOCK();
		return;
	}

	/* Destrory TSD */
	__pthread_destory_all_tsd(desr);

	/* Do cleanup */
	__pthread_destory_all_cleanup(desr);	

	/* Set exit code */
	desr->__exitcode = (int)retval;

	/* Get thread type */
	thread_status = desr->__status;

	/* Notify the waited thread for join */
	if( desr->__attr.__detachstate == PTHREAD_CREATE_JOINABLE ) 
	{
		__internal_unlock(&(desr->__lock));	

	} 
	else 
	{	/* Destroy pthread descriptor now */
		__pthread_del_desr(self);
		__pthread_desr_destroy(desr);
		free(desr);
	}

	GLOBAL_THREADMGR_UNLOCK();

	/* 
	 * If caller is 'Main' thread, here maybe 2 case
	 * 1. User Apllication use 'return' to exit, so pthread_exit is not be called directly by user,
	 *     And PTHREAD_FINISH() will auto excute pthread_exit(main), and at that time, now permit 
	 *    pthread_exit do PTHREAD_FINISH() again.
	 * 2. User Apllication use pthread_exit, now __g_finish_now == 0, so must do clean works and exit.
	 */
	if( thread_status & STATUS_THREAD_MAIN_THREAD ) /* Main thread call */
	{			
		if( !__g_finish_now ) /* Not excute at PTHREAD_FINISH() */
		{		
			PTHREAD_FINISH();
			tkse_ext_tsk();
		}
	} 
	else /* If child thread, exit directly */
	{					
		/* Real exit */
		tkse_ext_tsk();
	}
}

/**********************************************************************/
/* Function name: nes_posix_pthread_self                                        */
/* Description: Obtain the identifier of the current thread.          */
/* Return type:  Self thread id                                       */
/* Argument - void:                                                   */
/**********************************************************************/
pthread_t nes_posix_pthread_self(void)
{
	/*
	 * Here we can get nothing except threadID 
	 */
	return tkse_get_tid();
}

/**********************************************************************/
/* Function name: nes_posix_pthread_join                                        */
/* Description:Make calling thread wait for termination of the thread.*/
/*           The exit status of the thread is stored in *THREAD_RETURN*/
/*	     if THREAD_RETURN is not NULL.                            */
/*                                                                    */
/* Return value:			                              */
/*	ESRCH  No thread could be found                               */
/*	EINVAL The th thread has been detached.                       */
/*	EINVAL Another thread is already waiting on termination of th.*/
/*	EDEADLK  The th argument refers to the calling thread.	      */
/*                                                                    */
/* Argument - pthread_t thread:                                       */
/* Argument - void** thread_return:                                   */
/**********************************************************************/
int nes_posix_pthread_join(pthread_t thread, void** thread_return)
{
	int nRet = 0;		// Linux type return value
	__internal_lock_t t_lock;
	__pthread_desr* desr = NULL;

	/* Invalid param */
	if( thread <= 0  ) 
	{
		nRet = ESRCH;
		goto exit;
	}

	/* Join self must be error */
	if( nes_posix_pthread_equal(nes_posix_pthread_self(), thread) ) 
	{
		nRet = EDEADLK;
		goto exit;
	}	

	/*
	 *  Get thread desrc and check params.
	 */
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(thread);
	if( !desr )	/* Invalid thread id */
	{
		nRet = ESRCH;
		GLOBAL_THREADMGR_UNLOCK();
		goto exit;
	}
	
	if( desr->__attr.__detachstate != PTHREAD_CREATE_JOINABLE )	/* Thread run as detach mode*/
	{
		nRet = EINVAL;
		GLOBAL_THREADMGR_UNLOCK();
		goto exit;
	}

	if( desr->__status & STATUS_THREAD_JOIN_NOW ) /* Someone thread is pthread_join(th) now */
	{	
		nRet = EINVAL;
		GLOBAL_THREADMGR_UNLOCK();
		goto exit;
	}

	desr->__status |= STATUS_THREAD_JOIN_NOW;	/* Set Join now flag */
	t_lock = desr->__lock;
	GLOBAL_THREADMGR_UNLOCK();

	 /* 
	  * Begin to real join 
	  * Here the t_lock just used as semaphore, so need unlock it after lock ok.
	  */
	nRet = __internal_lock(&t_lock);			
	nRet = 	__internal_unlock(&t_lock);

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_del_desr_2(thread);
	GLOBAL_THREADMGR_UNLOCK();
	if( !desr )		/* Thread is not exist more */
	{
		nRet = ESRCH;
		goto exit;
	}
	
	nRet = 0;
	if( thread_return ) /* Get return value */	
	{
		*thread_return = (desr->__status & STATUS_THREAD_CANCEL_ALREADY) ? (void*)PTHREAD_CANCELED : (void*)(desr->__exitcode);
	}

	__pthread_desr_destroy(desr);
	free(desr);

exit:
	return nRet;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_detach                                      */
/* Description: Indicate that the thread TH is never to be joined with*/
/*              PTHREAD_JOIN.The resources of TH will therefore be    */
/*              freed immediately when it terminates, instead of      */
/*              waiting for another thread to perform PTHREAD_JOIN    */
/*              on it.                                                */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*           ESRCH : not found thread                                 */
/*           EINVAL: Thread has detached or some thread is joining it.*/
/* Argument - pthread_t th: Dest thread id                            */
/**********************************************************************/
int nes_posix_pthread_detach(pthread_t th)
{
	int ret = 0;
	__pthread_desr* desr = NULL;

	if( th <= 0 )	/* Invalid thread id */
	{
		ret = ESRCH;
		goto exit;
	}	
	
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(th);
	if( !desr )		/* Invalid thread id */
	{
		GLOBAL_THREADMGR_UNLOCK();
		ret = ESRCH;
		goto exit;
	} 

	if( desr->__status & STATUS_THREAD_JOIN_NOW ) 	/* Someone thread is pthread_join(th) now */
	{
		GLOBAL_THREADMGR_UNLOCK();
		ret = EINVAL;
		goto exit;
	}

	if( desr->__attr.__detachstate == 	PTHREAD_CREATE_JOINABLE ) 
	{ 
		desr->__attr.__detachstate = PTHREAD_CREATE_DETACHED;
		__internal_unlock(&(desr->__lock));		/* Make join failed */
		__internal_lock_destroy(&(desr->__lock));	/* Destory join lock */
		ret = 0;
	} 
	else 
	{
		ret = EINVAL;		/* Detached already */ 
	}
	
	GLOBAL_THREADMGR_UNLOCK();

exit:
	return ret;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_setschedparam                               */
/* Description: Set the scheduling parameters for TARGET_THREAD       */
/*              according to POLICY and *PARAM.                       */
/*              But now on PMC extension, not support this funtcions. */
/* Return type: int Always return ENOSYS                              */
/* Argument - pthread_t target_thread:  Dest thread                   */
/* Argument - int policy:  thread policy                              */
/* Argument - const struct sched_param* param:  policy params         */
/**********************************************************************/
int nes_posix_pthread_setschedparam(pthread_t target_thread, int policy, const struct sched_param* param)
{
	W wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	__pthread_desr* desr = NULL;

	/* We don't the policy */
	/* Use taskid and abs priority*/
	wCall = tkse_chg_pri(target_thread, param->sched_priority, P_ABS | P_TASK);

	if( wCall >= 0 )	/* Change successfully */
	{
		GLOBAL_THREADMGR_LOCK();
		desr = __pthread_find_desr(target_thread);
		if( !desr ) /* Invalid thread id */
		{
			GLOBAL_THREADMGR_UNLOCK();
			nRet = ESRCH;		/* Incorrect thread id */
			goto exit;
		}

		nRet = 0;
		desr->__attr.__schedpolicy = policy;			/* Update TCB */
		desr->__attr.__schedparam.sched_priority = param->sched_priority; /* Update TCB */
		GLOBAL_THREADMGR_UNLOCK();
		goto exit;
	}

	GLOBAL_THREADMGR_UNLOCK();

	/* Error code mapping */
	switch ( wCall )
	{
	case E_NOEXS:                  /* Param is invalid */
	case E_ID:                        /* Param is invalid */
		nRet = ESRCH;
		break;
	case E_PAR:			/* Param is invalid */
	default:				/* Unknow error */
		nRet = EINVAL;
		break;
	}

exit:
	return nRet;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_getschedparam                               */
/* Description: Return in *POLICY and *PARAM the scheduling parameters*/ 
/*              for TARGET_THREAD.                                    */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: the thread thread does not exist (e.g. it has */
/*              already termi-nated)                                  */
/* Argument - pthread_t target_thread:  Dest thread id                */
/* Argument - int* policy: Dest policy                                */
/* Argument - struct sched_param* param:Dest policy param             */
/**********************************************************************/
int nes_posix_pthread_getschedparam(pthread_t target_thread, int* policy, struct sched_param* param)
{
	__pthread_desr* desr = NULL;

	if( !policy || !param ) 
	{
		return EINVAL;
	}

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(target_thread);
	if( !desr ) /* Invalid thread id */
	{
		GLOBAL_THREADMGR_UNLOCK();
		return ESRCH;		/* Incorrect thread id */
	}

	*policy			= desr->__attr.__schedpolicy;
	param->sched_priority	= desr->__attr.__schedparam.sched_priority;

	GLOBAL_THREADMGR_UNLOCK();

	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_kill_other_threads_np                       */
/* Description:Terminate all threads in the program except the calling*/
/*              process.                                              */
/* Should be called just before invoking one of the exec*() functions.*/
/* Return type: void                                                  */
/* Argument - void                                                    */
/**********************************************************************/
void nes_posix_pthread_kill_other_threads_np(void)
{
	pthread_t key = 0;
	__pthread_desr* desr = NULL;
	struct hashtable_itr* itr = NULL;
	pthread_t self = nes_posix_pthread_self();

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(self);
	if( !desr )	/* Invalid thread id */
	{
		GLOBAL_THREADMGR_UNLOCK();
		return /* ESRCH */;
	}

	/* Only 'Main' thread can do this */
	if( !(desr->__status & STATUS_THREAD_MAIN_THREAD) ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		return /*EINVAL */;
	}

	while( 1 ) 
	{
		/* 
		 * If only 'Main' thread in __g_threadhash, means all thread has been killed
		 */
		if (hashtable_count(__g_threadhash) <= 1) 
		{
			break;
		}

		itr = hashtable_iterator(__g_threadhash);		/* Get access iterator */
		key = (pthread_t)hashtable_iterator_key(itr);
		if( key == self )						/* Can't kill 'Main' thread */
		{						
			hashtable_iterator_advance(itr);		/* Get next thread */
		}
		key = (pthread_t)hashtable_iterator_key(itr);	/* This thread must be NOT main thread */

		nes_posix_pthread_kill(key, 9/*SIGKILL*/);				/* Don't worry, __g_threadlock is a RECURSIVE lock */
		free(itr);
	}

	GLOBAL_THREADMGR_UNLOCK();
}


