/*
 * samma
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: samma_policy.c,v $
 * $Revision: 1.9 $
 * $Date: 2010/03/15 19:20:35 $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <gmime/gmime.h>
#include <dirent.h>
#include <pthread.h>
#include <libdgstr.h>

#include "mailzip_config.h"
#include "mailzip_tmpl.h"
#include "mailsave.h"
#include "zipconv.h"
#include "log.h"
#include "mailzip_db.h"
#include "maildrop.h"
#include "sendmail.h"
#include "global.h"
#include "samma_policy.h"

#define DEFALT_ENCSTR_YES	"yes"

pthread_mutex_t pw_lock = PTHREAD_MUTEX_INITIALIZER;

/*
 * search_rcptstr_bdb()
 *
 * Rcpt addr of RCTPDB is retrieved.(mail address or FQDN) 
 *
 * DB *dbp              		DBpointer
 * char *dbpath         		DB path string
 * char *searchstr			Search String
 * char *onetime_pass			Random Password
 * int *enc_status			Encription FLAG
 * struct rcptinfo **passlist		List encrypted by fixed password
 * struct rcptinfo ** rdmpasslist	List encrypted by random password
 *
 * return value
 *       ENC            encryption
 *       NOT_ENC        no encryption
 *       DB_ERROR       db error
 *       MALLOC_ERROR   alloc error
 */
int
search_rcptstr_bdb(DB *rcpt_dbp, const char *rcptdbpath, char *rcptaddr, char *searchstr, 
		   char *onetime_pass, int *enc_status, struct rcptinfo **passlist, 
		   struct rcptinfo **rdmpasslist)
{
    int ret;
    char *noenc_searchstr = NULL;
    char *passwd = NULL;

    /* make noenc_searchstr (addr + [!] + \0) */
    noenc_searchstr = malloc(strlen(searchstr) + 2);
    if (noenc_searchstr == NULL) {
        log(ERR_MEMORY_ALLOCATE, "search_rcptstr_bdb", "noenc_searchstr", strerror(errno));
        return MALLOC_ERROR;
    }

    /* search noenc_searchstr */
    sprintf(noenc_searchstr , "!%s", searchstr);

    ret = search_str_bdb(rcpt_dbp, rcptdbpath,
                             searchstr, noenc_searchstr, &passwd);
    if (ret == DB_ERROR) {
        if (passwd != NULL) {
            free(passwd);
            passwd = NULL;
        }
	free(noenc_searchstr);
        return DB_ERROR;
    }
    if (ret == NOT_ENC) {
        DEBUGLOG("Not Encryption: [!]Rcpt address found.(%s)", noenc_searchstr);
        if (passwd != NULL) {
            free(passwd);
            passwd = NULL;
        }
	free(noenc_searchstr);
	return NOT_ENC;
    }
    if (ret == ENC) {
        DEBUGLOG("Rcpt address found.(%s)", searchstr);
	*enc_status = ENC;
        if (passwd != NULL) {
            /* Add address list */
            if (add_enclist(&passlist, searchstr, rcptaddr, passwd) != 0) {
                return MALLOC_ERROR;
            }
            DEBUGLOG("Add keyword to passlist structure.(%s)", searchstr);
            free(passwd);
            passwd = NULL;
        } else {
            /* Add rdmpasslist list */
            if (add_enclist(&rdmpasslist, OTHERKEYWORD, rcptaddr, onetime_pass) != 0) {
                return MALLOC_ERROR;
            }
            DEBUGLOG("Add keyword to rdmpasslist structure.(%s)", searchstr);
        }
	free(noenc_searchstr);
	return ENC;
    }

    free(noenc_searchstr);
    return RECORD_NOTFOUND;
}

/*
 * free_search_rcptaddr_bdb()
 *
 */
void
free_search_rcptaddr_bdb(char *onetime_pass)
{
    if (onetime_pass != NULL) {
	free(onetime_pass);
    }
}

/*
 * search_rcptaddr_bdb()
 *
 * Rcpt addr of RCTPDB is retrieved.
 *
 * config *cfg      			cfg pointer
 * struct rcptaddr *rcptlist		rcpt addr list
 * struct rcptinfo **passlist		List encrypted by fixed password
 * struct rcptinfo ** rdmpasslist	List encrypted by random password
 *
 * return value
 *       ENC        	encryption
 *       NOT_ENC    	no encryption
 *       DB_ERROR   	db error
 *       MALLOC_ERROR   alloc error
 */
int
search_rcptaddr_bdb(struct config *cfg, struct rcptaddr *rcptlist, struct rcptinfo **passlist, struct rcptinfo ** rdmpasslist)
{
    DB *rcpt_dbp;
    char *passwd = NULL;
    char *subdomain = NULL;
    char *fqdn;
    int i, ret, enc_status = NOT_ENC;
    int default_policy = NOT_ENC;
    char *rcptaddr = NULL;
    int rcptaddrlen = 0;
    int pass_len = cfg->cf_passwordlength;
    int defalt_pass_len = strlen(cfg->cf_defaultpassword);
    char *onetime_pass = NULL;

    /* make onetime password */
    ret = mk_passwd(&onetime_pass, pass_len);
    if (ret != PASSWD_SUCCESS) {
        return MALLOC_ERROR;
    }
    DEBUGLOG("Make password ok");

    /* default policy set */
    if ((cfg->cf_defaultencryption[0] == 'Y') || (cfg->cf_defaultencryption[0] == 'y')) {
	default_policy = ENC;
    }

    /* rcptdb open */
    ret = db_open(&rcpt_dbp, cfg->cf_rcptdbpath, cfg->cf_rcptdbtype);
    if (ret == DB_ERROR) {
        free_search_rcptaddr_bdb(onetime_pass);
        return DB_ERROR;
    }

    for (i = 0; (rcptlist + i)->rcpt_addr != NULL; i++) {
        rcptaddr = (rcptlist + i)->rcpt_addr;
        rcptaddrlen = (rcptlist + i)->rcpt_addr_len;
	DEBUGLOG("Check Rcpt Addr.(%s)", rcptaddr);

	ret = search_rcptstr_bdb(rcpt_dbp, cfg->cf_rcptdbpath, rcptaddr, rcptaddr,
				 onetime_pass, &enc_status, passlist, rdmpasslist);
	if (ret < 0) {
	    free_search_rcptaddr_bdb(onetime_pass);
            db_close(rcpt_dbp);
            return DB_ERROR;
	}
        if ((ret == NOT_ENC) || (ret == ENC)) {
	    continue;
        }

        /* make fqdn */
        fqdn = strchr(rcptaddr, ATMARK);

	ret = search_rcptstr_bdb(rcpt_dbp, cfg->cf_rcptdbpath, rcptaddr, fqdn,
				 onetime_pass, &enc_status, passlist, rdmpasslist);
	if (ret < 0) {
	    free_search_rcptaddr_bdb(onetime_pass);
            db_close(rcpt_dbp);
            return DB_ERROR;
	}
        if ((ret == NOT_ENC) || (ret == ENC)) {
	    continue;
        }

        /* search subdomain */
        ret = search_subdomain_bdb(rcpt_dbp, cfg->cf_rcptdbpath, fqdn, &subdomain, &passwd);
	if (ret < 0) {
	    free_search_rcptaddr_bdb(onetime_pass);
            db_close(rcpt_dbp);
            return DB_ERROR;
	}
        if (ret == NOT_ENC) {
            if (passwd != NULL) {
                free(passwd);
                passwd = NULL;
            }
	    continue;
        }
        if (ret == ENC) {
            DEBUGLOG("Rcpt domain found.(%s)", subdomain);
	    enc_status = ENC;
            if (passwd != NULL) {
                /* Add address list */
                if (add_enclist(&passlist, subdomain, rcptaddr, passwd) != 0) {
		    free_search_rcptaddr_bdb(onetime_pass);
                    db_close(rcpt_dbp);
                    return MALLOC_ERROR;
                }
                DEBUGLOG("Add keyword to passlist structure.(%s)", rcptaddr);
                free(passwd);
                passwd = NULL;
            } else {
                /* Add rdmpasslist list */
                if (add_enclist(&rdmpasslist, OTHERKEYWORD, rcptaddr, onetime_pass) != 0) {
		    free_search_rcptaddr_bdb(onetime_pass);
                    db_close(rcpt_dbp);
                    return MALLOC_ERROR;
                }
                DEBUGLOG("Add keyword to rdmpasslist structure.(%s)", rcptaddr);
            }
            continue;
        }

	/* default policy check */
	if (default_policy == ENC) {
            enc_status = ENC;
	    if (defalt_pass_len > 0) {
	        DEBUGLOG("DefaultPolicy YES: DefaultPassword YES.(%s)", rcptaddr);
                /* Add address list */
                if (add_enclist(&passlist, rcptaddr, rcptaddr, 
							cfg->cf_defaultpassword) != 0) {
		    free_search_rcptaddr_bdb(onetime_pass);
                    db_close(rcpt_dbp);
                    return MALLOC_ERROR;
                }
                DEBUGLOG("Add keyword to passlist structure.(%s)", rcptaddr);
            } else {
	        DEBUGLOG("DefaultPolicy YES: DefaultPassword NO.(%s)", rcptaddr);
                /* Add rdmpasslist list */
                if (add_enclist(&rdmpasslist, OTHERKEYWORD, rcptaddr, onetime_pass) != 0) {
		    free_search_rcptaddr_bdb(onetime_pass);
                    db_close(rcpt_dbp);
                    return MALLOC_ERROR;
                }
                DEBUGLOG("Add keyword to rdmpasslist structure.(%s)", rcptaddr);
	    }
	}

    } // rcpt addrlist roop

    free_search_rcptaddr_bdb(onetime_pass);
    db_close(rcpt_dbp);

    return enc_status;
}

/*
 * add_enclist()
 *
 * struct rcptinfo 	***head
 * char 		*keyword
 * char 		*addr
 * char 		*passwd
 *
 * return value
 *       0		Success
 *       -1		Error
 */
int
add_enclist(struct rcptinfo ***head, char *keyword, char *addr, char *passwd)
{
    struct rcptinfo *p, *tmpp = NULL, *rcptinfo;

    if (*head != NULL) {
        for (p = **head; p != NULL; p = p->Next) {
            /* set last structure */
            tmpp = p;

            if (strncmp(p->keyword, keyword, p->keyword_len) != 0) {
                continue;
            }
            /* keyword found from rcptinfo list */
            if (push_rcptlist(&(p->rcptlist), addr) != 0) {
                return -1;
            }
            return 0;
        }
    }

    if (passwd == NULL) {
        log(ERR_PASS_MUST_BE_SET, "add_enclist");
        return -1;
    }

    /* create rcptinfo structure */
    rcptinfo = (struct rcptinfo *)malloc(sizeof(struct rcptinfo));
    if (rcptinfo == NULL) {
        log(ERR_MEMORY_ALLOCATE, "add_enclist", "rcptinfo", strerror(errno));
        return -1;
    }
    memset(rcptinfo, 0, sizeof *rcptinfo);

    /* not allocate memory */
    rcptinfo->keyword = keyword;
    rcptinfo->keyword_len = strlen(keyword);

    /* allocate memory */
    rcptinfo->passwd = strdup(passwd);
    if (rcptinfo->passwd == NULL) {
        log(ERR_MEMORY_ALLOCATE, "add_enclist", "rcptinfo->passwd", strerror(errno));
        return -1;
    }

    if (push_rcptlist(&(rcptinfo->rcptlist), addr) != 0) {
        free(rcptinfo);
        return -1;
    }
    rcptinfo->Next = NULL;

    if (**head != NULL) {
        tmpp->Next = rcptinfo;
    } else {
        **head = rcptinfo;
    }

    return 0;
}

/*
 * search_subdomain_bdb()
 *
 * Check if the subdomain exists in DB.
 *
 * args:
 * DB *dbp              pointer
 * char *dbpath         string
 * char *subdomain      string
 * char **match_domain  pointer
 * char **match_passwd  pointer
 *
 * return value
 *       ENC            	encryption
 *       NOT_ENC        	no encryption
 *       RECORD_NOTFOUND	record not found
 *       DB_ERROR       	db error
 *       MALLOC_ERROR   	alloc error
 */
int
search_subdomain_bdb(DB *dbp, const char *dbpath, char *subdomain,
                     char **match_domain, char **match_passwd)
{
    char *not_subdomain;
    char *passwd = NULL;
    int ret;

    /* make not_subdomain */
    not_subdomain = malloc(strlen(subdomain) + 2);
    if (not_subdomain == NULL) {
        log(ERR_MEMORY_ALLOCATE, "search_subdomain_bdb",
                                 "not_subdomain", strerror(errno));
        return MALLOC_ERROR;
    }

    do {
        /* pointer shift */
        subdomain++;

        /* make not_subdomain */
        sprintf(not_subdomain, "!%s", subdomain);

        /* search subdomain */
        ret = search_str_bdb(dbp, dbpath,
                             subdomain, not_subdomain, &passwd);
        if (ret == DB_ERROR) {
            if (passwd != NULL) {
                DEBUGLOG("free passwd.");
                free(passwd);
                passwd = NULL;
            }
            free(not_subdomain);
            return DB_ERROR;
        }
        if (ret == NOT_ENC) {
            DEBUGLOG("NOT_Subdomain found.(%s)", not_subdomain);
            if (passwd != NULL) {
                DEBUGLOG("free passwd.");
                free(passwd);
                passwd = NULL;
            }
            free(not_subdomain);
            return NOT_ENC;
        }
        if (ret == ENC) {
            DEBUGLOG("Subdomain found.(%s)", subdomain);
            DEBUGLOG("passwd.(%s)", passwd);
            *match_passwd = passwd;
            *match_domain = subdomain;
            free(not_subdomain);
            return ENC;
        }
    } while ((subdomain = strchr(subdomain + 1, DOT)) != NULL);
    
    if (passwd != NULL) {
	DEBUGLOG("free passwd.");
        free(passwd);
        passwd = NULL;
    }
    free(not_subdomain);
    return RECORD_NOTFOUND;
}

/*
 * search_fromaddr_bdb()
 *
 * Check if the address exists in senderDB.
 *
 * args:
 * config *cfg      pointer
 * char *fromaddr   string
 *
 * return value:
 *       ENC                  encryption
 *       NOT_ENC              no encryption
 *       DB_ERROR             db error
 *       MALLOC_ERROR         alloc error
 */
int
search_fromaddr_bdb(struct config *cfg, char *fromaddr)
{
    DB *sender_dbp;
    char *mdom = NULL;
    char *passwd = NULL;
    char *not_fromaddr;
    char *fqdn;
    int ret;

    /* senderdb open */
    ret = db_open(&sender_dbp, cfg->cf_senderdbpath, cfg->cf_senderdbtype);
    if (ret == DB_ERROR) {
        return DB_ERROR;
    }

    /* make not_fromaddr */
    not_fromaddr = malloc(strlen(fromaddr) + 2);
    if (not_fromaddr == NULL) {
        log(ERR_MEMORY_ALLOCATE, "search_fromaddr_bdb", "not_fromaddr", strerror(errno));
        db_close(sender_dbp);
        return MALLOC_ERROR;
    }
    sprintf(not_fromaddr , "!%s", fromaddr);

    /* search address */
    ret = search_str_bdb(sender_dbp, cfg->cf_senderdbpath,
                         fromaddr, not_fromaddr, &passwd);
    if (passwd != NULL) {
        DEBUGLOG("free passwd.", passwd);
        free(passwd);
        passwd = NULL;
    }
    if (ret == DB_ERROR) {
        free(not_fromaddr);
        db_close(sender_dbp);
        return DB_ERROR;
    }
    if (ret == NOT_ENC) {
        DEBUGLOG("Sender not_fromaddr found.(%s)", not_fromaddr);
        free(not_fromaddr);
        db_close(sender_dbp);
        return NOT_ENC;
    }
    if (ret == ENC) {
        DEBUGLOG("Sender fromaddr found.(%s)", fromaddr);
        free(not_fromaddr);
        db_close(sender_dbp);
        return ENC;
    }

    /* make fqdn */
    fqdn = strchr(fromaddr, ATMARK);

    /* make not_fqdn */
    sprintf(not_fromaddr , "!%s", fqdn);

    /* search fqdn */
    ret = search_str_bdb(sender_dbp, cfg->cf_senderdbpath,
                         fqdn, not_fromaddr, &passwd);
    if (passwd != NULL) {
        DEBUGLOG("free passwd.", passwd);
        free(passwd);
        passwd = NULL;
    }
    if (ret == DB_ERROR) {
        free(not_fromaddr);
        db_close(sender_dbp);
        return DB_ERROR;
    }
    if (ret == NOT_ENC) {
        DEBUGLOG("Sender not_fqdn found.(%s)", not_fromaddr);
        free(not_fromaddr);
        db_close(sender_dbp);
        return NOT_ENC;
    }
    if (ret == ENC) {
        DEBUGLOG("Sender fqdn found.(%s)", fqdn);
        free(not_fromaddr);
        db_close(sender_dbp);
        return ENC;
    }

    /* free not_fromaddr */
    free(not_fromaddr);

    /* search subdomain */
    ret = search_subdomain_bdb(sender_dbp, cfg->cf_senderdbpath, fqdn, &mdom, &passwd);
    if (passwd != NULL) {
        DEBUGLOG("free passwd.", passwd);
        free(passwd);
        passwd = NULL;
    }
    if (ret < 0) {
        /* DB_ERROR or MALLOC_ERROR */
        db_close(sender_dbp);
        return ret;
    }
    if ((ret == NOT_ENC) || (ret == RECORD_NOTFOUND)) {
        /* Not encryption */
        db_close(sender_dbp);
        return NOT_ENC;
    }
    DEBUGLOG("match subdomain: %s", mdom);
    db_close(sender_dbp);

    return ENC;
}

char seed[] = SEED;
/*
 * mk_passwd
 *
 * Args:
 *      char    **passwd        passwd string pointer
 *      int       count         passwd length
 *
 * Returns:
 *      PASSWD_SUCCESS          Success
 *      PASSWD_FAILED           Failed
 */
int
mk_passwd(char **passwd, int count)
{
    long val;
    int i;

    *passwd = (char *)malloc(sizeof(char *) * count + 1);
    if (*passwd == NULL) {
        log(ERR_MEMORY_ALLOCATE, "mk_passwd", "passwd", strerror(errno));
        return(PASSWD_FAILED);
    }

    pthread_mutex_lock(&pw_lock);

    for(i = 0; i < count; i++) {
        val = lrand48() % (sizeof(seed) - 1);

        *(*(passwd) + i) = seed[val];

    }
    *(*(passwd) + count) = '\0';

    pthread_mutex_unlock(&pw_lock);

    return(PASSWD_SUCCESS);
}


/*
 * search_str_bdb()
 *
 * Check if the string exists in DB.
 *
 * args:
 * config *cfg      pointer
 * char *fromaddr   string
 *
 * return value:
 *       ENC                  str found
 *       NOT_ENC              not str found
 *       RECORD_NOTFOUND      not found
 *       DB_ERROR             db error
 */
int
search_str_bdb(DB *dbp, const char *dbpath, char *search_str,
               char *not_search_str, char **match_passwd)
{
    int ret;

    /* search not_search_str */
    ret = db_search(dbp, dbpath, not_search_str, match_passwd);
    if (ret == DB_ERROR) {
        return DB_ERROR;
    }
    if (ret == FOUND) {
        DEBUGLOG("not_search_str found.(%s)", not_search_str);
        return NOT_ENC;
    }

    /* search search_str */
    ret = db_search(dbp, dbpath, search_str, match_passwd);
    if (ret == DB_ERROR) {
        return DB_ERROR;
    }
    if (ret == FOUND) {
        DEBUGLOG("search_str found.(%s)", search_str);
        return ENC;
    }

    return RECORD_NOTFOUND;

}
