/*
 * samma
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: mailzip_tmpl.c,v $
 * $Revision: 1.5 $
 * $Date: 2010/09/27 07:33:44 $
 */

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <libdgstr.h>
#include <gmime/gmime.h>
#include "log.h"
#include "mailzip_config.h"
#include "maildrop.h"
#include "global.h"
#include "mailzip_tmpl.h"

/*
 * tmpl_read()
 *
 * Read the template file.
 *
 * Args:
 *   char **tmp      pointer
 *   char *tmp_file  file path
 *
 * Return value:
 *   SUCCESS	   0    success
 *   ERROR	   -1    error
 *
 */
int
tmpl_read(char **tmp, char *tmpl_file) 
{
    int fd, ret;
    char buf[BUFSIZE];
    char *tmpr = NULL;
    int total = 0;

    // open template file
    fd = open(tmpl_file, O_RDONLY);
    if (fd == -1) {
        log(ERR_FILE_OPEN, "tmpl_read", tmpl_file);
        return(ERROR);
    }

    // read template file
    do {

        ret = read(fd, buf, BUFSIZE - 1);
        if (ret == -1) {
            log(ERR_FILE_GET_INFO, "tmpl_read", tmpl_file);
	    if (*tmp != NULL) {
		free(*tmp);
		*tmp = NULL;
	    }
            close(fd);
            return(ERROR);
        } else if (ret != 0) {
            buf[ret] = '\0';

            tmpr = (char *)realloc(*tmp, (total + ret + 1));
            if (tmpr == NULL) {
                if (*tmp != NULL) {
                    free(*tmp);
		    *tmp = NULL;
                } 
                log(ERR_MEMORY_ALLOCATE, "tmpl_read", "tmpr", strerror(errno));
                close(fd);
                return(ERROR);
            }
            *tmp = tmpr;

            strcpy((*tmp + total), buf);
            total = total + ret;
	}

    } while (ret != 0);

    //file close
    close(fd);

    if (*tmp == NULL) {
        log(ERR_TEMPLATE_READ, "tmpl_read", tmpl_file);
        return(ERROR);
    }

    return(SUCCESS);
}

#define NOSUBJECT	""
/*
 * tmpl_tag_replace()
 *
 * Args:
 *   char *           read file data 
 *   struct mailzip   reference data
 *   struct rcptinfo  reference data
 *   char *           recipient address 
 *   char **          replaced date
 *
 * Return:
 *   ERROR		-1   error
 *   SUCCESS		0   success
 */
int 
tmpl_tag_replace(char *tmp, struct mailzip mz, struct rcptinfo *other, char *sender, char **retp)
{

    int i;
    char *subject = NULL;
    char *filename = NULL;
    char *tmpl = NULL;
    char *rcptaddr, *tmpp = NULL, *tmptagp = NULL;
    int ret = 0, total = 0;
    int rcptaddr_len = 0;
    char *name, *tmpf = NULL;
    int name_len = 0, ret_len = 0, total_len = 0;
    int tagnum = 0;

    struct strtag tmpl_tag[] = {
        {"SUBJECT", 7, NULL},
        {"DATE", 4, NULL},
        {"PASSWORD", 8, NULL},
        {"RCPTLIST", 8, NULL},
        {"FILENAME", 8, NULL},
        {"TOADDR", 6, NULL}
    };
    tagnum = sizeof(tmpl_tag) / sizeof(struct strtag);

    // set subject
    if (mz.subject != NULL) {
	if (dg_str2code((char *)mz.subject, &subject, STR_UTF8, STR_JIS) != 0) {
            log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	    return(ERROR);
	}
        tmpl_tag[SUBJECT].st_str = subject;
    } else {
        tmpl_tag[SUBJECT].st_str = NOSUBJECT;
    }

    // set date
    tmpl_tag[DATE].st_str = mz.date;

    // set passwd
    tmpl_tag[PASSWORD].st_str = other->passwd;

    // set toaddr(sender address)
    tmpl_tag[TOADDR].st_str = sender;

    // rcptaddr total length
    for (i = 0; (other->rcptlist + i)->rcpt_addr != NULL; i++) {
        rcptaddr_len = (other->rcptlist + i)->rcpt_addr_len;
        ret += rcptaddr_len + 1;
    }

    // allocate memory
    tmpp = (char *)malloc(ret + 1);
    if (tmpp == NULL) {
        log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmpp", strerror(errno));
	if (subject != NULL) {
	    free(subject);
	    subject = NULL;
	}
        return(ERROR);
    }

    // reset memory
    *tmpp = '\0';

     // write memory
    for (i = 0; (other->rcptlist + i)->rcpt_addr != NULL; i++) {
        rcptaddr = (other->rcptlist + i)->rcpt_addr;
        rcptaddr_len = (other->rcptlist + i)->rcpt_addr_len;

        strcat(tmpp, rcptaddr);
        *(tmpp + total + rcptaddr_len) = '\n';
        *(tmpp + total + rcptaddr_len + 1) = '\0';
        total = total + rcptaddr_len + 1;
    }

    // last '\n' change into '\0'
    *(tmpp + total - 1) = '\0';

    // store all rcptaddr 
    tmpl_tag[RCPTLIST].st_str = tmpp;

    //file_name
    if (mz.namelist != NULL) {
        for (i = 0; (mz.namelist + i)->attach_name != NULL; i++) {
            // total length
            ret_len = ret_len + (mz.namelist + i)->attach_name_len + 1;
        }
        // allocate memory
        tmpf = (char *)malloc(ret_len + 1);
        if (tmpf == NULL) {
            log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmpf", strerror(errno));
	    free(tmpp);
	    if (subject != NULL) {
	        free(subject);
	    }
            return(ERROR);
        }
        // reset memory
        *tmpf = '\0';
    } else {
        log(ERR_NULL_FILE_NAME, "tmpl_tag_replace");    
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
        return(ERROR);
    }

    // write memory
    for (i = 0; (mz.namelist + i)->attach_name != NULL; i++) {
        name = (mz.namelist + i)->attach_name;
        name_len = (mz.namelist + i)->attach_name_len;

        strcat(tmpf, name);
        *(tmpf + total_len + name_len) = '\n';
        *(tmpf + total_len + name_len + 1) = '\0';
        total_len = total_len + name_len + 1;
    }

    // last '\n' change into '\0'
    *(tmpf + total_len - 1) = '\0';
    // store all filename 
    if (dg_str2code(tmpf, &filename, STR_UTF8, STR_JIS) != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
	free(tmpf);
	return(ERROR);
    }
    tmpl_tag[FILENAME].st_str = filename;
    free(tmpf);

    if (dg_str2code(tmp, &tmpl, STR_UTF8, STR_JIS) != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
	free(filename);
	return(ERROR);
    }

    //replace tag
    tmptagp = str_replace_tag(tmpl, STARTTAG, ENDTAG, tmpl_tag, tagnum);
    // resouce free
    free(tmpp);
    if (subject != NULL) {
	free(subject);
    }
    free(filename);
    free(tmpl);

    // str_replace_tag error check
    if (tmptagp == NULL) {
        log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmptagp", strerror(errno));
        return(ERROR);
    }

    //convert str code
    if (dg_str2code(tmptagp, retp, STR_UTF8, STR_JIS) != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmptagp);
	return(ERROR);
    }
    free(tmptagp);

    return(SUCCESS);
}

