﻿<?php
/**
 * データベースアクセスに関するパッケージです
 *
 * このパーッケージのデータベースへアクセスはPDOを使用します。
 * データベースの接続文字はこのパッケージ内に記述しています。
 * 例外が発生した時はこのクラスを呼び出し元で処理してください
 * 
 * @category   Database
 * @package    dbutil
 * @author     okotaneko
 * @version    1.0
 * @access     public
 */

/**
 * inlcude DocBlock です。
 *
 * inlcude ファイルの説明など書きます。
 */

require('dbparam.inc');


/**
 * データベース接続クラス
 *
 * データベースへの接続と切断を管理します。
 * データベース接続のインスタンスをデータアクセスクラスに渡します。
 * 
 * @category   database
 * @package    dbutil
 * @author     okotaneko
 * @version    0.1
 * @access     public
 */
class DBConnection {

	/**
	 * データベースインスタンス
	 * @var object 接続時のインスタンスを保持する
	 * @access public
	 */
	private $db = null;

	/**
	 * データベース接続状態の保持
	 * 
	 * @var Bool データベースの接続状態 接続中 = True 未接続 = False
	 * @access public
	 */
	private $ConnectFlg = False;

	/**
	 * データベース接続文字列
	 * 
	 * @var string データベースの接続文字列
	 * @access public
	 */
	private $DBparam = null;

	/**
	 * データベースユーザー名
	 * 
	 * @var string データベースのユーザー名
	 * @access public
	 */
	private $DBUser = null;

	/**
	 * データベースユーザーパスワード
	 * 
	 * @var string データベースのユーザーパスワード
	 * @access public
	 */
	private $DBPassWD = null;


	/**
	 * コンストラクタ
	 * 
	 * データベースの接続文字の設定は、データベースの種類（$DBType）が設定されていない場合デフォルト値を使います。
	 * 
	 * @param string $DBType データベースの種類
	 * @param string $Host データベース接続先ホスト名
	 * @param string $Port データベース接続先ポート名
	 * @param string $DBName データベース名
	 * @param string $UserName データベースユーザ名
	 * @param string $PassWD データベースパスワード
	 * @return void
	 * @access public
	 */	
	public function __construct($DBType = null, $Host = null, $Port = null, $DBName = null, $UserName = null, $PassWD = null) {

		if (is_null($DBType)) {
			$this->DBparam = DB_CON_TYPE . ':dbname=' . DB_CON_NAME . ';host=' . DB_CON_HOST . ' port=' . DB_CON_PORT;
			$this->DBUser = DB_CON_USER;
			$this->DBPassWD = DB_CON_PASS;
		} else {
			$this->Dbparam = $DBType . ':dbname=' . $DBName . ' host=' . $Host . ' port=' . $Port;
			$this->DBUser = $UserName;
			$this->DBPassWD = $PassWD;
		}
		
		
		$this->Connect();
	}

	/**
	 * デストラクタ
	 *
	 * @return void
	 * @access public
	 */
	public function __destruct() {
		$db = null;
	}

	/**
	 * データベースに接続します。。
	 * データベース接続のインスタンスを作成します。
	 *
	 * @return void 
	 * @access public
	 */
	public function Connect() {

		if (! $this->ConnectFlg) {
			$this->DisConnect();
	}

		$this->db = new PDO($this->DBparam, $this->DBUser, $this->DBPassWD);
	    $this->db->query('CHAR_SET');
		$this->ConnectFlg = True;
	}
	
	/**
	 * データベースの接続を切断します。。
	 * データベース接続のインスタンスを解放します。
	 *
	 * @return void 
	 * @access public
	 */
	public function DisConnect() {

		$this->db = null;
		$this->ConnectFlg = False;
	}

	/**
	 * データベース接続状態を返す
	 * 
	 * @return Bool $ConnectFlg 接続中=True 未接続=False
	 * @access public
	 */
	public function Connected() {

		return $this->ConnectFlg;
	}

	/**
	 * データベース接続インスタンスをセッションクラスに渡す
	 *
	 * @param object $DBsession セッションクラスのインスタンス
	 * @return void 
	 * @access public
	 */
	public function AssignDBConnection($DBsession){

		$DBsession->SetDBConnection($this->db);
	}

	/**
	 * クラスに存在しないプロパティへの書込みははエラーとする
	 *
	 * @param string $name アクセスするプロパティ名
	 * @param string $value プロパティ値
	 * @return void
	 * @access public
	 */
	function __set($name, $value) {

		throw new Exception('存在しないプロパティへのアクセス');
	}

	/**
	 * クラスに存在しないプロパティの読込はエラーとする
	 *
	 * @param string $name アクセスするプロパティ名
	 * @return oject
	 * @access public
	 */
	function __get($name) {

		throw new Exception('存在しないプロパティへのアクセス');
	}

	/**
	 * クラスに存在しないメソッドの実行ははエラーとする
	 *
	 * @param string $name アクセスするメソッド名
	 * @param array $arguments メソッドの引数(ハッシュ配列)
	 * @return void
	 * @access public
	 */
	function __call($name, $arguments) {

		throw new Exception('存在しないメソッドへのアクセス');
	}
}

/**
 * データベースアクセスクラス
 *
 * データベースへのセッションを保持します。。
 * データベース接続のインスタンスをコネクションクラスから取得しデータベースへアクセスします。
 * 例外が発生した時はこのクラスを呼び出し元で処理してください
 * 
 * @category   databese
 * @package    dbutil
 * @author     okotaneko
 * @version    0.1
 * @access     public
 */
class DBSession {
	
	/**
	 * データベースインスタンス
	 * データベースインスタンスを接続クラスからもらいます。(参照)
	 * @var object データベース接続インスタンス
	 * @access public
	 */	
	private $Db = null;
	
	/**
	 * コンストラクタ
	 *
	 * @param object $DBConnection データベース接続のインスタンス ない場合は渡さなくてもよい
	 * @return void
	 * @access public
	 */
	public function __construct($DBConnection = null) {
		
		if (! is_null($DBConnection)) {
			$this->SetDBConnection($DBConnection);
		}
	}
	
	/**
	 * デストラクタ
	 *
	 * @return void
	 * @access public
	 */
	public function __destruct() {

	}

    /**
     * データベースコネクションクラスをセットする
     * データベース接続のインスタンスの参照をセットします。
     *
     * @param object $DBConnection データベース接続のインスタンス
     * @return void 
     * @access public
     */
	public function SetDBConnection($DBConnection) {

		$this->Db = $DBConnection;
	}

	/**
	 * select文を発行する
	 *
	 * @param array $s PDOStatementオブジェクト
	 * @param array $param バインド変数の項目と値のハッシュ配列
	 * @return void
	 * @access private
	 */
	private function setBindParam(&$s, &$param) {

		if (! empty($param)) {
			foreach ($param as $key => $value) {
				$s->bindParam("$key", $value);
			}
		}
	}

	/**
	 * select文を発行する
	 *
	 * @param string $sql 実行するSQL文必ずバインド変数を使用すること
	 * @param array $param バインド変数の項目と値のハッシュ配列
	 * @return array string 結果セット
	 * @return void
	 * @access public
	 */
	public function query_select(&$sql, &$param) {

		$s = $this->Db->prepare($sql);
		$this->setBindParam($s, $param);

		if (! $s->execute($param)) {
			throw new Exception('0030001', SYSTEM_ERROR);
		}

		return $s->fetchAll();

	}

	/**
	 * insert update delete文を発行する
	 *
	 * @param string $sql 実行するSQL文必ずバインド変数を使用すること
	 * @param array $param バインド変数の項目と値のハッシュ配列
	 * @return integer 更新した行数
	 * @access public
	 */
	public function query_update(&$sql, &$param) {

		$s = $this->Db->prepare($sql);
		$this->setBindParam($s, $param);

		if (! $s->execute($param)) {
			throw new Exception('0030002', SYSTEM_ERROR);
		}

		return $s->rowCount();

	}

	/**
	 * クラスに存在しないプロパティへの書込みははエラーとする
	 *
	 * @param string $name アクセスするプロパティ名
	 * @param string $value プロパティ値
	 * @return void
	 * @access public
	 */
	function __set($name, $value) {

		throw new Exception('存在しないプロパティへのアクセス');
	}

	/**
	 * クラスに存在しないプロパティの読込はエラーとする
	 *
	 * @param string $name アクセスするプロパティ名
	 * @return oject
	 * @access public
	 */
	function __get($name) {

		throw new Exception('存在しないプロパティへのアクセス');
	}

	/**
	 * クラスに存在しないメソッドの実行ははエラーとする
	 *
	 * @param string $name アクセスするメソッド名
	 * @param array $arguments メソッドの引数(ハッシュ配列)
	 * @return void
	 * @access public
	 */
	function __call($name, $arguments) {

		throw new Exception('存在しないメソッドへのアクセス');
	}
	
}

