#include "saphire.h"
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <locale.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <glob.h>
#include <libgen.h>
#include <dirent.h>
#include <oniguruma.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "config.h"

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

#include <limits.h>
#include <sys/stat.h>

#if defined(__CYGWIN__)
#include <sys/stat.h>
#endif

#ifdef __LINUX__

/*
#define __USE_POSIX
#define __USE_XOPEN_EXTENDED
#define __USE_POSIX199309
#define __USE_UNIX98
*/
#include <signal.h>
#include <sys/wait.h>

#else

#include <signal.h>
#include <sys/stat.h>

#endif

#if defined(__DARWIN__)
#include <sys/syslimits.h>
#endif

#ifdef __FREEBSD__
#include <limits.h>
#endif

#include "saphire_extra.h"
/*
#if !defined(__FREEBSD__)
void srandom(unsigned int seed);
#endif
int setenv(const char *name, const char *value, int overwrite);
int unsetenv(const char *name);
int strcasecmp(const char *s1, const char *s2);
*/

/// from readline.c ///
void readline_init();
void readline_final();

//////////////////////////////////////////////////////////////////////
// exit時に実行される関数
//////////////////////////////////////////////////////////////////////
static void atexit_fun()
{
    if(mis_raw_mode()) {
        mendwin();
        //mreset_tty();
    }
}


//////////////////////////////////////////////////////////////////////
// メイン関数
//////////////////////////////////////////////////////////////////////
static void usage()
{
    printf("usage sash or saphire [-c command] [ -s -w -e ] [ -Lu -Lm -Lw ] [--version ] [ script file]\n\n");
    printf("-c : eval a command on saphire\n");
    printf("-rs : use run time a source run time script\n");
    printf("-ro : use run time a compiled run time script(default)\n");
    printf("-rn : no use run time script\n");
    printf("-ts : set termianl kanjicode sjis\n");
    printf("-tw : set termianl kanjicode utf8\n");
    printf("-te : set termianl kanjicode eucjp\n");
    printf("-s : set saphire kanjicode sjis\n");
    printf("-w : set saphire kanjicode utf8\n");
    printf("-e : set saphire kanjicode eucjp\n");
    printf("-Lu : set line field LF\n");
    printf("-Lm : set line field CR\n");
    printf("-Lw : set line field CRLF\n");
    printf("--version : display saphire version\n");

    exit(0);
}

static void version()
{
    printf("saphire version %s.", getenv("SAPHIRE_VERSION"));
    printf("author Daisuke Minato.\n");
    
    exit(0);
}

void main_saphire_job_done(int job_num, char* job_title)
{
    printf("%d: %s done.\n", job_num+1, job_title);
}

static int mreadline_signal()
{
    struct sigaction sa2;
    memset(&sa2, 0, sizeof(sa2));
    sa2.sa_handler = SIG_DFL;
//    sa2.sa_flags |= SA_RESTART;
    if(sigaction(SIGINT, &sa2, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }
}

#if defined(ENABLE_READLINE)
void sig_int_readline(int signo)
{
    rl_kill_full_line(0,0);
    rl_reset_line_state();
    rl_crlf();
    rl_redisplay();
}

static int readline_signal()
{
    struct sigaction sa2;
    memset(&sa2, 0, sizeof(sa2));
//    sa2.sa_handler = SIG_DFL;
    sa2.sa_handler = sig_int_readline;
    sa2.sa_flags |= SA_RESTART;
    if(sigaction(SIGINT, &sa2, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }
}
#endif

int main(int argc, char* argv[])
{
    /// メモリリークの監視を開始 ///
    CHECKML_BEGIN();

    /// 乱数初期化 ///
    srandom(1000);

    /// ロケール初期化 ///
    setlocale(LC_CTYPE, "ja_JP.UTF-8");

    /// オプション処理 ///
    char send_cmd[BUFSIZ];
    char file_name[PATH_MAX];
    file_name[0] = 0;

    setenv("LINEFIELD","LF", 1);
    enum eTerminalKanjiCode code = kTKUtf8;

    char cwd[PATH_MAX];
    mygetcwd(cwd);
    setenv("PWD", cwd, 1);

    int i;
    enum eRuntimeScript runtime_script = kRSObject;
    for(i=1 ; i<argc; i++){
        // ファイル名
        if(argv[i][0] != '-') {
            strcpy(file_name, argv[i]);
        }
        // 引数
        else {
            if(strcmp(argv[i], "--version") == 0) {
                saphire_set_signal();
                saphire_init(kATConsoleApp, FALSE, runtime_script);
                version();
                break;
            }
            else if(strcmp(argv[i], "-ts") == 0) {
                code = kTKSjis;
            }
            else if(strcmp(argv[i], "-tw") == 0) {
                code = kTKUtf8;
            }
            else if(strcmp(argv[i], "-te") == 0) {
                code = kTKEucjp;
            }
            else if(strcmp(argv[i], "-rs") == 0) {
                runtime_script = kRSSource;
            }
            else if(strcmp(argv[i], "-ro") == 0) {
                runtime_script = kRSObject;
            }
            else if(strcmp(argv[i], "-rn") == 0) {
                runtime_script = kRSNoRead;
            }
            else {
                switch (argv[i][1]){
                    case 'c':
                        if(i+1 < argc) {
                            snprintf(send_cmd, BUFSIZ, "c%s", argv[i+1]);
                            i++;
                        }
                        else {
                            usage();
                        }
                        break;

                    case 's':
                        gKanjiCode = kSjis;
                        break;

                    case 'e':
                        gKanjiCode = kEucjp;
                        break;

                    case 'w':
                        gKanjiCode = kUtf8;
                        break;

                    case 'L':
                        if(strcmp(argv[i], "-Lu") == 0) {
                            gLineField = kLF;
                        }
                        else if(strcmp(argv[i], "-Lw") == 0) {
                            gLineField = kCRLF;
                        }
                        else if(strcmp(argv[i], "-Lm") == 0) {
                            gLineField = kCR;
                        }
                        else {
                            fprintf(stderr, "invalid option %s", argv[i]);
                            exit(1);
                        }
                        break;

                    default:
                        usage();
                }
            }
        }
    }

    /// 引数のファイル名がディレクトリじゃなければスクリプトと判断する ///
    if(strcmp(file_name, "") != 0) {
        struct stat stat_;
        if(stat(file_name, &stat_) < 0) {
            printf("argument file name is invalid\n");
            exit(1);
        }

        if(!S_ISDIR(stat_.st_mode)) {
            mcurses_init(code);
            saphire_set_signal();
            saphire_init(kATConsoleApp, FALSE, runtime_script);
            sWFd* pipeout = sWFd_new(STDOUT_FILENO);
            int rcode = saphire_load(file_name, pipeout, STDIN_FILENO, STDERR_FILENO);
            sWFd_flash(pipeout);
            sWFd_delete(pipeout);
            if(rcode != 0) {
                fprintf(stderr, "run time err\n");
                fprintf(stderr, "%s", string_c_str(gErrMsg));
            }
            saphire_final();
            mcurses_final();
            CHECKML_END();

            return rcode;
        }
    }

    /// オプションで指定されているならシェルで実行する ///
    if(send_cmd[0] == 'c') {
        mcurses_init(code);
        saphire_set_signal();
        saphire_init(kATOptC, FALSE, runtime_script);
        sWFd* pipeout = sWFd_new(STDOUT_FILENO);
        sRFd* pipein = sRFd_new(STDIN_FILENO);
        int rcode = saphire_shell(send_cmd + 1, "saphire -a", pipeout, pipein, STDERR_FILENO);
        sWFd_flash(pipeout);
        sWFd_delete(pipeout);
        sRFd_delete(pipein);
        if(rcode < 0) { 
            fprintf(stderr, "run time err\n");
            fprintf(stderr, "%s", string_c_str(gErrMsg));
        }
        saphire_final();
        mcurses_final();
        CHECKML_END();

        return rcode;
    }

#if !defined(ENABLE_READLINE)
    fprintf(stderr, "if you use saphire as interactive shell, use sash command instead\n");
    return 0;
#else
    /// インタラクティブシェル開始 ///
    mcurses_init(code);

    saphire_set_signal();
    saphire_init(kATConsoleApp, TRUE, runtime_script);
    saphire_rehash();
    saphire_job_done = main_saphire_job_done;
    readline_init();

    /// ヒストリーを読み込む ///
    char* histfile = getenv("SAPHIRE_HISTFILE");
    if(histfile) {
        read_history(histfile);
    }
    int histsize = atoi(getenv("SAPHIRE_HISTSIZE"));
    if(histsize < 0) histsize = 1000;

    //saphire_set_signal();

    char* buf;
    int history_num = 0;
    while(1) {
        /// 一行読み込み ///
        readline_signal();

        /// 終了コードの保存 ///
        char end_code[128];
        char* lvar =  saphire_get_local_var("RCODE");
        if(lvar) 
            snprintf(end_code, 128, "%s", lvar);
        else 
            snprintf(end_code, 128, "0");

        /// プロンプト ///
        char* prompt = getenv("SAPHIRE_PROMPT");
        if(prompt) {
            string_obj* ret = STRING_NEW("");
            if(saphire_shell3(ret, prompt, "SAPHIRE_PROMPT") == -1) {
                buf = readline("> ");
            }
            else {
                buf = readline(string_c_str(ret));
            }

            string_delete(ret);
        }
        else {
            buf = readline("> ");
        }

        /// 終了コードの復帰 ///
        saphire_set_local_var("RCODE", end_code);

        /// 実行 ///
        if(buf == NULL) {
            break;
        }
        else if(*buf) {
            saphire_set_signal();
            sWFd* pipeout = sWFd_new(STDOUT_FILENO);
            sRFd* pipein = sRFd_new(STDIN_FILENO);
            int rcode = saphire_shell(buf, "saphire", pipeout, pipein, STDERR_FILENO);
            sWFd_flash(pipeout);
            sWFd_delete(pipeout);
            sRFd_delete(pipein);
            if(rcode < 0) {
                /// ヒストリに追加 ///
                add_history(buf);
                history_num++;
                if(history_num > histsize) {
                    HIST_ENTRY* history = remove_history(0);
                    free(history);
                }

                fprintf(stderr, "run time err\n");
                fprintf(stderr, "%s", string_c_str(gErrMsg));
            }
            else {
                /// ヒストリに追加 ///
                add_history(buf);
                history_num++;
                if(history_num > histsize) {
                    HIST_ENTRY* history = remove_history(0);
                    free(history);
                }
            }
        }

        /// バックグランド待ち ///
        saphire_wait_background_job();

        free(buf);

        if(gKitutukiExit >= 0) {
            break;        // exitコマンドが実行された
        }
    }

    readline_final();
    saphire_kill_all_jobs();
    saphire_final();
    mcurses_final();

    /// ヒストリーを書き込む ///
    if(histfile) {
        write_history(histfile);
    }
    
    mreset_tty();

    /// メモリリーク検出 ///
    CHECKML_END();
#endif

    return 0;
}

