/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.lite;

import net.morilib.lang.number.Rational;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public abstract class LispExactReal extends LispReal {

	/**
	 * 
	 * @return
	 */
	public abstract Rational toRational();

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#uminus()
	 */
	public abstract LispExactReal uminus();

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#toExact()
	 */
	public LispExactReal toExact() {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#isNaN()
	 */
	public boolean isNaN() {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#isInfinity()
	 */
	@Override
	public boolean isInfinity() {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.algebra.Addable#multiply(int)
	 */
	public LispReal multiply(int n) {
		return multiply(new LispSmallInt(n));
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#isExact()
	 */
	@Override
	public boolean isExact() {
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.algebra.Multipliable#power(int)
	 */
	public LispReal power(int n) {
		if(n < 0) {
			return power(-n).inv().getReal();
		} else if(n == 0) {
			return LispInteger.ONE;
		} else if(n == 1) {
			return this;
		} else {
			LispReal r = this;

			for(int i = 1; i < n; i++) {
				r = r.multiply(this);
			}
			return r;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#add(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber add(LispNumber x) {
		if(x instanceof LispDouble) {
			return new LispDouble(getRealDouble() + x.getRealDouble());
		} else if(x instanceof LispIrrational) {
			return new LispDouble(getRealDouble() + x.getRealDouble());
		} else if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return LispComplex.newComplex(
					add(c.getReal()), c.getImag());
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#sub(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispDouble) {
			return new LispDouble(getRealDouble() - x.getRealDouble());
		} else if(x instanceof LispIrrational) {
			return new LispDouble(getRealDouble() - x.getRealDouble());
		} else if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return LispComplex.newComplex(
					subtract(c.getReal()), c.getImag().uminus());
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#mul(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispDouble) {
			return new LispDouble(getRealDouble() * x.getRealDouble());
		} else if(x instanceof LispIrrational) {
			return new LispDouble(getRealDouble() * x.getRealDouble());
		} else if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return LispComplex.newComplex(
					multiply(c.getReal()), multiply(c.getImag()));
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#div(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber div(LispNumber x) {
		if(x instanceof LispDouble) {
			return new LispDouble(getRealDouble() / x.getRealDouble());
		} else if(x instanceof LispIrrational) {
			return new LispDouble(getRealDouble() / x.getRealDouble());
		} else if(x instanceof LispComplex) {
			LispReal xr = x.getReal();
			LispReal xi = x.getImag();
			LispReal xn = xr.multiply(xr).add(xi.multiply(xi));

			return LispComplex.newComplex(
					multiply(xr).divide(xn),
					multiply(xi).uminus().divide(xn));
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#remainder(net.morilib.lisp.LispReal)
	 */
	public LispReal remainder(LispReal r) {
		LispReal b;
		double x;

		if(!r.isExact()) {
			x = Math.IEEEremainder(getRealDouble(), r.getRealDouble());
			return new LispDouble(x);
		} else if(r.signum() > 0) {
			for(b = r; compareTo(r) > 0; b = b.subtract(r));
			return b;
		} else {
			for(b = r; compareTo(r) < 0; b = b.add(r));
			return b;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#toLispString(int, int)
	 */
	@Override
	public LispString toLispString(int radix, int precision) {
		return toLispString(radix);
	}

}
