# -*- coding: utf-8 -*-

# GUIのメインフレームにおけるSeed1特定Notebookのview
# 2013/12/15 written by kei9
import wx
from wx import xrc
import wx.grid

import constnumbers


class NoteBookSeed1View():
    u""" メインのフレームのSeed1特定タブ """
    SEPARATER = u"/"
    def __init__(self, frame):
        self.frame = frame
        self._init_view()
        self._init_event()

    def _init_view(self):
        # initialize view

        # buttons
        self.radiobox_tenun = xrc.XRCCTRL(self.frame, "RadioBoxTenun")
        self.button_search = xrc.XRCCTRL(self.frame, "ButtonSeed1Search")
        self.button_clear = xrc.XRCCTRL(self.frame, "ButtonSeed1Clear")

        self.spin_ctrl_amulet_num = xrc.XRCCTRL(self.frame, "SpinCtrlSeed1AmuletNum")

        # load combo boxes
        self.combobox_amulet_list = [xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet1"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet2"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet3"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet4"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet5"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet6"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Amulet7")]
        self.combobox_skill_list = [xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill1"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill2"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill3"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill4"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill5"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill6"),
                xrc.XRCCTRL(self.frame, "ComboBoxSeed1Skill7")]

        # text ctrl
        self.text_ctrl_result = xrc.XRCCTRL(self.frame, "TextCtrl1Seed1Result")

        # grid
        self.grid_seed1 = xrc.XRCCTRL(self.frame, "GridSeed1Table")
        u"""Tableを結果TextCtrlで示し、No.をrowラベル、行にはseed1, 天運個数、
        1-7枠目の値を表示する。1-7枠目をお守りの種類とスキル名で分けると面倒なので
        合体させて1つの行で表示した方がいいかもしれない。この場合は行数は9となる"""
        self.grid_seed1.CreateGrid(constnumbers.NUM_SEED1_GRID_ROW, constnumbers.NUM_SEED1_GRID_COL)
        self.grid_seed1.SetColLabelValue(0, constnumbers.LABEL_FORMAT_SEED1_GRID_COL1)
        self.grid_seed1.SetColLabelValue(1, constnumbers.LABEL_FORMAT_SEED1_GRID_COL2)
        for i in range(2, constnumbers.NUM_SEED1_GRID_COL):
            self.grid_seed1.SetColLabelValue(i, 
                    constnumbers.LABEL_FORMAT_SEED1_GRID_COL_AFTER2.format(i-1))
        self.grid_seed1.DisableDragGridSize()
        self.grid_seed1.EnableEditing(False)

        self._amulet_names = []
        self._amuletname2skillnames_dict = {}

    def set_amuletname_skillnames_dict(self, amulet_names, amuletname2skillnames_dict):
        u"""お守り名とそのお守りに対応する第1スキルの辞書をセットする"""
        self._amulet_names = amulet_names
        self._amuletname2skillnames_dict = amuletname2skillnames_dict
        for amu_combo, skill_combo in zip(self.combobox_amulet_list, self.combobox_skill_list):
            amu_combo.SetItems(self._amulet_names)
            amu_combo.SetSelection(0)
            skill_combo.SetItems(self._amuletname2skillnames_dict[self._amulet_names[0]])
            skill_combo.SetSelection(0)

    def _init_event(self):
        u""" イベントの登録 """
        self.spin_ctrl_amulet_num.Bind(wx.EVT_SPINCTRL, self.OnSpinCtrlChanged)
        self.grid_seed1.Bind(wx.EVT_SIZE, self.OnGridSizeChanged)
        for amu_combo in self.combobox_amulet_list:
            amu_combo.Bind(wx.EVT_COMBOBOX, self.OnAmuletComboChanged)

    def OnSpinCtrlChanged(self, event):
        u""" SpinCtrlの値が変わった時のイベント。
        設定できるコンボボックスの数を調節する """
        val = self.spin_ctrl_amulet_num.GetValue()
        for combo1, combo2 in zip(self.combobox_amulet_list[:val], self.combobox_skill_list[:val]):
            combo1.Enable()
            combo2.Enable()
        for combo1, combo2 in zip(self.combobox_amulet_list[val:], self.combobox_skill_list[val:]):
            combo1.Disable()
            combo2.Disable()

    def OnGridSizeChanged(self, event):
        u""" Gridサイズの変更時 """
        width, height = self.grid_seed1.GetParent().GetSize()
        each_width = (width - 60) / (constnumbers.NUM_SEED1_GRID_COL) # consider margin size
        self.grid_seed1.SetRowLabelSize(45)
        for col in range(constnumbers.NUM_SEED1_GRID_COL):
            self.grid_seed1.SetColSize(col, each_width) 

    def OnAmuletComboChanged(self, event):
        u""" お守りの種類を選択するコンボボックス変更時 """
        obj = event.GetEventObject()
        for amu_combo, skill_combo in zip(self.combobox_amulet_list, self.combobox_skill_list):
            if obj == amu_combo:
                amu_name = amu_combo.GetStringSelection()
                if amu_name in self._amuletname2skillnames_dict:
                    skill_combo.SetItems(self._amuletname2skillnames_dict[amu_name])
                    skill_combo.SetSelection(0)
                break

    def bind_button_search(self, event_func, evt=wx.EVT_BUTTON):
        u""" 検索ボタンへのイベント登録 """
        self.button_search.Bind(evt, event_func)

    def bind_button_clear(self, event_func, evt=wx.EVT_BUTTON):
        u""" クリアボタンへのイベント登録 """
        self.button_clear.Bind(evt, event_func)

    def get_tenun_radio_key(self):
        u""" 天運の錬金で使用したお守りの種類を示すkeyを返す """
        idx = self.radiobox_tenun.GetSelection()
        if idx == 0:
            return constnumbers.KEY_TENUN555
        elif idx == 1:
            return constnumbers.KEY_TENUN888
        else:
            return None

    def get_selected_amulets_and_names(self):
        u"""選択されているお守りの種類名とスキル1の名前のペアをリストとして返す"""
        result_num = self.spin_ctrl_amulet_num.GetValue()
        result_list = []
        for i, amu_combo, skill_combo in zip(
                range(result_num), self.combobox_amulet_list, self.combobox_skill_list):
            amulet_name = amu_combo.GetStringSelection()
            skill_name = skill_combo.GetStringSelection()
            result_list.append((amulet_name, skill_name))
        return result_list

    def clear_highlight(self):
        u""" セルの強調表示をクリアする """
        for row in range(self.grid_seed1.GetNumberRows()):
            for col in range(self.grid_seed1.GetNumberCols()):
                self.grid_seed1.SetCellBackgroundColour(row, col, wx.NullColor)

    def update_highlight(self, skill_names, color=u"Yellow"):
        u""" 指定されたスキル名に一致するセルを強調表示する """
        for row in range(self.grid_seed1.GetNumberRows()):
            for col in range(2, self.grid_seed1.GetNumberCols()):
                val = self.grid_seed1.GetCellValue(row, col)
                try:
                    amu_name, skill_name = val.split(self.SEPARATER)
                    if skill_name in skill_names:
                        self.grid_seed1.SetCellBackgroundColour(row, col, color)
                    else:
                        self.grid_seed1.SetCellBackgroundColour(row, col, wx.NullColor)
                except ValueError as e:
                    #cannot unpack
                    self.grid_seed1.SetCellBackgroundColour(row, col, wx.NullColor)

    def set_seed1_dict(self, seed1_dict, select_seed1=None):
        u""" seed1から(no, table_no, result_num, amulet_names, skill_names)への辞書を元に、
        noを列ラベルとする表を作成する
        select_seed1がNoneでなければ、対応する行を選択状態とする"""
        row_num = len(seed1_dict)
        diff = row_num - self.grid_seed1.GetNumberRows()
        if diff > 0:
            self.grid_seed1.AppendRows(diff, False)
        elif diff < 0:
            self.grid_seed1.DeleteRows(0, -diff, False)

        for row, seed1 in enumerate(
                sorted(seed1_dict.keys(), key=lambda x: seed1_dict[x][0])): # order by no
            no, table_no, result_num, amu_names, skill_names = seed1_dict[seed1]
            self.grid_seed1.SetRowLabelValue(row, u"{0}".format(no))
            self.grid_seed1.SetCellValue(row, 0, "{0}".format(seed1))
            self.grid_seed1.SetCellValue(row, 1, "{0}".format(result_num))
            for col, amu_name, skill_name in zip(
                    range(2, constnumbers.NUM_SEED1_GRID_COL), amu_names, skill_names):
                if amu_name is not None and skill_name is not None:
                    val = u"{0}{1}{2}".format(
                        constnumbers.AMULET_NAME_TO_SHORT_NAME_DICT[amu_name], self.SEPARATER, skill_name)
                    self.grid_seed1.SetCellValue(row, col, val)
                else:
                    self.grid_seed1.SetCellValue(row, col, u"")
            if seed1 == select_seed1:
                self.grid_seed1.MakeCellVisible(row, 0)
                self.grid_seed1.SelectRow(row)

    def clear_combobox(self):
        u""" comboboxの選択されたインデックスを0にする """
        for amu_combo, skill_combo in zip(self.combobox_amulet_list, self.combobox_skill_list):
            amu_combo.SetSelection(0)
            skill_combo.SetSelection(0)

    def clear_seed1_grid(self):
        u""" 表の値をクリアする """
        for row in range(self.grid_seed1.GetNumberRows()):
            self.grid_seed1.SetRowLabelValue(row, u"")
            for col in range(self.grid_seed1.GetNumberCols()):
                self.grid_seed1.SetCellValue(row, col, u"")
        self.clear_highlight()

    def clear_text_result(self):
        u""" 結果を示すTextCtrlから値のクリア """
        self.text_ctrl_result.SetValue(u"")

    def set_text_result(self, value):
        u""" 結果を示すTextCtrlへの値のセット """
        self.text_ctrl_result.SetValue(value)

