/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.expr;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import net.morilib.awk.AwkLocation;
import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.namespace.AwkClosureNamespace;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.value.AwkFunction;
import net.morilib.awk.value.AwkUserFunction;
import net.morilib.awk.value.AwkValue;

/**
 * クロージャを割り当てる前の関数を定義します。
 *
 * @author MORIGUCHI, Yuichiro 2013/03/08
 */
public class AwkFunctionPrototype extends AwkExpression {

	private AwkExpression function;
	private List<String> argnames;
	private AwkLocation name;
	private transient AwkFunction memo = null;

	/**
	 * 中間表現を生成します。
	 * 
	 * @param name     場所
	 * @param function 関数の記述の表現
	 * @param argnames 引数の名前
	 */
	public AwkFunctionPrototype(AwkLocation name,
			AwkExpression function,
			Collection<String> argnames) {
		this.name     = name;
		this.function = function;
		this.argnames = new ArrayList<String>(argnames);
	}

	/**
	 * 無名関数の中間表現を生成します。
	 * 
	 * @param function 関数の記述の表現
	 * @param argnames 引数の名前
	 */
	public AwkFunctionPrototype(AwkExpression function,
			Collection<String> argnames) {
		this.name     = null;
		this.function = function;
		this.argnames = new ArrayList<String>(argnames);
	}

	/**
	 * 関数の表現を得ます。
	 */
	public AwkExpression getFunction() {
		return function;
	}

	/**
	 * 引数の名称を取得します。
	 */
	public List<String> getArgnames() {
		return Collections.unmodifiableList(argnames);
	}

	/**
	 * 関数名を取得します。
	 */
	public String getName() {
		return name.getName();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#eval(net.morilib.awk.AwkNamespace, net.morilib.awk.io.AwkFiles)
	 */
	@Override
	public AwkValue eval(AwkNamespace ns, AwkFiles f) {
		if(memo == null) {
			memo = new AwkUserFunction(
					name != null ? name.getName() : "<anonymous>",
					new AwkClosureNamespace(ns),
					function,
					argnames);
		}
		return memo;
	}

	/**
	 * 名前空間に関数を定義します。
	 * 
	 * @param ns 名前空間
	 * @param useClosure クロージャを使用するときtrue
	 */
	public void defun(AwkNamespace ns, boolean useClosure) {
		AwkNamespace ev, s2 = ns;

		for(int i = 0; i < name.sizeOfPath() - 1; i++) {
			s2 = s2.subnamespace(name.getPathOf(i));
		}
		ev = useClosure ? new AwkClosureNamespace(s2) : null;
		s2.bindFunction(name.getName(), new AwkUserFunction(this, ev));
	}

}
