/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.io.Reader;

import net.morilib.sh.misc.ConsoleInputStream;

/**
 * 改行したときにプロンプトを表示するReaderです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class ShPromptReader extends Reader {

	//
	private int unread = -1;
	private int lineno = 1;
	private ShEnvironment env;
	private String prompt;
	private boolean isno = false;
	private boolean iseof = false;
	private Reader reader;
	private PrintStream promptout;
	int count = 0;

	/**
	 * 
	 * @param env
	 * @param promptout
	 */
	public ShPromptReader(ShEnvironment env, PrintStream promptout) {
		this(ConsoleInputStream.console(), env, promptout);
		env.bind("LINENO", Integer.toString(lineno));
	}

	/**
	 * 
	 * @param ins
	 * @param env
	 * @param promptout
	 */
	public ShPromptReader(InputStream ins, ShEnvironment env,
			PrintStream promptout) {
		this.env = env;
		this.reader = new InputStreamReader(ins);
		this.promptout = promptout;
		promptprint(env.find("PS1"));
		env.bind("LINENO", Integer.toString(lineno));
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.misc.PushbackLineNumberReadable#getLineNumber()
	 */
	public int getLineNumber() {
		return lineno;
	}

	/* (non-Javadoc)
	 * @see java.io.Closeable#close()
	 */
	public void close() throws IOException {
		// do nothing
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.misc.PushbackLineNumberReadable#read(char[], int, int)
	 */
	public int read(char[] cbuf, int off, int len) throws IOException {
		int c;

		for(int i = off; i < off + len; i++) {
			if((c = read()) < 0) {
				return i - off;
			} else {
				cbuf[i] = (char)c;
			}
		}
		return len;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.misc.PushbackLineNumberReadable#read(char[])
	 */
	public int read(char[] cbuf) throws IOException {
		return read(cbuf, 0, cbuf.length);
	}

	private int _read() throws IOException {
		return reader.read();
	}

	private void promptprint(String prompt) {
		if(promptout != null)  promptout.print(prompt);
	}

	private void inclineno() {
		env.bind("LINENO", Integer.toString(lineno++));
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.misc.PushbackLineNumberReadable#read()
	 */
	public int read() throws IOException {
		int c;

		if(isno) {
			if(count > 0) {
				promptprint(prompt);
			} else {
				return -1;
			}
		}

		isno = false;
		if(unread >= 0) {
			c = unread;
			unread = -1;
			return c;
		} else if((c = _read()) < 0) {
			iseof = true;
			return -1;
		} else if(c == '\r') {
			if((c = _read()) >= 0 && c != '\n')  unread = c;
			isno = true;
			if(count > 0) {
				inclineno();
				prompt = env.find("PS2");
				return c;
			} else {
				inclineno();
				return -1;
			}
		} else if(c != '\n') {
			return c;
		} else if(count > 0) {
			inclineno();
			prompt = env.find("PS2");
			isno = true;
			return c;
		} else {
			inclineno();
			isno = true;
			return -1;
		}
	}

	/**
	 * 
	 */
	public void resetPrompt() {
		isno   = false;
		prompt = env.find("PS1");
		promptprint(prompt);
	}

	/**
	 * 
	 * @return
	 */
	public boolean isEof() {
		return iseof;
	}

}
