/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.bytes;

import java.util.Collection;

/**
 * バイトに関するコレクションのインターフェースです。
 * 
 * @author MORIGUCHI, Yuichiro 2010/10/11
 */
public interface ByteCollection extends Collection<Byte> {

	/**
	 * バイトを追加します。
	 * 
	 * @param v バイト
	 * @return 追加されたときtrue
	 */
	public boolean addByte(byte v);

	/**
	 * バイトを追加します。
	 * 
	 * @param v バイト
	 * @return 追加されたときtrue
	 * @throws IllegalArgumentException 引数が0～255の外にあるとき
	 */
	public boolean add(int v);

	/**
	 * 与えられたコレクションのすべての要素を追加します。
	 * 
	 * @param a Byteコレクション
	 * @return 追加されたときtrue
	 */
	public boolean addAllByte(ByteCollection a);

	/**
	 * 与えられたコレクションのすべての要素を追加します。
	 * 
	 * @param as Byteコレクション
	 * @return 追加されたときtrue
	 */
	public boolean addAllByte(ByteCollection... as);

	/**
	 * 与えられたコレクションのすべての要素を追加します。
	 * 
	 * @param as Byteコレクションのコレクション
	 * @return 追加されたときtrue
	 */
	public boolean addAllByte(
			Collection<? extends ByteCollection> as);

	/**
	 * コレクションをクリアします。
	 */
	public void clear();

	/**
	 * バイトを含むときtrueを得ます。
	 * 
	 * @param v テストする値
	 * @return バイトを含むときtrue
	 */
	public boolean containsByte(byte v);

	/**
	 * バイトを含むときtrueを得ます。
	 * 
	 * @param v テストする値
	 * @return バイトを含むときtrue
	 * @throws IllegalArgumentException 引数が0～255の外にあるとき
	 */
	public boolean contains(int v);

	/**
	 * 与えられたコレクションの要素をすべて含むときtrueを得る。
	 * 
	 * @param a テストするコレクション
	 * @return コレクションを含むときtrue
	 */
	public boolean containsAllByte(ByteCollection a);

	public boolean isEmpty();

	/**
	 * バイトのイテレータを得ます。
	 * 
	 * @return イテレータ
	 */
	public ByteIterator byteIterator();

	/**
	 * 引数のバイトを削除します。
	 * 
	 * @param v 削除するバイト
	 * @return 削除されたときtrue
	 */
	public boolean removeByte(byte v);

	/**
	 * 引数のバイトを削除します。
	 * 
	 * @param v 削除するバイト
	 * @return 削除されたときtrue
	 */
	public boolean removeElement(int v);

	/**
	 * 引数のコレクションにあるすべてのバイトを削除します。
	 * 
	 * @param a コレクション
	 * @return 削除されたときtrue
	 */
	public boolean removeAllByte(ByteCollection a);

	/**
	 * 引数のコレクションにないすべてのバイトを削除します。
	 * 
	 * @param a コレクション
	 * @return 削除されたときtrue
	 */
	public boolean retainAllByte(ByteCollection a);

	public int size();

	/**
	 * byteの配列に変換します。
	 * 
	 * @return 配列
	 */
	public byte[] toByteArray();

	/**
	 * byteの配列に変換します。
	 * 
	 * @param a 上書きする配列
	 * @return 配列
	 */
	public byte[] toByteArray(byte[] a);

	/**
	 * コレクションが無限集合のときtrueを得ます。
	 * 
	 * @return 無限集合のときtrue
	 */
	public boolean isInfinite();

	/**
	 * コレクションの重複を取り除き、集合に変換します。
	 * 
	 * @return 集合
	 */
	public ByteSet toSet();

}
