/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.notes.store;

import java.io.*;
import java.text.ParseException;
import java.util.*;

import jp.co.connectone.common.PropertyHandlerBaseImpl;
import jp.co.connectone.eai.notes.user.NotesAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.store.*;
import lotus.domino.*;
import jp.co.connectone.log.Log;
public abstract class EAINotesBase
{
	protected abstract String getElmentName();
	public IRecordObject getFolderIndexFromString(String path) throws IncorrectData
	{
		if (path==null) path=getElmentName();
		BasicRecordObject rc = new BasicRecordObject();
		rc.setOid(new NotesFolderIndex(path));
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("IFolderIndex",rc.getOid());
		try {
			rc.setFieldSet(h);
		}
		catch (IllegalAccessException e)
		{
			Log.error("getFolderIndexFromString:error on setting IFolderIndex",e);
			throw new IncorrectData("getFolderIndexFromString:error on setting IFolderIndex");
		}
		return rc;
	}
	
	protected static Properties prop;

	private String getIORFromServer(String strMailSvr)throws IncorrectData
	{
		String l_strIor = null;
		String l_strURI1 = null;
		String l_strURI2 = null;
Log.debug("getIORFromServer:strMailSvr="+strMailSvr);
		if (strMailSvr.indexOf(":")>0) {
			l_strURI1 = strMailSvr;
		}
		else {
			l_strURI1 = strMailSvr + ":80";
			l_strURI2 = strMailSvr + ":8080";
		}
Log.debug("l_strURI1="+l_strURI1);
Log.debug("l_strURI2="+l_strURI2);
		try
		{
			l_strIor = NotesFactory.getIOR(l_strURI1);
		}
        catch(NotesException e)
		{
        	if (l_strURI2!=null) {
        		try {
        			l_strIor = NotesFactory.getIOR(l_strURI2);
        		}
                catch(NotesException l_notesExObj) {
        			throw new IncorrectData(l_notesExObj.id + l_notesExObj.text);
                }
        	}
		}
		return l_strIor;
	}

	private String getIORFromFile(String strMailServer)throws DataNotFound
	{
		String l_strIor = null;

		try
		{
			String strIorFile = prop.getProperty("diiop");
			strIorFile = strIorFile + strMailServer + NotesConstants.FILE_TYPE ;
Log.debug("IOR_filename:"+strIorFile);
			FileInputStream l_fileInStreamObj = new FileInputStream(strIorFile);
			InputStreamReader l_inStremReaderObj = new InputStreamReader(l_fileInStreamObj);
			BufferedReader l_bufReaderObj = new BufferedReader(l_inStremReaderObj);
			l_strIor = l_bufReaderObj.readLine();
			l_fileInStreamObj.close();
		}
		catch(Exception l_fexcepObj)
		{
			throw new DataNotFound(l_fexcepObj.getMessage());
		}

		return l_strIor;
	}

	private void _init(String strUser, String strPassword, String strMailServer, String strMailServer2) throws NoSuchRights, ServerDown, DataNotFound, IncorrectData, HandleException
 	{
Log.debug("_init:strUser="+strUser);// delete password output on log
		if (prop == null) {
			try {
				prop = PropertyHandlerBaseImpl.getInstance().getProperties("jp.co.connectone.eai.notes");
			}
			catch (Exception e) {
				Log.error("",e);
			}
		}
		String l_strIor = null;
		try {
			l_strIor = this.getIORFromFile(strMailServer);
		}
		catch (DataNotFound e) {
			//do nothing
		}
		catch (Exception e) {
			Log.error("",e);
			throw new HandleException("SYSTEM ERROR");
		}
		if (l_strIor==null) {
			try {
				l_strIor = this.getIORFromServer(strMailServer);
			}
			catch (IncorrectData ie) {
				//do nothing.
			}
		}
		if (l_strIor==null) {
			Log.warn("server "+strMailServer+" not respond. try "+strMailServer2+".");
			try {
				l_strIor = this.getIORFromServer(strMailServer2);
			}
			catch (IncorrectData ie) {
				//do nothing.
			}
		}
Log.debug("_init:l_strIor="+l_strIor);
		if (l_strIor==null) {
			throw new ServerDown("could not get IOR from server: assume server is down");
		}
		try {
			   	m_sessionObj = NotesFactory.createSessionWithIOR(l_strIor, strUser, strPassword);
Log.trace("_init:userName="+m_sessionObj.getUserName());
Log.trace("_init:commonUserName="+m_sessionObj.getCommonUserName());
			   	m_dbDirObj = m_sessionObj.getDbDirectory(null);
		}
		catch(NotesException ne) {
			int errCode = ne.id;
			String msg = ne.getMessage();
			this.recycleObjects();
			throw new NoSuchRights("(" + errCode + ")" + msg);
		}
	}

	protected void initSession(NotesAccountData acc) throws NoSuchRights, ServerDown, DataNotFound, IncorrectData, HandleException
	{
		String strUser = acc.getUserID();
		String strPassword = acc.getInternetPassword();
		NotesServiceInfo info =  (NotesServiceInfo)acc.getServiceInfo();
		String strMailServer =info.getServerAddress();
		String strMailServer2 =info.getServerAddress2();
		String strMailDb = info.getDatabase();

Log.debug("initSession:user="+strUser+":server="+strMailServer+":server2="+strMailServer2+":db="+strMailDb);
		_init(strUser,strPassword,strMailServer,strMailServer2);
		try {
			if (strMailDb==null) {
		        m_dbObj = m_dbDirObj.openMailDatabase();
			}
			else {
		        m_dbObj = m_dbDirObj.openDatabase(strMailDb);
			}
	    }
		catch (NotesException l_notesExObj)
		{
Log.debug("failed to open mail database via openDatabase()/openMailDatabase()");
			throw new NoSuchRights(l_notesExObj.getMessage());
		}
	}

	protected void initSession(NotesAccountData acc, IDatabaseIndex database)throws NoSuchRights, ServerDown, DataNotFound, IncorrectData, HandleException
	{
		String strUser = acc.getUserID();
		String strPassword = acc.getInternetPassword();
		NotesServiceInfo info =  (NotesServiceInfo)acc.getServiceInfo();
		String strMailServer =info.getServerAddress();
		String strMailServer2 =info.getServerAddress2();
		String strDBName = database.getIndex().toString();
Log.debug("initSession:user="+strUser+":server="+strMailServer+":server2="+strMailServer2+":DB="+strDBName);
		_init(strUser,strPassword,strMailServer,strMailServer2);
		try {
			if (strDBName==null) {
		        m_dbObj = m_dbDirObj.openMailDatabase();
			}
			else {
		        m_dbObj = m_dbDirObj.openDatabase(strDBName);
			}
	    }
		catch (NotesException l_notesExObj)
		{
			Log.error("filed to Open Database "+strDBName,l_notesExObj);
			if (l_notesExObj.id == 4003) {
				throw new StoreNotFound("Store '"+ strDBName + "' not found.");
			}
			throw new NoSuchRights(l_notesExObj.getMessage());
		}
	}

	protected DateTime getOffsetDateTime(int offsetDays) throws ServerDown , HandleException
	{
		if (m_sessionObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		DateTime l_dateTimeObj = null;
	    try
		{
	        l_dateTimeObj = m_sessionObj.createDateTime(NotesConstants.DATE_TODAY);
	        l_dateTimeObj.adjustDay((-1)*offsetDays);
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
		return l_dateTimeObj;
	}

	protected DocumentCollection search(String query,DateTime datetime) throws ServerDown, HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		DocumentCollection col = null;
		try {
			col = m_dbObj.search(query,datetime);
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
	    return col;
	}

	protected DateTime createDateTime(String dateStr) throws NullPointerException,ParseException
	{
		if (m_sessionObj==null) {
			throw new NullPointerException("EAINotesBase.initSession() must be called first.");
		}
		if (dateStr==null) {
			throw new NullPointerException("dateStr must not be null.");
		}
		DateTime l_dateTimeObj = null;
	    try
		{
	        l_dateTimeObj = m_sessionObj.createDateTime(dateStr);
		}
	    catch(NotesException l_excepObj){
	    	throw new ParseException("date string format couldn't parsed by Notes Session Object",0);
	    }
		return l_dateTimeObj;
	}

	protected DateTime createDateTime(Date date) throws NullPointerException
	{
		if (m_sessionObj==null) {
			throw new NullPointerException("EAINotesBase.initSession() must be called first.");
		}
		if (date==null) {
			throw new NullPointerException("date must not be null.");
		}
		DateTime l_dateTimeObj = null;
	    try
		{
	        l_dateTimeObj = m_sessionObj.createDateTime(date);
		}
	    catch(NotesException l_excepObj)
		{
		}
		return l_dateTimeObj;
	}

	protected DateRange createDateRange(Date startDate,Date endDate) throws ServerDown , HandleException
	{
		if (m_sessionObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		DateTime l_dateTimeStartObj = null;
		DateTime l_dateTimeEndObj = null;
		DateRange l_dateRangeObj = null;
	    try
		{
	    	l_dateTimeStartObj = m_sessionObj.createDateTime(startDate);
	    	l_dateTimeEndObj = m_sessionObj.createDateTime(endDate);
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
	    try
		{
            l_dateRangeObj = m_sessionObj.createDateRange(l_dateTimeStartObj, l_dateTimeEndObj);
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
	    finally {
	    	try {
	    		if (l_dateTimeStartObj!=null) {
	    			l_dateTimeStartObj.recycle();
	    		}
	    		if (l_dateTimeEndObj!=null) {
	    			l_dateTimeEndObj.recycle();
	    		}
	    	}
	    	catch (NotesException e) {
	    		// error level WARN
	    	}
	    }
	    return l_dateRangeObj;
	}

	protected DateRange createDateRange(String datePoint,int offsetDays,boolean isPointStartDate) throws ServerDown , HandleException
	{
		if (m_sessionObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		DateTime l_dateTimeStartObj = null;
		DateTime l_dateTimeEndObj = null;
		DateRange l_dateRangeObj = null;
	    try
		{
	    	l_dateTimeStartObj = m_sessionObj.createDateTime(datePoint);
	    	l_dateTimeEndObj = m_sessionObj.createDateTime(datePoint);
		    if (isPointStartDate) {
		    	l_dateTimeEndObj.adjustDay(offsetDays);
		    }
		    else {
		    	l_dateTimeStartObj.adjustDay((-1)*offsetDays);
		    }
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
	    try
		{
            l_dateRangeObj = m_sessionObj.createDateRange(l_dateTimeStartObj, l_dateTimeEndObj);
		}
	    catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
	    finally {
	    	try {
	    		if (l_dateTimeStartObj!=null) {
	    			l_dateTimeStartObj.recycle();
	    		}
	    		if (l_dateTimeEndObj!=null) {
	    			l_dateTimeEndObj.recycle();
	    		}
	    	}
	    	catch (NotesException e) {
	    		// error level WARN
	    	}
	    }
	    return l_dateRangeObj;
	}

	protected Document CreateDocument() throws DataNotFound, HandleException
	{
		if (m_dbObj == null)
		{
			throw new HandleException(
					"EAINotesBase.initSession() must be called first.");
		}
		Document d = null;
		try
		{
			d = m_dbObj.createDocument();
		} catch (NotesException l_notesExcepObj)
		{
			throw new DataNotFound(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		return d;
	}

	@SuppressWarnings("unchecked")
	protected Vector<View> getViews() throws ServerDown,HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		Vector<View> v = null;
		try {
			v = m_dbObj.getViews();
		}
		catch (NotesException e) {
			//ERROR level
	    	throw new ServerDown(e.id + e.text);
		}
		return v;
	}

	protected View getView(IFolderIndex folder) throws ServerDown,HandleException
	{
		String viewName = folder.getIndex().toString();
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		View v = null;
		try {
			v = m_dbObj.getView(viewName);
		}
		catch (NotesException e) {
			Log.error("NotesException on getView",e);
	    	throw new ServerDown(e.id + e.text);
		}
		return v;
	}
	
	protected String getUNID(IObjectIndex idx)
	{
		return (String)idx.getIndex();
	}

	protected Document getDocument(IObjectIndex idx) throws DataNotFound,HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		Document d = null;
		try {
			d = m_dbObj.getDocumentByUNID(getUNID(idx));
		}
		catch (NotesException l_notesExcepObj)
		{
			throw new DataNotFound(l_notesExcepObj.id + l_notesExcepObj.text);
	    }
		return d;
	}

	protected void deleteDocument(IObjectIndex idx) throws DataNotFound,HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		Document d = null;
		String unid = getUNID(idx);
		try {
			d = m_dbObj.getDocumentByUNID(unid);
			d.remove(true);
			d.recycle();
			d=null;
			Log.debug("document "+unid+" deleted");
		}
		catch (NotesException l_notesExcepObj) {
			throw new DataNotFound(l_notesExcepObj.id + l_notesExcepObj.text);
	    }
	}

	protected Document getDocumentByUNID(IObjectIndex strReferenceUID) throws DataNotFound,HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		Document d = null;
		try {
			d = m_dbObj.getDocumentByUNID((String)strReferenceUID.getIndex());
		}
		catch (NotesException l_notesExcepObj)
		{
			throw new DataNotFound(l_notesExcepObj.id + l_notesExcepObj.text);
	    }
		return d;
	}

	protected Document createDocument() throws NoSuchRights,HandleException
	{
		if (m_dbObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		Document d = null;
		try {
			d = m_dbObj.createDocument();
		}
		catch (NotesException l_notesExcepObj)
		{
			throw new NoSuchRights(l_notesExcepObj.id + l_notesExcepObj.text);
	    }
		return d;
	}

	protected void recycleObjects() throws ServerDown
	{
		try {
			if(m_dbObj !=null) {
				m_dbObj.recycle();
			}
			if(m_dbDirObj !=null) {
				m_dbDirObj.recycle();
			}
			if(m_sessionObj !=null) {
				m_sessionObj.recycle();
			}
		}
		catch(NotesException l_exceptionObj) {
			throw new ServerDown(l_exceptionObj.id + l_exceptionObj.text);
		}
	}
	
	@SuppressWarnings("unchecked")
	protected void updateFieldsFromNotesDBFieldSet(Document doc,IRecordObject objRecord) throws HandleException
	{
		HashMap<String,Object> fieldSet = null;
		try {
			fieldSet = objRecord.getFieldSet();
		}
		catch (Exception e) {
			Log.error("",e);
			throw new HandleException(e.getMessage());
		}
		Vector<Item> items = null;
		try {
			items = doc.getItems();
		}
		catch(NotesException l_exceptionObj){
			throw new HandleException(l_exceptionObj.id + l_exceptionObj.text);
		}
		Enumeration<Item> en = items.elements();
		while (en.hasMoreElements()) {
			Item item = en.nextElement();
			try {
				String name = item.getName();
				String strValue = item.getValueString();
Log.trace("Item ("+name+":"+strValue+")");
				String name_ignoreCase = name.toUpperCase();
				Object o = fieldSet.get(name_ignoreCase);
				if (o != null) {
					if (o instanceof Date) {
						Date d = (Date)o;
						item.setDateTimeValue(createDateTime(d));
					}
					else if (o instanceof String) {
						String s = (String)o;
						item.setValueString(s);
					}
					else if (o instanceof Integer) {
						Integer itg = (Integer)o;
						int i = itg.intValue();
						item.setValueInteger(i);
					}
					else if (o instanceof Double) {
						Double dbl = (Double)o;
						double d = dbl.doubleValue();
						item.setValueDouble(d);
					}
				}
				strValue = item.getValueString();
Log.trace("changed ("+name+":"+strValue+")");
			}
			catch(NotesException l_exceptionObj){
				throw new HandleException(l_exceptionObj.id + l_exceptionObj.text);
			}
		}
		
	}
	@SuppressWarnings("unchecked")
	protected HashMap<String,Object> populateHashMap(Document doc) throws HandleException
	{
		HashMap<String,Object> rc = new HashMap<String,Object>();
		Vector<Item> items = null;
		try {
			items = doc.getItems();
		}
		catch(NotesException l_exceptionObj)
		{
			throw new HandleException(l_exceptionObj.id + l_exceptionObj.text);
		}
		Enumeration<Item> en = items.elements();
		while (en.hasMoreElements()) {
			Item item = en.nextElement();
			try {
				String name = item.getName();
				Object value = null;
		          switch (item.getType()) {
		            case Item.ERRORITEM :
		            case Item.UNAVAILABLE :
		            case Item.UNKNOWN :
		            	throw new HandleException("Notes item returned ERRITEM/UNAVAILABLE/UNKNOWN on getType");
		            case Item.ATTACHMENT ://attachment=file=byte array
		            case Item.EMBEDDEDOBJECT ://object=byte array?
		            case Item.ICON ://icon=file=byte array?
		            case Item.OTHEROBJECT ://object=byte array?
		            	break;
		            case Item.ASSISTANTINFO ://String
		            case Item.AUTHORS ://String
		            case Item.FORMULA ://String
		            case Item.HTML ://String
		            case Item.LSOBJECT ://String
		            case Item.NAMES ://String
		            case Item.NOTELINKS ://String
		            case Item.NOTEREFS ://String
		            case Item.RICHTEXT ://String
		            case Item.READERS ://String
		            case Item.SIGNATURE ://String
		            case Item.TEXT ://String
		            case Item.USERID ://String
		            	value = item.getValueString();
		            	break;
		            case Item.DATETIMES ://Date
		            	try {
			            	value = item.getDateTimeValue().toJavaDate();
		            	}
		            	catch (Exception e) {
		            		Log.trace("DATETIME conversion error:"+name);
		            	}
		            	break;
		            case Item.NUMBERS ://integer or double
		            	int in = item.getValueInteger();
		            	double dn = item.getValueDouble();
Log.trace("integer number="+in+".double number="+dn);
						if (in == dn) {
			            	value = new Integer(in);
						}
						else {
							value = new Double(dn);
						}
		            	break;
		            case Item.USERDATA :
		            	break;
		            case Item.VIEWMAPDATA :
		            	break;
		            case Item.VIEWMAPLAYOUT :
		            	break;
		            case Item.ACTIONCD :
		            	break;
		            case Item.COLLATION :
		            	break;
		            case Item.QUERYCD :
		            	break;
		            default :
		            }
				rc.put(name,value);
			}
			catch(NotesException l_exceptionObj)
			{
				throw new HandleException(l_exceptionObj.id + l_exceptionObj.text);
			}
		}
		return rc;
	}
	protected String getUserID(NotesAccountData acc)
	{
		if (acc==null) return null;
		return acc.getUserID();
	}
	protected String getInternetPassword(NotesAccountData acc)
	{
		if (acc==null) return null;
		return acc.getInternetPassword();
	}
	protected String getServerAddress(NotesAccountData acc)
	{
		if (acc==null) return null;
		return acc.getServiceInfo().getServerAddress();
	}
	
	protected String getCommonUserNameFromSession() throws HandleException
	{
		String commonUserName="";
		if (m_sessionObj==null) {
			throw new HandleException("EAINotesBase.initSession() must be called first.");
		}
		
		try {
			commonUserName = m_sessionObj.getCommonUserName();
		}
		catch(NotesException l_excepObj)
		{
	    	throw new ServerDown(l_excepObj.id + l_excepObj.text);
		}
      	return commonUserName;
	}

	protected String getAccountStringFromFullname(String src)
	{
		String notesId="";
      	if (src.startsWith("CN=")) {
      		int atIndx = src.indexOf('@');
      		if (atIndx != -1) src = src.substring(0, atIndx);
      		// Parse for "/"
         	StringTokenizer sParser = new StringTokenizer(src,"/");
         	String sPart="";
         	// Rebuild the Id
        	while (sParser.hasMoreTokens()) {
	            sPart = sParser.nextToken();
//	            if (sPart.indexOf("CN=") != -1) fnln = sPart;
//	            if (sPart.indexOf("O=") != -1) nDomain = sPart;
	         	sPart = sPart.substring(sPart.indexOf("=")+1);
            	if (sParser.hasMoreTokens()) {
              		notesId = notesId+sPart+"/";
            	}
            	else {
                	//internet name or alias
                	notesId = notesId+sPart;
            	}
          	}
      	}
      	if (notesId.length()==0) return src;
      	return notesId;
	}

	protected String getFilePath() throws HandleException
	{
		String dbName = null;
		try {
			dbName = m_dbObj.getFilePath();
		}
		catch(NotesException ne)
		{
			throw new HandleException(ne.id + ne.text);
		}
		return dbName;
	}

	private Session m_sessionObj = null;
	private DbDirectory m_dbDirObj = null;
	private Database m_dbObj = null;
}
