package jp.cssj.cti2;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;

import jp.cssj.cti2.message.MessageHandler;
import jp.cssj.cti2.progress.ProgressListener;
import jp.cssj.cti2.results.Results;
import jp.cssj.resolver.MetaSource;
import jp.cssj.resolver.Source;
import jp.cssj.resolver.SourceResolver;

/**
 * ドキュメント変換処理を実行するためのサーバーとの接続です。
 * 
 * @author <a href="mailto:miyabe at gnn.co.jp">MIYABE Tatsuhiko </a>
 * @version $Id: CTISession.java 526 2011-01-19 02:36:52Z miyabe $
 */
public interface CTISession {
	/** きりのよいところまで処理する中断処理の定数です。abortメソッドに渡します。 */
	public static final byte ABORT_NORMAL = 1;

	/** 強制的に中断する処理の定数です。abortメソッドに渡します。 */
	public static final byte ABORT_FORCE = 2;

	/**
	 * サーバー情報を返します。
	 * 
	 * @param uri
	 *            サーバー情報を選択するためのURI。
	 * @return サーバー情報データのストリーム。
	 */
	public InputStream getServerInfo(URI uri) throws IOException;

	/**
	 * <p>
	 * 出力先を設定します。
	 * </p>
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * 
	 * @param results
	 *            出力先。
	 */
	public void setResults(Results results) throws IOException;

	/**
	 * <p>
	 * メッセージを受け取るためのオブジェクトを設定します。
	 * </p>
	 * 
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * 
	 * @see MessageHandler
	 * @param messageHandler
	 *            メッセージハンドラ
	 */
	public void setMessageHandler(MessageHandler messageHandler)
			throws IOException;

	/**
	 * <p>
	 * 進行状況を監視するためのオブジェクトを設定します。
	 * </p>
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * <p>
	 * これは、データの変換状況を監視する目的以外に、出力結果の長さを取得するためにも用いられます。
	 * HTTP通信などでContent-Lengthヘッダを送出する場合は重要です。
	 * </p>
	 * 
	 * @see ProgressListener
	 * @param progressListener
	 *            進行状況リスナ
	 */
	public void setProgressListener(ProgressListener progressListener)
			throws IOException;

	/**
	 * <p>
	 * プロパティを設定します。
	 * </p>
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * 
	 * @param name
	 *            プロパティ名
	 * @param value
	 *            値
	 * @throws IOException
	 */
	public void property(String name, String value) throws IOException;

	/**
	 * <p>
	 * リソースを送信するための出力ストリームを返します。
	 * </p>
	 * <p>
	 * <strong>リソースを送信した後、出力ストリームは必ずクローズしてください。 </strong>
	 * </p>
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * 
	 * @param metaSource
	 *            リソースデータのメタ情報。
	 * @return サーバーへの出力ストリーム。
	 * @throws IOException
	 */
	public OutputStream resource(MetaSource metaSource) throws IOException;

	/**
	 * <p>
	 * リソースを送信します。
	 * </p>
	 * <p>
	 * このメソッドは各transcodeメソッドの前に呼ぶ必要があります。
	 * </p>
	 * 
	 * @param source
	 *            リソースのデータソース。
	 * @throws IOException
	 */
	public void resource(Source source) throws IOException;

	/**
	 * <p>
	 * リソースを読み込むためのオブジェクトを設定します。
	 * </p>
	 * 
	 * @param resolver
	 *            サーバー側から要求したリソースを取得するためのSourceResolver。
	 */
	public void setSourceResolver(SourceResolver resolver) throws IOException;

	/**
	 * <p>
	 * メインドキュメントを送信するための出力ストリームを返します。
	 * </p>
	 * <p>
	 * <strong>本体を送信した後、出力ストリームは必ずクローズしてください。 </strong>
	 * </p>
	 * 
	 * @param metaSource
	 *            メインドキュメントのメタ情報。
	 * @return サーバーへの出力ストリーム。
	 * @throws IOException
	 */
	public OutputStream transcode(MetaSource metaSource) throws IOException;

	/**
	 * <p>
	 * 指定されたアドレスへサーバー側からアクセスしてメインドキュメントを取得して変換します。
	 * </p>
	 * 
	 * @param uri
	 *            メインドキュメントのURI。
	 * @throws IOException
	 */
	public void transcode(URI uri) throws IOException, TranscoderException;

	/**
	 * <p>
	 * メインドキュメントをデータソースから取得して変換します。
	 * </p>
	 * 
	 * @param source
	 *            メインドキュメントのデータソース。
	 * @throws IOException
	 */
	public void transcode(Source source) throws IOException,
			TranscoderException;

	/**
	 * <p>
	 * 複数の結果を結合するモードに切り替えます。
	 * </p>
	 * 
	 * @param continuous
	 *            trueであればjoinにより結果を結合するモードにします。
	 * @throws IOException
	 */
	public void setContinuous(boolean continuous) throws IOException;

	/**
	 * <p>
	 * 結果を結合して出力します。
	 * </p>
	 * 
	 * @throws IOException
	 */
	public void join() throws IOException;

	/**
	 * <p>
	 * 変換を中断します。 変換済みのページが含まれた出力結果が得られます。
	 * </p>
	 * 
	 * @param mode
	 *            きりのよいところまで出力する場合はABORT_NORMAL、強制的に処理を停止するにはABORT_FORCEを指定します。
	 * @throws IOException
	 */
	public void abort(byte mode) throws IOException;

	/**
	 * <p>
	 * 送られたリソースと、プロパティ、メッセージハンドラ等の全ての設定をクリアして初期状態に戻します。
	 * </p>
	 * 
	 * @throws IOException
	 */
	public void reset() throws IOException;

	/**
	 * <p>
	 * セッションをクローズします。
	 * <p>
	 * 
	 * このメソッドを呼び出した後は、セッションに対して何も出来ません。
	 * 
	 * @throws IOException
	 */
	public void close() throws IOException;
}