package jp.cssj.cr.epub;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.net.URLDecoder;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.cssj.print.epub.ArchiveFile;
import jp.cssj.resolver.Source;
import jp.cssj.resolver.SourceResolver;
import jp.cssj.resolver.SourceValidity;
import jp.cssj.resolver.helpers.AbstractSource;
import jp.cssj.resolver.helpers.ValidSourceValidity;
import jp.cssj.resolver.zip.ZipFileSource;

/**
 * ZIPファイルをデータ源とするSourceResolverです。
 * 
 * @author <a href="mailto:tatsuhiko at miya.be">MIYABE Tatsuhiko </a>
 * @version $Id: FileSourceResolver.java 324 2009-08-02 05:51:25Z miyabe $
 */
public class ArchiveFileSourceResolver implements SourceResolver {
	protected final ArchiveFile archive;

	public ArchiveFileSourceResolver(ArchiveFile archive) {
		this.archive = archive;
	}

	public Source resolve(URI uri) throws IOException {
		ArchiveFileSource source = new ArchiveFileSource(this.archive, uri);
		return source;
	}

	public void release(Source source) {
		((ArchiveFileSource) source).close();
	}
}

class ArchiveFileSource extends AbstractSource {
	private static final Logger LOG = Logger.getLogger(ZipFileSource.class
			.getName());

	private final ArchiveFile archive;

	private final String path;

	private final String encoding;

	private String mimeType = null;

	public ArchiveFileSource(ArchiveFile archive, URI uri, String mimeType,
			String encoding) {
		super(uri);
		if (archive == null) {
			throw new NullPointerException();
		}
		String path = uri.getSchemeSpecificPart();
		try {
			path = URLDecoder.decode(path, "UTF-8");
		} catch (Exception e) {
			LOG.log(Level.WARNING, "URIをデコードできません。", e);
		}
		this.path = path;
		this.archive = archive;
		this.mimeType = mimeType;
		this.encoding = encoding;
	}

	public ArchiveFileSource(ArchiveFile archive, URI uri, String mimeType) {
		this(archive, uri, mimeType, null);
	}

	public ArchiveFileSource(ArchiveFile archive, URI uri) throws IOException {
		this(archive, uri, null);
	}

	public String getMimeType() throws IOException {
		if (this.mimeType == null) {
			String name = this.path;
			int dot = name.indexOf('.');
			if (dot != -1) {
				String suffix = name.substring(dot, name.length());
				if (suffix.equalsIgnoreCase(".html")
						|| suffix.equalsIgnoreCase(".htm")) {
					this.mimeType = "text/html";
				} else if (suffix.equalsIgnoreCase(".xml")
						|| suffix.equalsIgnoreCase(".xhtml")
						|| suffix.equalsIgnoreCase(".xht")) {
					this.mimeType = "text/xml";
				}
			}
		}
		return this.mimeType;
	}

	public String getEncoding() {
		return this.encoding;
	}

	public boolean exists() throws IOException {
		return this.archive.exists(this.path);
	}

	public boolean isFile() throws IOException {
		return false;
	}

	public boolean isInputStream() throws IOException {
		return true;
	}

	public boolean isReader() throws IOException {
		return this.encoding != null;
	}

	public InputStream getInputStream() throws IOException {
		return this.archive.getInputStream(this.path);
	}

	public Reader getReader() throws IOException {
		if (!this.isReader()) {
			throw new UnsupportedOperationException();
		}
		return new InputStreamReader(this.getInputStream(), this.encoding);
	}

	public File getFile() {
		throw new UnsupportedOperationException();
	}

	public long getLength() throws IOException {
		if (this.exists()) {
			return -1;
		}
		return 0;
	}

	public SourceValidity getValidity() throws IOException {
		return ValidSourceValidity.SHARED_INSTANCE;
	}
}