//  Copyright (c) 2013 NAO + dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Jun-4/2013.
//

#include "ndngesturevision.h"

// Opencv includes.
#include <opencv2/opencv.hpp>

#include <iostream>
#include <string>
#include <limits>
#include <vector>
#include <set>

NDNGestureVision::NDNGestureVision(int imageWidth, int imageHeight, int scanningStep, float minFlowThreshold, int numOfFlows)
    : m_imageWidth(imageWidth), m_imageHeight(imageHeight), m_scanningStep(scanningStep), m_numOfFlows(numOfFlows), m_drawProcessView(true)
{
    assert(imageWidth > 0 && imageHeight > 0 && scanningStep > 0);

    m_scanningXMax = imageWidth / scanningStep;
    m_scanningYMax = imageHeight / scanningStep;
    m_mapSize = m_scanningXMax * m_scanningYMax;
    m_map = cv::Mat(m_scanningYMax, m_scanningXMax, CV_8U);
    m_minFlowThresholdP2 = minFlowThreshold * minFlowThreshold;
    m_tempSortedFlowNums = new int[numOfFlows * 2];
    m_tempSortedFlowExtents = new int[numOfFlows * 2];

    m_backgroundImage = cv::Mat::zeros(cv::Size(imageWidth, imageHeight), CV_8UC3);
    m_backgroundImageAge = cv::Mat::zeros(cv::Size(imageWidth, imageHeight), CV_8U);
    m_trackingImage = cv::Mat::zeros(cv::Size(imageWidth, imageHeight), CV_8UC3);

    // TODO
    // Setting parameters here.
    // It may be better to find the mechanism to adjust them automatically.
    m_minExt = 5;
    m_minFocusUpdateSearchRange = m_imageWidth / 8;
    m_fousedRegionMatchingGoodScore = 0.95f;
    m_fousedRegionMatchingAcceptScore = 0.85f;
    m_backgroundDiffThreshold = 50;
    m_backgroundMaxAge = 64;
    m_backgroundUpdateAgeThreshold = 32;
    m_backgroundAgeToAddForMovingPart = 4;
    m_backgroundAgeToSubForChangedPart = 2;
    m_forgroundNoiseThreashold = m_imageWidth / 100;
    m_focusedRegionMaxAge = 20;
    m_focusedRegionInitAge = 10;
}

NDNGestureVision::~NDNGestureVision()
{
    if (m_tempSortedFlowNums)
    {
        delete [] m_tempSortedFlowNums;
    }
    if (m_tempSortedFlowExtents)
    {
        delete [] m_tempSortedFlowExtents;
    }
}

const NDNGestureVision::FlowRegion* NDNGestureVision::getFlow(int index) const
{
    if (index != 0)
    {
        printf("NDNGestureVision::getFlow() only support index 0 now..\n");
    }
    return &m_focusedFlowRegion;
}

void NDNGestureVision::calculateFlow(const cv::Mat &newImage, cv::Mat *pGestureVisionDrawMat)
{
    cv::Mat newGrayScaleImage;
    cv::cvtColor(newImage, newGrayScaleImage, CV_BGR2GRAY);

    if( m_prevGray.data )
    {
        /*-------------------
         * Find flow regions
         */
        cv::Mat flowMat;
        cv::calcOpticalFlowFarneback(m_prevGray, newGrayScaleImage, flowMat, 0.5, 3, 15, 3, 5, 1.2, 0);
        m_map = cv::Mat::zeros(m_map.size(),CV_8U);

        std::vector<FlowRegion> flowRegions;
        FlowRegion dummy;
        flowRegions.push_back(dummy);

        int nextRegionNum = 1;
        for(int y = 0; y < m_scanningYMax; y++)
        {
            int x = 0;
            while(x < m_scanningXMax)
            {
                const cv::Point2f& fxy = flowMat.at<cv::Point2f>(y * m_scanningStep, x * m_scanningStep);

                if (fxy.x * fxy.x + fxy.y * fxy.y <= m_minFlowThresholdP2)
                {
                    x++;
                }
                else
                {
                    int extent = 1;
                    float dxTotal = fxy.x;
                    float dyTotal = fxy.y;
                    std::set<int> attachingRegions;
                    int upperY = y - 1;
                    if (upperY >= 0)
                    {
                        if (m_map.at<uchar>(upperY, x)  > 0)
                        {
                            attachingRegions.insert(m_map.at<uchar>(upperY,x));
                        }
                    }
                    int ry = y * m_scanningStep;
                    int sp = x;
                    int xp = x + 1;
                    int rx = xp * m_scanningStep;
                    while (xp < m_scanningXMax)
                    {
                        const cv::Point2f& rfxy = flowMat.at<cv::Point2f>(ry, rx);
                        if (rfxy.dot(fxy) < 0 || rfxy.x * rfxy.x + rfxy.y * rfxy.y < m_minFlowThresholdP2)
                            break;

                        dxTotal += rfxy.x;
                        dyTotal += rfxy.y;
                        extent++;
                        if (upperY >= 0 && m_map.at<uchar>(upperY, xp) > 0)
                        {
                            attachingRegions.insert(m_map.at<uchar>(upperY, xp));
                        }
                        xp++;
                        rx += m_scanningStep;
                    }
                    int ep = xp;

                    cv::Point2f direction;
                    direction.x = dxTotal / extent;
                    direction.y = dyTotal / extent;
                    float distance = sqrt(direction.x *direction.x + direction.y * direction.y);
                    direction.x = direction.x / distance;
                    direction.y = direction.y / distance;

                    std::set<int>::iterator it = attachingRegions.begin();
                    int mapNum = 0;
                    float maxR = 0;
                    while(it != attachingRegions.end())
                    {
                        float r = flowRegions[*it].direction.dot(direction);
                        if (r > 0 && maxR < r)
                        {
                            maxR = r;
                            mapNum = *it;
                        }
                        ++it;
                    }

                    int regionNum = 0;
                    if (maxR > 0)
                    {
                        regionNum = mapNum;
                        float rate = ((float)extent) / ((float)(flowRegions[regionNum].extent + extent));
                        direction.x = flowRegions[regionNum].direction.x * (1.0f - rate) + direction.x * rate;
                        direction.y = flowRegions[regionNum].direction.y * (1.0f - rate) + direction.y * rate;
                        float adj = 1.0f / sqrt(direction.x * direction.x + direction.y * direction.y);
                        flowRegions[regionNum].direction.x = direction.x * adj;
                        flowRegions[regionNum].direction.y = direction.y * adj;
                        flowRegions[regionNum].distance = flowRegions[regionNum].distance * (1.0f -rate) + distance * rate;
                        flowRegions[regionNum].extent = flowRegions[regionNum].extent + extent;
                        if (sp * m_scanningStep < flowRegions[regionNum].xmin)
                        {
                            flowRegions[regionNum].xmin = sp * m_scanningStep;
                        }
                        if (flowRegions[regionNum].xmax < (ep - 1) * m_scanningStep)
                        {
                            flowRegions[regionNum].xmax = (ep - 1) * m_scanningStep;
                        }
                        flowRegions[regionNum].ymax = y * m_scanningStep;
                    }
                    else
                    {
                        FlowRegion flowRegion;
                        flowRegion.extent = extent;
                        flowRegion.direction = direction;
                        flowRegion.distance = distance;
                        flowRegion.xmin = sp * m_scanningStep;
                        flowRegion.xmax = (ep - 1) * m_scanningStep;
                        flowRegion.ymin = y * m_scanningStep;
                        flowRegion.ymax = flowRegion.ymin;

                        flowRegions.push_back(flowRegion);
                        regionNum = nextRegionNum;
                        nextRegionNum++;

                        assert(nextRegionNum == flowRegions.size());
                    }

                    for (int i = sp; i < ep; i++)
                    {
                        m_map.at<uchar>(y, i) = regionNum;
                    }
                    x = ep;
                }
            }
        }

        for (int i = 0; i < m_numOfFlows * 2; i++)
        {
            m_tempSortedFlowNums[i] = 0;
            m_tempSortedFlowExtents[i] = 0;
        }

        int sortedFlowSizeMax = nextRegionNum <= m_numOfFlows * 2 ? nextRegionNum - 1 : m_numOfFlows * 2;

        /*-------------------
         * Sort flowRegions with its extents and put the region numbers into m_tempSortedFlowNums[], m_tempSortedFlowExtents[]
         * The size of the list is stored in sortedFlowSize.
         */
        int cnt = 0;
        for (int i = 1 ; i < nextRegionNum; i++)
        {
            if (flowRegions[i].extent <= m_minExt)
                continue;

            for (int j = 0; j < sortedFlowSizeMax; j++)
            {
                if (flowRegions[i].extent > m_tempSortedFlowExtents[j])
                {
                    for (int k = sortedFlowSizeMax - 2; k >= j; k--)
                    {
                        m_tempSortedFlowNums[k + 1] = m_tempSortedFlowNums[k];
                        m_tempSortedFlowExtents[k + 1] = m_tempSortedFlowExtents[k];
                    }
                    m_tempSortedFlowNums[j] = i;
                    m_tempSortedFlowExtents[j] = flowRegions[i].extent;
                    cnt++;
                    break;
                }
            }
        }
        int sortedFlowSize = sortedFlowSizeMax < cnt ? sortedFlowSizeMax : cnt;

        /*-------------------
         * Calculate center of the regions.
         */
        if (sortedFlowSize > 0)
        {
           flowRegions[m_tempSortedFlowNums[0]].calculateCenter();
            for (int i = 0; i < sortedFlowSize; i++)
            {
                flowRegions[m_tempSortedFlowNums[0]].calculateCenter();
            }
        }

        /*-------------------
         * Check if the flow is made by global move like camera movement.
         */
        bool isGlobalMove = false;
        if (sortedFlowSize > 0 && flowRegions[m_tempSortedFlowNums[0]].extent > m_mapSize / 3)
        {
            //global move.
            isGlobalMove = true;
        }


        /*-------------------
         * Update focusing region.
         */
        cv::Mat foregroundImg;
        cv::vector<cv::Point> fingures;
        bool focusedRegionMoving = false;
        if (m_focusedFlowRegion.age > 0)
        {
            int expectedFlowDirectionX = 0;
            int expectedFlowDirectionY = 0;

            int trackRegionX0 = m_focusedFlowRegion.trackRegionX0 + m_focusedFlowRegion.xmin;
            int trackRegionY0 = m_focusedFlowRegion.trackRegionY0 + m_focusedFlowRegion.ymin;
            int trackImageWidth = m_focusedFlowRegion.trackRegionX1 - m_focusedFlowRegion.trackRegionX0 + 1;
            int trackImageHeight = m_focusedFlowRegion.trackRegionY1 - m_focusedFlowRegion.trackRegionY0 + 1;

            int yrange = trackImageHeight*2;
            if (yrange < m_minFocusUpdateSearchRange) yrange = m_minFocusUpdateSearchRange;

            int xrange = trackImageWidth*2;
            if (xrange < m_minFocusUpdateSearchRange) xrange = m_minFocusUpdateSearchRange;

            int range = yrange < xrange ? xrange : yrange;

            // find out the flow region which matches to the focused region and make estimation of the movement.
            cv::Point focusPointPrev = m_focusedFlowRegion.getCenter();
            if (sortedFlowSize > 0)
            {
                int idx = 0;
                cv::Point c = flowRegions[m_tempSortedFlowNums[0]].getCenter();
                int dmin = (c.x - focusPointPrev.x) * (c.x - focusPointPrev.x) + (c.y - focusPointPrev.y) * (c.y - focusPointPrev.y);
                for (int i = 1; i < sortedFlowSize; i++)
                {
                    c = flowRegions[m_tempSortedFlowNums[i]].getCenter();
                    int d = (c.x - focusPointPrev.x) * (c.x - focusPointPrev.x) + (c.y - focusPointPrev.y) * (c.y - focusPointPrev.y);
                    if (d < dmin)
                    {
                        dmin = d;
                        idx = i;
                    }
                }

                if (dmin < range * range)
                {
                    cv::Point2f direction = flowRegions[m_tempSortedFlowNums[idx]].direction;

                    if (direction.y != 0)
                        expectedFlowDirectionY = direction.y < 0 ? -1 : 1;

                    if (direction.x != 0)
                        expectedFlowDirectionX = direction.x < 0 ? -1 : 1;

                    focusedRegionMoving = true;
                }
            }


            // Look around current tracking area and find where the focused part is moved now.
            int maxScore = 255 * trackImageHeight * trackImageWidth * 2;
            float highScore = compareTrackingImage(trackRegionX0, trackRegionY0, trackImageWidth, trackImageHeight, newImage) / maxScore;
            int hx = trackRegionX0;
            int hy = trackRegionY0;
            float score = 0;
            int step = range / 10;

            for (int r = 1 ; r < range; r += step)
            {
                int sx = -range;
                int ex = range;
                int sy = -range + step;
                int ey = range - step;

                if (expectedFlowDirectionX > 0)
                    sx = 0;
                else if (expectedFlowDirectionX < 0)
                    ex = 0;

                if (expectedFlowDirectionY > 0)
                    sy = 0;
                else if (expectedFlowDirectionY < 0)
                    ey = 0;

                for (int y = sy; y <= ey; y += step)
                {
                    if (expectedFlowDirectionX <= 0)
                    {
                        score = compareTrackingImage(trackRegionX0 - r, trackRegionY0 + y, trackImageWidth, trackImageHeight, newImage) / maxScore;
                        if (score > highScore)
                        {
                            highScore = score;
                            hx = trackRegionX0 - r;
                            hy = trackRegionY0 + y;
                        }
                    }

                    if (expectedFlowDirectionX >= 0)
                    {
                        score = compareTrackingImage(trackRegionX0 + r, trackRegionY0 + y, trackImageWidth, trackImageHeight, newImage) / maxScore;
                        if (score > highScore)
                        {
                            highScore = score;
                            hx = trackRegionX0 + r;
                            hy = trackRegionY0 + y;
                        }
                    }
                }

                for (int x = sx; x <= ex; x += step)
                {
                    if (expectedFlowDirectionY <= 0)
                    {
                        score = compareTrackingImage(trackRegionX0 + x, trackRegionY0 - r, trackImageWidth, trackImageHeight, newImage) / maxScore;
                        if (score > highScore)
                        {
                            highScore = score;
                            hx = trackRegionX0 + x;
                            hy = trackRegionY0 - r;
                        }
                    }

                    if (expectedFlowDirectionY >= 0)
                    {
                        score = compareTrackingImage(trackRegionX0 + x, trackRegionY0 + r, trackImageWidth, trackImageHeight, newImage) / maxScore;
                        if (score > highScore)
                        {
                            highScore = score;
                            hx = trackRegionX0 + x;
                            hy = trackRegionY0 + r;
                        }
                    }
                }


                if (highScore > m_fousedRegionMatchingGoodScore)
                {
                    break;
                }
            }

            //debug
            {
//                printf("score:%.2f\n", highScore);
            }

            if (highScore > m_fousedRegionMatchingAcceptScore)
            {
                int dx = hx - trackRegionX0;
                int dy = hy - trackRegionY0;
                float distance = sqrt((float) dx*dx + dy*dy);
                if (distance > 0)
                {
                    m_focusedFlowRegion.direction.x = (float)dx / distance;
                    m_focusedFlowRegion.direction.y = (float)dy / distance;
                }
                else
                {
                    m_focusedFlowRegion.direction.x = 0;
                    m_focusedFlowRegion.direction.y = 0;
                }
                m_focusedFlowRegion.prevDistance = m_focusedFlowRegion.distance;
                m_focusedFlowRegion.distance = distance;
                m_focusedFlowRegion.xmin += dx;
                m_focusedFlowRegion.xmax += dx;
                m_focusedFlowRegion.ymin += dy;
                m_focusedFlowRegion.ymax += dy;
                m_focusedFlowRegion.calculateCenter();
            }
            else
            {
                if (m_focusedFlowRegion.age > 0)
                    m_focusedFlowRegion.age --;
                m_focusedFlowRegion.direction.x = 0;
                m_focusedFlowRegion.direction.y = 0;
                m_focusedFlowRegion.prevDistance = m_focusedFlowRegion.distance;
                m_focusedFlowRegion.distance = 0;
            }
        }

        if (!focusedRegionMoving)
        {
            if (!isGlobalMove)
            {
                // Create foreground image and find hand shape from it.
                // OpenCV 1.x APIs are used here. Somehow, OpenCV 2.x APIs equivalent to them caused some assert failure
                // when its executed on Windows (Tested on 2.4.6 and 2.3.1)
                //
                IplImage *foreground = cvCreateImage(m_backgroundImage.size(), IPL_DEPTH_8U, 1);
                cvSet(foreground, cvScalar(0));
                CvMemStorage *storage = cvCreateMemStorage(0);

                if (m_focusedFlowRegion.age > 0)
                {
                    int wr = (m_focusedFlowRegion.xmax - m_focusedFlowRegion.xmin)/3;
                    int hr = (m_focusedFlowRegion.ymax - m_focusedFlowRegion.ymin)/3;
                    int x0 = m_focusedFlowRegion.xmin - wr;
                    if (x0 < 0) x0 = 0;
                    int x1 = m_focusedFlowRegion.xmax + wr;
                    if (x1 >= m_imageWidth) x1 = m_imageWidth - 1;
                    int y0 = m_focusedFlowRegion.ymin - hr;
                    if (y0 < 0) y0 = 0;
                    int y1 = m_focusedFlowRegion.ymax + hr;
                    if (y1 >= m_imageHeight) y1 = m_imageHeight - 1;
                    paintForeground(foreground, newImage, x0, y0, x1, y1);
                }
                else
                {
                    for (int i = 0; i < sortedFlowSize; i++)
                    {
                        int x0 = flowRegions[m_tempSortedFlowNums[i]].xmin - m_scanningStep;
                        if (x0 < 0) x0 = 0;
                        int x1 = flowRegions[m_tempSortedFlowNums[i]].xmax + m_scanningStep;
                        if (x1 >= m_imageWidth) x1 = m_imageWidth - 1;
                        int y0 = flowRegions[m_tempSortedFlowNums[i]].ymin - m_scanningStep;
                        if (y0 < 0) y0 = 0;
                        int y1 = flowRegions[m_tempSortedFlowNums[i]].ymax + m_scanningStep;
                        if (y1 >= m_imageHeight) y1 = m_imageHeight - 1;
                        paintForeground(foreground, newImage, x0, y0, x1, y1);
                    }
                }

                if (pGestureVisionDrawMat)
                {
                    foregroundImg = foreground;
                    cv::resize(foregroundImg, foregroundImg, cv::Size(foregroundImg.size().width/2, foregroundImg.size().height/2));
                }

                CvSeq *contours = NULL;
                cvFindContours(foreground, storage, &contours, sizeof(CvContour), CV_RETR_EXTERNAL , CV_CHAIN_APPROX_SIMPLE);
                bool fingureFound = false;
                for(CvSeq* c = contours; c != NULL ; c = c->h_next )
                {
                    cv::Rect handRect = findFingures(c, fingures);

                    int fingureRootY = m_imageHeight;
                    if (fingures.size() > 0)
                    {
                        for (unsigned int j = 0; j < fingures.size(); j+= 2)
                        {
                            cv::Point p1 = fingures[j];
                            if (p1.y < fingureRootY)
                            {
                                fingureRootY = p1.y;
                            }
                        }
                        int s = fingures[6].x - fingures[2].x;
                        if (s > 0)
                        {
                            int hx = fingures[2].x;
                            cv::Rect trackRect(hx, fingureRootY, s, s);

                            m_trackingImage = cv::Mat::zeros(m_trackingImage.size(), CV_8UC3);

                            int fy = 0;
                            for (int y = trackRect.y ; y < trackRect.y + trackRect.height  && y < m_imageHeight; y++)
                            {
                                int fx = 0;
                                for (int x = trackRect.x ; x < trackRect.x + trackRect.width && x < m_imageWidth; x++)
                                {
                                    m_trackingImage.at<cv::Vec3b>(fy,fx) = newImage.at<cv::Vec3b>(y,x);
                                    fx++;
                                }
                                fy++;
                            }

                            fingureFound = true;
                            if (m_focusedFlowRegion.age < m_focusedRegionInitAge)
                                m_focusedFlowRegion.age = m_focusedRegionInitAge;
                            else if (m_focusedFlowRegion.age < m_focusedRegionMaxAge)
                                m_focusedFlowRegion.age++;

                            m_focusedFlowRegion.xmin = handRect.x;
                            m_focusedFlowRegion.xmax = handRect.x + handRect.width;
                            m_focusedFlowRegion.ymin = handRect.y;
                            m_focusedFlowRegion.ymax = handRect.y + handRect.height;
                            m_focusedFlowRegion.trackRegionX0 = trackRect.x - handRect.x;
                            m_focusedFlowRegion.trackRegionX1 = trackRect.width + m_focusedFlowRegion.trackRegionX0 - 1;
                            m_focusedFlowRegion.trackRegionY0 = trackRect.y - handRect.y;
                            m_focusedFlowRegion.trackRegionY1 = trackRect.height + m_focusedFlowRegion.trackRegionY0 - 1;
                            m_focusedFlowRegion.direction.x = 0;
                            m_focusedFlowRegion.direction.y = 0;
                            m_focusedFlowRegion.distance = 0;
                            m_focusedFlowRegion.prevDistance = 0;
                            m_focusedFlowRegion.extent = 0;
                            m_focusedFlowRegion.calculateCenter();

                        }
                        break;
                    }

                }
                if (!fingureFound)
                {
                    if (m_focusedFlowRegion.age > 0)
                        m_focusedFlowRegion.age--;
                }
                cvReleaseMemStorage(&storage);
                cvReleaseImage(&foreground);
            }
        }

        /* -------------------
         * Update background image
         */
        if (isGlobalMove)
        {
            //global move. Update background
            if (m_focusedFlowRegion.age > 0)
            {
                float dx = m_focusedFlowRegion.direction.x * m_focusedFlowRegion.distance;
                float dy = m_focusedFlowRegion.direction.y * m_focusedFlowRegion.distance;
                int xmin_prev = m_focusedFlowRegion.xmin - dx;
                if (xmin_prev < 0) xmin_prev = 0;
                else if (xmin_prev >= m_imageWidth) xmin_prev = m_imageWidth - 1;
                int width_prev = m_focusedFlowRegion.xmax - m_focusedFlowRegion.xmin;
                if (width_prev + xmin_prev > m_imageWidth )
                {
                    width_prev = m_imageWidth - xmin_prev;
                }
                int ymin_prev = m_focusedFlowRegion.ymin - dy;
                if (ymin_prev < 0) ymin_prev = 0;
                else if (ymin_prev >= m_imageHeight) ymin_prev = m_imageHeight - 1;
                int height_prev = m_focusedFlowRegion.ymax - m_focusedFlowRegion.ymin;
                if (height_prev + ymin_prev > m_imageHeight)
                {
                    height_prev = m_imageHeight - ymin_prev;
                }

                if (width_prev > 0 && height_prev > 0)
                {
                    cv::Rect prevRect = cv::Rect(xmin_prev, ymin_prev, width_prev, height_prev);

                    cv::Mat tempMat(m_backgroundImage.size(), CV_8UC3);
                    cv::Mat tempROIMat(tempMat, prevRect);
                    m_backgroundImage.copyTo(tempROIMat);

                    for (int y = 0; y < m_imageHeight; y++)
                    {
                        int py = y - dy;
                        if (m_focusedFlowRegion.ymin > y || m_focusedFlowRegion.ymax < y || py < 0 || py >= m_imageHeight)
                        {
                            for (int x = 0; x < m_imageWidth; x++)
                            {
                                m_backgroundImage.at<cv::Vec3b>(y,x) = newImage.at<cv::Vec3b>(y,x);
                                m_backgroundImageAge.at<uchar>(y,x) = 1;
                            }
                        }
                        else
                        {
                            for (int x = 0; x < m_imageWidth; x++)
                            {
                                if (m_focusedFlowRegion.xmin == x)
                                {
                                    int xx = x;
                                    for (; xx < m_focusedFlowRegion.xmax && xx < m_imageWidth; xx++)
                                    {
                                        int px = xx -dx;
                                        if (px >= 0 && px < m_imageWidth)
                                            m_backgroundImage.at<cv::Vec3b>(y,xx) = tempROIMat.at<cv::Vec3b>(py, px);
                                    }
                                    x = xx - 1;
                                }
                                else
                                {
                                    m_backgroundImage.at<cv::Vec3b>(y,x) = newImage.at<cv::Vec3b>(y,x);
                                    m_backgroundImageAge.at<uchar>(y,x) = 1;
                                }
                            }
                        }
                    }
                }
            }
            else
            {
                newImage.copyTo(m_backgroundImage);
                m_backgroundImageAge = cv::Mat::ones(m_backgroundImageAge.size(), CV_8U);
            }
        }
        else
        {
            int stillPointCnt = 0;
            int stillPointChangedPCnt = 0;
            int stillPointChangePSum = 0;
            for (int my = 0; my < m_scanningYMax; my++)
            {
                for (int mx = 0; mx < m_scanningXMax; mx++)
                {
                    if (m_map.at<uchar>(my,mx) > 0)
                        continue;

                    int ix = mx * m_scanningStep;
                    int iy = my * m_scanningStep;

                    stillPointCnt++;
                    int d = calcColorDiff(m_backgroundImage.at<cv::Vec3b>(iy,ix), newImage.at<cv::Vec3b>(iy,ix));
                    if (d > m_backgroundDiffThreshold)
                    {
                        stillPointChangedPCnt++;
                        stillPointChangePSum += d;
                    }
                }
            }

            if (stillPointChangedPCnt * 2 > stillPointCnt)
            {
                m_backgroundImageAge = cv::Mat::zeros(m_backgroundImageAge.size(), CV_8U);
            }
        }

        for (int my = 0; my < m_scanningYMax; my++)
        {
            for (int mx = 0; mx < m_scanningXMax; mx++)
            {

                int ix = mx * m_scanningStep;
                int iy = my * m_scanningStep;

                bool isInBackground = true;

                if (m_focusedFlowRegion.age > 0
                        && m_focusedFlowRegion.ymin <= iy && m_focusedFlowRegion.ymax >= iy
                        && m_focusedFlowRegion.xmin <= ix && m_focusedFlowRegion.xmax >= ix)
                {
                    mx = (m_focusedFlowRegion.xmax / m_scanningStep) + 1;
                    int ex = mx * m_scanningStep;
                    if (ex > m_imageWidth - 1)
                    {
                        ex = m_imageWidth - 1;
                    }
                    isInBackground = false;

                    for (int y = iy; y < iy + m_scanningStep; y++)
                    {
                        for (int x = ix; x < ex; x++)
                        {
                            int h = m_backgroundImageAge.at<uchar>(y,x);
                            if (h < m_backgroundMaxAge)
                            {
                                h += m_backgroundAgeToAddForMovingPart;
                                m_backgroundImageAge.at<uchar>(y,x) = h;
                            }
                        }
                    }
                }

                if (!isInBackground)
                {
                    continue;
                }

                for (int i = 0; i < sortedFlowSize; i++)
                {
                    if (flowRegions[m_tempSortedFlowNums[i]].ymin <= iy && flowRegions[m_tempSortedFlowNums[i]].ymax >= iy
                            && flowRegions[m_tempSortedFlowNums[i]].xmin <= ix && flowRegions[m_tempSortedFlowNums[i]].xmax >= ix)
                    {
                        mx = (flowRegions[m_tempSortedFlowNums[i]].xmax / m_scanningStep) + 1;
                        int ex = mx * m_scanningStep;
                        isInBackground = false;

                        for (int y = iy; y < iy + m_scanningStep; y++)
                        {
                            for (int x = ix; x < ex; x++)
                            {
                                int h = m_backgroundImageAge.at<uchar>(y,x);
                                if (h < m_backgroundMaxAge)
                                {
                                    h += m_backgroundAgeToAddForMovingPart;
                                    m_backgroundImageAge.at<uchar>(y,x) = h;
                                }
                            }
                        }
                        break;
                    }
                }

                if (!isInBackground)
                {
                    continue;
                }

                if (m_map.at<uchar>(my,mx) > 0)
                {
                    for (int y = iy; y < iy + m_scanningStep; y++)
                    {
                        for (int x = ix; x < ix + m_scanningStep; x++)
                        {
                            int h = m_backgroundImageAge.at<uchar>(y,x);
                            if (h < m_backgroundMaxAge)
                            {
                                h += m_backgroundAgeToAddForMovingPart;
                                m_backgroundImageAge.at<uchar>(y,x) = h;
                            }
                        }
                    }
                    continue;
                }

                for (int y = iy; y < iy + m_scanningStep; y++)
                {
                    for (int x = ix; x < ix + m_scanningStep; x++)
                    {
                        int h = m_backgroundImageAge.at<uchar>(y,x);
                        if (calcColorDiff(m_backgroundImage.at<cv::Vec3b>(y,x), newImage.at<cv::Vec3b>(y,x)) > m_backgroundDiffThreshold)
                        {
                            if (h > m_backgroundAgeToSubForChangedPart)
                            {
                                h -= m_backgroundAgeToSubForChangedPart;
                                m_backgroundImageAge.at<uchar>(y,x) = h;
                            }
                            else
                            {
                                h = 1;
                                m_backgroundImageAge.at<uchar>(y,x) = 1;
                            }
                        }
                        else
                        {
                            if (h < m_backgroundMaxAge)
                            {
                                h++;
                                m_backgroundImageAge.at<uchar>(y,x) = h;
                            }
                        }
                        if (h < m_backgroundUpdateAgeThreshold)
                        {
                            m_backgroundImage.at<cv::Vec3b>(y,x)[0] = (m_backgroundImage.at<cv::Vec3b>(y,x)[0] * h + newImage.at<cv::Vec3b>(y,x)[0]) / (h+1);
                            m_backgroundImage.at<cv::Vec3b>(y,x)[1] = (m_backgroundImage.at<cv::Vec3b>(y,x)[1] * h + newImage.at<cv::Vec3b>(y,x)[1]) / (h+1);
                            m_backgroundImage.at<cv::Vec3b>(y,x)[2] = (m_backgroundImage.at<cv::Vec3b>(y,x)[2] * h + newImage.at<cv::Vec3b>(y,x)[2]) / (h+1);
                        }
                    }
                }
            }
        }

        /*-------------------
         * Draw output mat
         */
        if (pGestureVisionDrawMat)
        {
            if (m_drawProcessView)
            {
                int xs = pGestureVisionDrawMat->size().width / 4;
                int ys = pGestureVisionDrawMat->size().height / 4;

                int xb = 0;
                int scale = m_backgroundImage.size().width / xs;
                int yy = 0;
                int xx = 0;
                for(int y = 0; y < ys; y++)
                {
                    xx = 0;
                    for(int x = 0; x < xs; x++)
                    {
                        pGestureVisionDrawMat->at<cv::Vec3b>(y,x + xb) =  m_backgroundImage.at<cv::Vec3b>(yy,xx);
                        xx += scale;
                    }
                    yy += scale;
                }

                xb += xs;
                yy = 0;
                int rt = 255 / m_backgroundMaxAge;
                for(int y = 0; y < ys; y++)
                {
                    xx = 0;
                    for(int x = 0; x < xs; x++)
                    {
                        int f = m_backgroundImageAge.at<uchar>(yy, xx) * rt;
                        pGestureVisionDrawMat->at<cv::Vec3b>(y,x + xb) =  cv::Vec3b(f,f,f);
                        xx += scale;
                    }
                    yy += scale;
                }

                if (m_focusedFlowRegion.age > 0)
                {
                    xb += xs;
                    float xscale = (float)(m_focusedFlowRegion.trackRegionX1 - m_focusedFlowRegion.trackRegionX0 + 1) / (float)xs;
                    float yscale = (float)(m_focusedFlowRegion.trackRegionY1 - m_focusedFlowRegion.trackRegionY0 + 1) / (float)ys;
                    float fscale = xscale < yscale ? yscale : xscale;

                    float fyy = 0;
                    for(int y = 0; y < ys; y++)
                    {
                        float fxx = 0;
                        for(int x = 0; x < xs; x++)
                        {
                            pGestureVisionDrawMat->at<cv::Vec3b>(y,x + xb) =  m_trackingImage.at<cv::Vec3b>((int)fyy, (int)fxx);
                            fxx += fscale;
                        }
                        fyy += fscale;
                    }
                }

                if (foregroundImg.size().width > 0)
                {
                    xb += xs;
                    scale = foregroundImg.size().width / xs;
                    yy = 0;
                    for(int y = 0; y < ys; y++)
                    {
                        xx = 0;
                        for(int x = 0; x < xs; x++)
                        {
                            int f = foregroundImg.at<uchar>(yy, xx);
                            {
                                pGestureVisionDrawMat->at<cv::Vec3b>(y,x + xb) =  cv::Vec3b(f,f,f);
                            }
                            xx += scale;
                        }
                        yy += scale;
                    }
                }
            }

            if (m_focusedFlowRegion.age > 0)
            {
                cv::Point2i p1(m_focusedFlowRegion.xmin,m_focusedFlowRegion.ymin);
                cv::Point2i p2(m_focusedFlowRegion.xmax,m_focusedFlowRegion.ymax);
                cv::rectangle(*pGestureVisionDrawMat, p1,p2,CV_RGB(255,0,0));

                cv::Point pc = m_focusedFlowRegion.getCenter();
                cv::Point2f pv = m_focusedFlowRegion.direction;
                if (m_focusedFlowRegion.distance > 0)
                {
                    pv.x = pc.x + pv.x * m_focusedFlowRegion.distance;
                    pv.y = pc.y + pv.y * m_focusedFlowRegion.distance;
                    cv::line(*pGestureVisionDrawMat, pc, pv, CV_RGB(0,255,0),2);
                }
            }

            if (fingures.size() > 0)
            {
                for (unsigned int i = 0; i < fingures.size(); i += 2)
                {
                    cv::line(*pGestureVisionDrawMat,fingures[i], fingures[i+1], CV_RGB(255,0,0), 2);
                    cv::circle(*pGestureVisionDrawMat,fingures[i+1], 4, CV_RGB(0,255,0));
                }

            }
        }
    }

    newGrayScaleImage.copyTo(m_prevGray);
}

cv::Rect NDNGestureVision::findFingures(CvSeq *contour, std::vector<cv::Point>& fingures)
{
    int minFingureLengthP2 = m_imageHeight *  m_imageHeight / 400;

    fingures.clear();
    cv::Rect handRect(0,0,0,0);

    CvMemStorage *storage1 = cvCreateMemStorage (0);
    CvMemStorage *storage2 = cvCreateMemStorage (0);

    CvSeq *hull = NULL;
    if (contour->total > 0)
    {
        hull = cvConvexHull2(contour, storage1);
    }

    if(hull && hull->total > 0)
    {
        //We calculate convexity defects
        CvSeq* defects = cvConvexityDefects(contour, hull, storage2);
        if (defects->total >= 20 || defects->total == 0)
        {
            // don't process complex shapes. It won't be a hand shape.
            cvReleaseMemStorage(&storage1);
            cvReleaseMemStorage(&storage2);
            return handRect;
        }

        CvConvexityDefect defectArray[20];
        cvCvtSeqToArray(defects, defectArray, CV_WHOLE_SEQ);
        //printf("DefectArray %i %i\n",defectArray->end->x, defectArray->end->y);
        //We store defects points in the convexDefects parameter.

        // sort corner points by x coord of depth point.
        std::map<int,int> depth_pointMap;
        for(int i = 0; i<defects->total; i++)
        {
            depth_pointMap.insert(std::map<int, int>::value_type(defectArray[i].depth_point->x,i));
        }

        std::map<int, int>::iterator it = depth_pointMap.begin();
        int prx = 0;
        std::vector<cv::Point> dls;
        std::vector<cv::Point> drs;
        std::vector<cv::Point> dds;

        while( it != depth_pointMap.end() )
        {
            int idx = (*it).second;

            int crx = defectArray[idx].start->x;
            int clx = defectArray[idx].end->x;
            int cry = 0;
            int cly = 0;
            if (crx < clx)
            {
                int tmp = crx;
                crx = clx;
                clx = tmp;
                cry = defectArray[idx].end->y;
                cly = defectArray[idx].start->y;
            }
            else
            {
                cry = defectArray[idx].start->y;
                cly = defectArray[idx].end->y;
            }

            if (prx <= crx)
            {
                cv::Point p;

                p.x = defectArray[idx].depth_point->x;
                p.y = defectArray[idx].depth_point->y;
                if (p.y > cly)
                {
                    dds.push_back(p);

                    p.x = clx;
                    p.y = cly;
                    dls.push_back(p);

                    p.x = crx;
                    p.y = cry;
                    drs.push_back(p);

                    prx = crx;
                }
            }
            ++it;
        }

        std::vector<cv::Point> fingureRootPoints;
        std::vector<cv::Point> fingureEdgePoints;
        int minY = m_imageHeight;
        int midFingureLength = 0;
        if (dds.size() >= 4)
        {
            fingureRootPoints.push_back(dds[0]);
            fingureEdgePoints.push_back(dls[0]);
            cv::Point fingureRootPointPre = dds[0];
            cv::Point fingureEdgePointPre = dls[0];
            for (unsigned int i = 1 ; i < dls.size(); i++)
            {
                cv::Point fingureRootPoint;
                fingureRootPoint.x = ((dds.at(i).x - dds.at(i - 1).x) / 2) + dds.at(i - 1).x;
                fingureRootPoint.y = ((dds.at(i).y - dds.at(i - 1).y) / 2) + dds.at(i - 1).y;

                cv::Point rvec = fingureRootPointPre - fingureRootPoint;
                cv::Point evec = fingureEdgePointPre - dls[i];
                if (rvec.x * rvec.x + rvec.y * rvec.y > evec.x * evec.x + evec.y * evec.y)
                    continue;

                fingureRootPointPre = fingureRootPoint;
                fingureEdgePointPre = dls[i];
                fingureRootPoints.push_back(fingureRootPoint);
                fingureEdgePoints.push_back(dls[i]);
            }
            fingureRootPoints.push_back(dds.at(dds.size() - 1));
            fingureEdgePoints.push_back(drs.at(drs.size() - 1));
        }

        if (fingureEdgePoints.size() >= 5)
        {
            int fingureIdx = -1;
            for (unsigned int i = 4; i < fingureEdgePoints.size(); i++)
            {
                cv::Point dir1 = fingureRootPoints[i] - fingureEdgePoints[i];
                cv::Point dir2 = fingureRootPoints[i-1] - fingureEdgePoints[i-1];
                cv::Point dir3 = fingureRootPoints[i-2] - fingureEdgePoints[i-2];
                cv::Point dir4 = fingureRootPoints[i-3] - fingureEdgePoints[i-3];
                cv::Point dir5 = fingureRootPoints[i-4] - fingureEdgePoints[i-4];

                int f1lP2 = dir1.x *dir1.x + dir1.y * dir1.y;
                int f2lP2 = dir2.x *dir2.x + dir2.y * dir2.y;
                int f3lP2 = dir3.x *dir3.x + dir3.y * dir3.y;
                int f4lP2 = dir4.x *dir4.x + dir4.y * dir4.y;
                int f5lP2 = dir5.x *dir5.x + dir5.y * dir5.y;

                if (minFingureLengthP2 > f1lP2 || minFingureLengthP2 > f2lP2 || minFingureLengthP2 > f3lP2 || minFingureLengthP2 > f4lP2 || minFingureLengthP2 > f5lP2)
                    continue;

                int fmaxlP2 = (int)(f3lP2 * 2.2f);
                int fminlP2 = f3lP2 / 4;
                if (f1lP2 < fmaxlP2 && f2lP2 < fmaxlP2 && f4lP2 < fmaxlP2 && f5lP2 < fmaxlP2
                        && f1lP2 > fminlP2 && f2lP2 > fminlP2 && f4lP2 > fminlP2 && f5lP2 > fminlP2)
                {
                    midFingureLength = (int)sqrt((float)f3lP2);
                    int yrmax = fingureRootPoints[i-2].y + midFingureLength;
                    int yrmin = yrmax - midFingureLength / 2;

                    bool notFingure = false;
                    for (int j = i; j < 5; j++)
                    {
                        if (fingureRootPoints[i].y < yrmin && fingureRootPoints[i].y > yrmax)
                        {
                            notFingure = true;
                            break;
                        }
                    }

                    if (!notFingure && dir3.dot(dir1) > 0 && dir2.dot(dir2) > 0 && dir3.dot(dir4) > 0 && dir3.dot(dir5))
                    {
                        fingureIdx = i - 4;
                        break;
                    }
                }
            }

            if (fingureIdx >= 0)
            {
                for (int i = fingureIdx; i < fingureIdx + 5; i++)
                {
                    if (minY > fingureEdgePoints[i].y)
                    {
                        minY = fingureEdgePoints[i].y;
                    }
                    fingures.push_back(fingureRootPoints[i]);
                    fingures.push_back(fingureEdgePoints[i]);
                }
            }
        }


        if (fingures.size() == 10)
        {
            handRect.x = fingures[1].x;
            handRect.y = minY;
            handRect.width = fingures[9].x - fingures[1].x + 1;
            handRect.height = (int)(midFingureLength * 2.5f);
        }

        cvReleaseMemStorage(&storage1);
        cvReleaseMemStorage(&storage2);
    }

    return handRect;
}

float NDNGestureVision::compareTrackingImage(int x, int y, int width, int height, const cv::Mat &compImage)
{
    float score = 0;
    for (int fy = 0; fy < height; fy++)
    {
        int iy = fy + y;
        for (int fx = 0; fx < width; fx++)
        {
            int ix = fx + x;
            if ( iy < 0 || iy >= m_imageHeight || ix < 0 || ix >= m_imageWidth)
                continue;

            score += (510 - calcColorDiff(m_trackingImage.at<cv::Vec3b>(fy, fx), compImage.at<cv::Vec3b>(iy, ix)));
        }
    }

    return score;
}

void NDNGestureVision::paintForeground(IplImage *foreground, const cv::Mat &newImage, int x0, int y0, int x1, int y1)
{
    for (int y = y0; y <= y1; y++)
    {
        int p0 = foreground->widthStep * y;
        bool pixIn = false;
        for (int x = x0; x <= x1; x++)
        {
            if (!pixIn)
            {
                // take only the pixels continues more than m_forgroundNoiseThreashold times in column into account.
                if (x < m_imageWidth - m_forgroundNoiseThreashold)
                {
                    pixIn = true;
                    for (int nx = 0; nx < m_forgroundNoiseThreashold; nx++)
                    {
                        if (calcColorDiff(m_backgroundImage.at<cv::Vec3b>(y,x + nx), newImage.at<cv::Vec3b>(y,x + nx)) < m_backgroundDiffThreshold)
                        {
                            pixIn = false;
                            break;
                        }
                    }
                    if (pixIn)
                    {
                        for (int nx = 0; nx < m_forgroundNoiseThreashold; nx++)
                        {
                            foreground->imageData[p0 + x + nx] = 255;
                        }
                    }
                }
                x += m_forgroundNoiseThreashold - 1;
            }
            else
            {
                if (calcColorDiff(m_backgroundImage.at<cv::Vec3b>(y,x), newImage.at<cv::Vec3b>(y,x)) > m_backgroundDiffThreshold)
                {
                    foreground->imageData[p0 + x] = 255;
                }
                else
                {
                    pixIn = false;
                }
            }
        }
    }
}

int NDNGestureVision::calcColorDiff(const cv::Vec3b &baseColor, const cv::Vec3b &compColor)
{
    if (baseColor[1] > baseColor[0])
    {
        if (baseColor[2] > baseColor[1])
        {
            return abs(baseColor[2] - compColor[2]) + abs(baseColor[1] - compColor[1]);
        }
        else
        {
            if (baseColor[0] > baseColor[2])
            {
                return abs(baseColor[1] - compColor[1]) + abs(baseColor[0] - compColor[0]);
            }
            else
            {
                return abs(baseColor[1] - compColor[1]) + abs(baseColor[2] - compColor[2]);
            }
        }
    }
    else
    {
        if (baseColor[2] > baseColor[0])
        {
            return abs(baseColor[2] - compColor[2]) + abs(baseColor[0] - compColor[0]);
        }
        else
        {
            if (baseColor[2] > baseColor[1])
            {
                return abs(baseColor[0] - compColor[0]) + abs(baseColor[2] - compColor[2]);
            }
            else
            {
                return abs(baseColor[0] - compColor[0]) + abs(baseColor[1] - compColor[1]);
            }
        }
    }
}
