//  Copyright (c) 2013 NAO + dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Jun-4/2013.
//

#ifndef NDNGESTUREVISION_H
#define NDNGESTUREVISION_H

#include <opencv2/core/core.hpp>

class NDNGestureVision
{

public:

	class FlowRegion
	{
		friend class NDNGestureVision;

	public:
        FlowRegion() : extent(0), distance(0), prevDistance(0), xmin(0), xmax(0), ymin(0), ymax(0), age(0)
		{
		}

		FlowRegion(const FlowRegion &flowRegion)
		{
			extent = flowRegion.extent;
			direction = flowRegion.direction;
			distance = flowRegion.distance;
			xmin = flowRegion.xmin;
			xmax = flowRegion.xmax;
			ymin = flowRegion.ymin;
			ymax = flowRegion.ymax;
            prevDistance = flowRegion.prevDistance;
            _center = flowRegion._center;
			age = flowRegion.age;
		}

		int			extent;
		cv::Point2f direction;
		float		distance;
        float		prevDistance;
        int			xmin;
		int			xmax;
		int			ymin;
		int			ymax;
		int			age;

        cv::Point	getCenter() const { return _center; }

	private:
        void		calculateCenter()
        {
            assert(xmax >= xmin && ymax >= ymin);
            _center.x = (xmax - xmin + 1) / 2 + xmin;
            _center.y = (ymax - ymin + 1) / 2 + ymin;
        }

        cv::Point	_center;

        int         trackRegionX0;
        int         trackRegionY0;
        int         trackRegionX1;
        int         trackRegionY1;

	};

private:
	int				m_imageWidth;
	int				m_imageHeight;
	int				m_scanningXMax;
	int				m_scanningYMax;
	int				m_scanningStep;
	int				m_mapSize;
    cv::Mat         m_map;
    float			m_minFlowThresholdP2;
	int				m_numOfFlows;
	cv::Mat			m_prevGray;

	int				*m_tempSortedFlowNums;
    int				*m_tempSortedFlowExtents;

    FlowRegion      m_focusedFlowRegion;
    cv::Mat         m_trackingImage;

    cv::Mat         m_backgroundImage;
    cv::Mat         m_backgroundImageAge;

    cv::Rect findFingures(CvSeq *contour, std::vector<cv::Point>& fingures);
    void paintForeground(IplImage *foreground, const cv::Mat &newImage, int x0, int y0, int x1, int y1);
    inline float compareTrackingImage(int x0, int y0, int width, int height, const cv::Mat &compImage);
    inline int calcColorDiff(const cv::Vec3b &baseColor, const cv::Vec3b &compColor);


    //parameters
    int             m_minExt;
    int             m_minFocusUpdateSearchRange;
    float           m_fousedRegionMatchingGoodScore;
    float           m_fousedRegionMatchingAcceptScore;
    int             m_backgroundDiffThreshold;
    int             m_backgroundMaxAge;
    int             m_backgroundUpdateAgeThreshold;
    int             m_backgroundAgeToAddForMovingPart;
    int             m_backgroundAgeToSubForChangedPart;
    int             m_forgroundNoiseThreashold;
    int             m_focusedRegionMaxAge;
    int             m_focusedRegionInitAge;

public:
    NDNGestureVision(int imageWidth, int imageHeight, int scanningStep = 4, float minFlowThreshold = 1, int numOfFlows = 3);
	~NDNGestureVision();

    float           getTrackingImageGoodScore() const { return m_fousedRegionMatchingGoodScore; }
    float           getTrackingImageAcceptScore() const { return m_fousedRegionMatchingAcceptScore; }
    int             getFocusedRegionMaxAge() const { return m_focusedRegionMaxAge; }
    int             getFocusedRegionInitAge() const { return m_focusedRegionInitAge; }
    int             getImageDiffThreshold() const { return m_backgroundDiffThreshold; }

    void            setTrackingImageGoodScore(float in) { m_fousedRegionMatchingGoodScore = in; }
    void            setTrackingImageAcceptScore(float in) { m_fousedRegionMatchingAcceptScore = in; }
    void            setFocusedRegionMaxAge(int in) { m_focusedRegionMaxAge = in; }
    void            setFocusedRegionInitAge(int in) { m_focusedRegionInitAge = in; }
    void            setImageDiffThreshold(int in) { m_backgroundDiffThreshold = in; }

    const FlowRegion* getFlow(int index = 0) const;
    void calculateFlow(const cv::Mat &newImage, cv::Mat *pGestureVisionDrawMat = NULL);

};

#endif //NDNGESTUREVISION_H
