﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using Sasa.QualityTools.DrFx.Core;
using Sasa.QualityTools.DrFx.Core.Translation;
using System;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;

namespace Sasa.QualityTools.DrFx.Console
{
    internal sealed class Program
    {
        private Translator translator;



        /// <summary>
        /// 新しいインスタンスを初期化します。
        /// </summary>
        internal Program()
        {
            this.translator = new Translator();
            string directory = Path.Combine(Path.GetDirectoryName(GetType().Assembly.Location), "Resources");
            foreach (string file in Directory.GetFiles(directory, "*.xml"))
            {
                string rule = File.ReadAllText(file);
                this.translator.AddRules(TranslationRule.Parse(rule));
            }
        }



        private static int Main(string[] args)
        {
            Program program = new Program();
            return program.Boot(args);
        }

        internal int Boot(string[] args)
        {
            FxCopProcessConfiguration configuration = CreateConfiguration();

            try
            {
                ConfigureWithCommandLineArguments(configuration, args);
            }
            catch (InvalidOperationException e)
            {
                System.Console.Error.WriteLine(e.Message);
                return 1;
            }

            try
            {
                using (Process process = FxCopProcessGenerator.GenerateFxCopCmdProcess(configuration))
                {
                    process.OutputDataReceived += OnFxCopCmdMessageReceived;
                    process.ErrorDataReceived += OnFxCopCmdErrorMessageReceived;
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardOutput = true;
                    process.StartInfo.RedirectStandardError = true;
                    process.Start();
                    process.BeginOutputReadLine();
                    process.BeginErrorReadLine();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        System.Console.Error.WriteLine("FxCopCmd.exe が終了コード {0} で異常終了したため、レポートの翻訳を中止します。", process.ExitCode);
                        return 1;
                    }
                }
            }
            catch (ValidationException e)
            {
                System.Console.Error.WriteLine(e.Message);
                return 1;
            }

            string content = File.ReadAllText(configuration.OutputFile);
            content = this.translator.Translate(content);

            File.WriteAllText(configuration.OutputFile, content);

            return 0;
        }

        internal static FxCopProcessConfiguration CreateConfiguration()
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();

            if (ConfigurationManager.AppSettings["FxCopInstallDirectory"] != null)
            {
                configuration.FxCopInstallDirectory = ConfigurationManager.AppSettings["FxCopInstallDirectory"];
            }

            return configuration;
        }

        internal static void ConfigureWithCommandLineArguments(FxCopProcessConfiguration configuration, params string[] args)
        {
            Getopt getopt = new Getopt();
            getopt.ShortOption =  "d:ho:r:t:v";
            getopt.AddLongOption("dependency", 'd', true);
            getopt.AddLongOption("dictionary", (char)0xff01, true);
            getopt.AddLongOption("help", 'h', false);
            getopt.AddLongOption("output", 'o', true);
            getopt.AddLongOption("rule", 'r', true);
            getopt.AddLongOption("search-gac", (char)0xff02, false);
            getopt.AddLongOption("target", 't', true);
            getopt.AddLongOption("version", 'v', false);

            getopt.Parse(args, (opt, optarg) =>
            {
                switch (opt)
                {
                    case 'd':
                        {
                            string[] directories = FileSystemSearcher.SearchDirectories(optarg);
                            if (directories.Length == 0)
                            {
                                string message = "依存先アセンブリ検索ディレクトリのパスを修正してください。'" + optarg + "' に一致するディレクトリが見つかりません。";
                                throw new InvalidOperationException(message);
                            }
                            foreach (string directory in directories)
                            {
                                configuration.DependencyDirectories.Add(directory);
                            }
                        }
                        break;
                    case (char)0xff01:
                        configuration.DictionaryFile = optarg;
                        break;
                    case 'h':
                        PrintHelp();
                        Environment.Exit(0);
                        break;
                    case 'o':
                        configuration.OutputFile = optarg;
                        break;
                    case 'r':
                        {
                            string[] files = FileSystemSearcher.SearchFiles(optarg);
                            if (files.Length == 0)
                            {
                                string message = "ルールアセンブリのパスを修正してください。'" + optarg + "' に一致するファイルが見つかりません。";
                                throw new InvalidOperationException(message);
                            }
                            foreach (string file in files)
                            {
                                configuration.RuleFiles.Add(file);
                            }
                        }
                        break;
                    case (char)0xff02:
                        configuration.EnableSearchGac = true;
                        break;
                    case 't':
                        {
                            string[] files = FileSystemSearcher.SearchFiles(optarg);
                            if (files.Length == 0)
                            {
                                string message = "分析対象アセンブリのパスを修正してください。'" + optarg + "' に一致するファイルが見つかりません。";
                                throw new InvalidOperationException(message);
                            }
                            foreach (string file in files)
                            {
                                configuration.TargetFiles.Add(file);
                            }
                        }
                        break;
                    case 'v':
                        PrintVersion();
                        Environment.Exit(0);
                        break;
                    default:
                        throw new InvalidOperationException("認識できないオプション '" + optarg + "' が指定されました。");
                }
            });
        }

        private static void PrintHelp()
        {
            string[] help = new string[]{
                "Usage: drfx-console.exe --target <FILE> [--target <FILE> [...]] [OPTION]",
                "",
                "drfx-console は FxCopCmd.exe を実行してアセンブリの静的コード分析を行い、",
                "ルール違反メッセージを日本語化するツールです。",
                "",
                "コマンドラインオプションの説明",
                "",
                "--target <FILE> (略記 -t)",
                "分析対象のアセンブリを指定するためのオプションです。少なくとも 1 度は",
                "このオプションを指定してください。ワイルドカードを指定することも可能です。",
                "",
                "--output <FILE> (略記 -o)",
                "コード分析結果レポート出力先を指定するためのオプションです。省略した場合は",
                "作業ディレクトリの codeanalysisreport.xml に出力されます。",
                "",
                "--rule <FILE> (略記 -r)",
                "カスタムルールアセンブリを指定するためのオプションです。ワイルドカードを",
                "指定することも可能です。",
                "",
                "--dependency <DIRECTORY> (略記 -d)",
                "分析対象アセンブリが他のアセンブリに依存している場合に、依存先アセンブリが",
                "保存されているディレクトリを指定するためのオプションです。ワイルドカードを",
                "指定することも可能です。",
                "",
                "--dictionary <FILE>",
                "カスタム辞書ファイルを指定するためのオプションです。",
                "",
                "--search-gac",
                "分析対象アセンブリの依存関係解決のため、GAC をアセンブリ検索対象に含める",
                "ためのオプションです。",
                "",
                "--help (略記 -h)",
                "このヘルプメッセージを表示します。",
                "",
                "--version (略記 -v)",
                "バージョン情報を表示します。",
            };
            System.Console.WriteLine(String.Join(Environment.NewLine, help));
        }

        private static void PrintVersion()
        {
            System.Console.WriteLine("drfx-console.exe " + typeof(Program).Assembly.GetName().Version);
        }

        internal static Warning ParseMessage(string message)
        {
            string pattern = "^(?<SourceContext>.+?) : (?<Level>.+?)  : (?<CheckId>.+?) : (?<Category>.+?) : (?<Issue>.+?)$";
            Regex regex = new Regex(pattern, RegexOptions.Compiled | RegexOptions.Singleline);
            Match match = regex.Match(message);
            if (match.Success)
            {

                Warning warning = new Warning();
                warning.SourceContext = (match.Groups["SourceContext"].Value == "[Location not stored in Pdb]") ? "[ソース情報なし]" : match.Groups["SourceContext"].Value;
                warning.Level = (match.Groups["Level"].Value == "warning") ? "警告" : "エラー";
                warning.Category = match.Groups["Category"].Value;
                warning.CheckId = match.Groups["CheckId"].Value;
                warning.Issue = match.Groups["Issue"].Value;
                return warning;
            }
            else
            {
                return null;
            }
        }

        private string Translate(string message)
        {
            Warning warning = ParseMessage(message);
            if (warning != null)
            {
                string issue = this.translator.TranslateIssue(warning.Category, warning.CheckId, warning.Issue);
                return warning.SourceContext + " : " + warning.Level + " : " + warning.CheckId + " : " + warning.Category + " : " + issue;
            }
            else
            {
                return message;
            }
        }

        private void OnFxCopCmdMessageReceived(object sender, DataReceivedEventArgs e)
        {
            if (String.IsNullOrEmpty(e.Data))
            {
                System.Console.WriteLine();
            }
            else
            {
                System.Console.WriteLine(Translate(e.Data));
            }
        }

        private void OnFxCopCmdErrorMessageReceived(object sender, DataReceivedEventArgs e)
        {
            if (String.IsNullOrEmpty(e.Data))
            {
                System.Console.Error.WriteLine();
            }
            else
            {
                System.Console.Error.WriteLine(Translate(e.Data));
            }
        }
    }
}
