﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.IO;
using System.Xml;

namespace Sasa.QualityTools.DrFx.Core.Translation
{
    /// <summary>
    /// 翻訳ルールを表します。
    /// </summary>
    public class TranslationRule
    {
        /// <summary>
        /// 解決方法を表すメッセージのコレクション。
        /// </summary>
        private List<Resolution> resolutions;



        /// <summary>
        /// FxCop ルールのチェック ID。
        /// </summary>
        public string CheckId { get; private set; }

        /// <summary>
        /// FxCop ルールのカテゴリ。
        /// </summary>
        public string Category { get; private set; }

        /// <summary>
        /// FxCop ルール名。
        /// </summary>
        public string Name { get; private set; }

        /// <summary>
        /// ルール違反の解決方法を表すオブジェクトのコレクション。
        /// </summary>
        public IList<Resolution> Resolutions
        {
            get
            {
                return this.resolutions;
            }
        }



        /// <summary>
        /// 新しいインスタンスを初期化します。
        /// </summary>
        private TranslationRule()
        {
            this.resolutions = new List<Resolution>();
        }



        /// <summary>
        /// 翻訳ルールコンテンツを解析し、翻訳ルールコレクションを作成します。
        /// </summary>
        /// <param name="content">翻訳ルールコンテンツ。</param>
        /// <returns>翻訳ルールコレクション。</returns>
        public static IEnumerable<TranslationRule> Parse(string content)
        {
            List<TranslationRule> rules = new List<TranslationRule>();
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(content);
            foreach (XmlNode node in doc.SelectNodes("/Rules/Rule"))
            {
                TranslationRule rule = new TranslationRule();
                rule.CheckId = node.Attributes["CheckId"].Value;
                rule.Category = node.Attributes["Category"].Value;
                rule.Name = node["Name"].InnerText;
                rule.ParseResolutions(node);
                rules.Add(rule);
            }
            return rules;
        }

        /// <summary>
        /// 指定された Rule ノードを解析し、ルール違反の解決方法を表すオブジェクトを生成して、
        /// フィールドで保持します。
        /// </summary>
        /// <param name="ruleNode">解析対象の Rule ノード。</param>
        private void ParseResolutions(XmlNode ruleNode)
        {
            foreach (XmlNode node in ruleNode.SelectNodes("Resolution"))
            {
                Resolution resolution = new Resolution();
                resolution.Content = node["Content"].InnerText;
                resolution.Pattern = node["Pattern"].InnerText;
                this.resolutions.Add(resolution);
            }
        }
    }
}
