﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using Sasa.QualityTools.DrFx.Explorer.Report;
using System;
using System.Collections.Generic;
using System.Xml;

namespace Sasa.QualityTools.DrFx.Explorer.Tests.Unit.Report
{
    /// <summary>
    /// <see cref="Sasa.QualityTools.DrFx.Explorer.Report.FxCopReport"/> クラスのテストクラスです。
    /// </summary>
    [TestFixture]
    public class FxCopReportTest
    {
        /// <summary>
        /// 違反情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("違反情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateIssuesWithNull()
        {
            List<FxCopIssue> issues = FxCopReport.CreateIssues(null);
            Assert.AreEqual(0, issues.Count);
        }

        /// <summary>
        /// 違反情報リスト作成メソッドに空の Messages ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("違反情報リスト作成メソッドに空の Messages ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateIssuesWithEmptyNode()
        {
            string content = "<Messages></Messages>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopIssue> issues = FxCopReport.CreateIssues(node);
            Assert.AreEqual(0, issues.Count);
        }

        /// <summary>
        /// 違反情報リスト作成メソッドに Issue を１つだけ含む Messages ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("違反情報リスト作成メソッドに Issue を１つだけ含む Messages ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateIssuesWithSingleIssue()
        {
            string content =
                @"<Messages>" +
                @"    <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"        <Issue Level='Error' Path='C:\tmp' File='src.cs' Line='100'>issue message</Issue>" +
                @"    </Message>" +
                @"</Messages>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopIssue> issues = FxCopReport.CreateIssues(node);
            Assert.AreEqual(1, issues.Count);
            Assert.AreEqual("type", issues[0].TypeName);
            Assert.AreEqual("category", issues[0].Category);
            Assert.AreEqual("checkid", issues[0].CheckId);
            Assert.AreEqual("Active", issues[0].Status);
            Assert.AreEqual("Error", issues[0].Level);
            Assert.AreEqual(@"C:\tmp\src.cs", issues[0].FilePath);
            Assert.AreEqual(100, issues[0].Line);
            Assert.AreEqual("issue message", issues[0].Content);
        }

        /// <summary>
        /// 違反情報リスト作成メソッドに Issue を複数含む Messages ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("違反情報リスト作成メソッドに Issue を複数含む Messages ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateIssuesWithMultiIssue()
        {
            string content =
                @"<Messages>" +
                @"    <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"        <Issue Level='Error' Path='C:\tmp' File='src.cs' Line='100'>issue message 1</Issue>" +
                @"        <Issue Level='Warning'>issue message 2</Issue>" +
                @"    </Message>" +
                @"</Messages>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopIssue> issues = FxCopReport.CreateIssues(node);
            Assert.AreEqual(2, issues.Count);
            Assert.AreEqual("type", issues[0].TypeName);
            Assert.AreEqual("category", issues[0].Category);
            Assert.AreEqual("checkid", issues[0].CheckId);
            Assert.AreEqual("Active", issues[0].Status);
            Assert.AreEqual("Error", issues[0].Level);
            Assert.AreEqual(@"C:\tmp\src.cs", issues[0].FilePath);
            Assert.AreEqual(100, issues[0].Line);
            Assert.AreEqual("issue message 1", issues[0].Content);

            Assert.AreEqual("type", issues[1].TypeName);
            Assert.AreEqual("category", issues[1].Category);
            Assert.AreEqual("checkid", issues[1].CheckId);
            Assert.AreEqual("Active", issues[1].Status);
            Assert.AreEqual("Warning", issues[1].Level);
            Assert.IsNull(issues[1].FilePath);
            Assert.IsNull(issues[1].Line);
            Assert.AreEqual("issue message 2", issues[1].Content);
        }

        /// <summary>
        /// メンバー警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("メンバー警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateMemberWarningsWithNull()
        {
            List<MemberWarning> warnings = FxCopReport.CreateMemberWarnings(null);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// メンバー警告情報リスト作成メソッドに空の Members ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("メンバー警告情報リスト作成メソッドに空の Members ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateMemberWarningsWithEmptyNode()
        {
            string content = "<Members></Members>";
            XmlNode node = CreateXmlNode(content);
            List<MemberWarning> warnings = FxCopReport.CreateMemberWarnings(node);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// メンバー警告情報リスト作成メソッドに Member を１つだけ含む Members ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("メンバー警告情報リスト作成メソッドに Member を１つだけ含む Members ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateMemberWarningsWithSingleMember()
        {
            string content =
                @"<Members>" +
                @"    <Member Name='membername' Kind='Method'>" +
                @"        <Messages>" +
                @"            <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"                <Issue Level='Error' Path='C:\tmp\src.cs' Line='100'>issue message</Issue>" +
                @"            </Message>" +
                @"        </Messages>" +
                @"    </Member>" +
                @"</Members>";
            XmlNode node = CreateXmlNode(content);
            List<MemberWarning> warnings = FxCopReport.CreateMemberWarnings(node);
            Assert.AreEqual(1, warnings.Count);
            Assert.AreEqual("membername", warnings[0].Name);
            Assert.AreEqual("Method", warnings[0].Kind);
            Assert.AreEqual(1, warnings[0].Issues.Count);
        }

        /// <summary>
        /// メンバー警告情報リスト作成メソッドに Member を複数含む Members ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("メンバー警告情報リスト作成メソッドに Member を複数含む Members ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateMemberWarningsWithMultiMember()
        {
            string content =
                @"<Members>" +
                @"    <Member Name='membername1' Kind='Property'/>" +
                @"    <Member Name='membername2' Kind='Event'>" +
                @"        <Messages>" +
                @"            <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"                <Issue Level='Error' Path='C:\tmp\src.cs' Line='100'>issue message</Issue>" +
                @"            </Message>" +
                @"        </Messages>" +
                @"    </Member>" +
                @"</Members>";
            XmlNode node = CreateXmlNode(content);
            List<MemberWarning> warnings = FxCopReport.CreateMemberWarnings(node);
            Assert.AreEqual(2, warnings.Count);
            Assert.AreEqual("membername1", warnings[0].Name);
            Assert.AreEqual("Property", warnings[0].Kind);
            Assert.AreEqual(0, warnings[0].Issues.Count);

            Assert.AreEqual("membername2", warnings[1].Name);
            Assert.AreEqual("Event", warnings[1].Kind);
            Assert.AreEqual(1, warnings[1].Issues.Count);
        }

        /// <summary>
        /// 型警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("型警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateTypeWarningsWithNull()
        {
            List<TypeWarning> warnings = FxCopReport.CreateTypeWarnings(null);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// 型警告情報リスト作成メソッドに空の Types ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("型警告情報リスト作成メソッドに空の Types ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateTypeWarningsWithEmptyNode()
        {
            string content = "<Types></Types>";
            XmlNode node = CreateXmlNode(content);
            List<TypeWarning> warnings = FxCopReport.CreateTypeWarnings(node);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// 型警告情報リスト作成メソッドに Type を１つだけ含む Types ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("型警告情報リスト作成メソッドに Type を１つだけ含む Types ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateTypeWarningsWithSingleType()
        {
            string content =
                @"<Types>" +
                @"    <Type Name='typename' Kind='Class'>" +
                @"        <Messages>" +
                @"            <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"                <Issue Level='Error' Path='C:\tmp\src.cs' Line='100'>issue message</Issue>" +
                @"            </Message>" +
                @"        </Messages>" +
                @"    </Type>" +
                @"</Types>";
            XmlNode node = CreateXmlNode(content);
            List<TypeWarning> warnings = FxCopReport.CreateTypeWarnings(node);
            Assert.AreEqual(1, warnings.Count);
            Assert.AreEqual("typename", warnings[0].Name);
            Assert.AreEqual("Class", warnings[0].Kind);
            Assert.AreEqual(0, warnings[0].Members.Count);
            Assert.AreEqual(1, warnings[0].Issues.Count);
        }

        /// <summary>
        /// 型警告情報リスト作成メソッドに Type を複数含む Types ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("型警告情報リスト作成メソッドに Type を複数含む Types ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateTypeWarningsWithMultiType()
        {
            string content =
                @"<Types>" +
                @"    <Type Name='typename1' Kind='Interface'/>" +
                @"    <Type Name='typename2' Kind='Enum'>" +
                @"        <Members>" +
                @"            <Member Name='membername' Kind='Property'/>" +
                @"        </Members>" +
                @"    </Type>" +
                @"</Types>";
            XmlNode node = CreateXmlNode(content);
            List<TypeWarning> warnings = FxCopReport.CreateTypeWarnings(node);
            Assert.AreEqual(2, warnings.Count);
            Assert.AreEqual("typename1", warnings[0].Name);
            Assert.AreEqual("Interface", warnings[0].Kind);
            Assert.AreEqual(0, warnings[0].Members.Count);
            Assert.AreEqual(0, warnings[0].Issues.Count);

            Assert.AreEqual("typename2", warnings[1].Name);
            Assert.AreEqual("Enum", warnings[1].Kind);
            Assert.AreEqual(1, warnings[1].Members.Count);
            Assert.AreEqual(0, warnings[1].Issues.Count);
        }

        /// <summary>
        /// 名前空間警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("名前空間警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateNamespaceWarningsWithNull()
        {
            List<NamespaceWarning> warnings = FxCopReport.CreateNamespaceWarnings(null);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// 名前空間警告情報リスト作成メソッドに空の Namespaces ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("名前空間警告情報リスト作成メソッドに空の Namespaces ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateNamespaceWarningsWithEmptyNode()
        {
            string content = "<Namespaces></Namespaces>";
            XmlNode node = CreateXmlNode(content);
            List<NamespaceWarning> warnings = FxCopReport.CreateNamespaceWarnings(node);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// 名前空間警告情報リスト作成メソッドに Namespace を１つだけ含む Namespaces ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("名前空間警告情報リスト作成メソッドに Namespace を１つだけ含む Namespaces ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateNamespaceWarningsWithSingleNamespace()
        {
            string content =
                @"<Namespaces>" +
                @"    <Namespace Name='namespace'>" +
                @"        <Messages>" +
                @"            <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"                <Issue Level='Error' Path='C:\tmp\src.cs' Line='100'>issue message</Issue>" +
                @"            </Message>" +
                @"        </Messages>" +
                @"    </Namespace>" +
                @"</Namespaces>";
            XmlNode node = CreateXmlNode(content);
            List<NamespaceWarning> warnings = FxCopReport.CreateNamespaceWarnings(node);
            Assert.AreEqual(1, warnings.Count);
            Assert.AreEqual("namespace", warnings[0].Name);
            Assert.AreEqual(0, warnings[0].Types.Count);
            Assert.AreEqual(1, warnings[0].Issues.Count);
        }

        /// <summary>
        /// 名前空間警告情報リスト作成メソッドに Namespace を複数含む Namespaces ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("名前空間警告情報リスト作成メソッドに Namespace を複数含む Namespaces ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateNamespaceWarningsWithMultiNamespace()
        {
            string content =
                @"<Namespaces>" +
                @"    <Namespace Name='namespace1'/>" +
                @"    <Namespace Name='namespace2'>" +
                @"        <Types>" +
                @"            <Type Name='typename1' Kind='Interface'/>" +
                @"        </Types>" +
                @"    </Namespace>" +
                @"</Namespaces>";
            XmlNode node = CreateXmlNode(content);
            List<NamespaceWarning> warnings = FxCopReport.CreateNamespaceWarnings(node);
            Assert.AreEqual(2, warnings.Count);
            Assert.AreEqual("namespace1", warnings[0].Name);
            Assert.AreEqual(0, warnings[0].Types.Count);
            Assert.AreEqual(0, warnings[0].Issues.Count);
            Assert.AreEqual("namespace2", warnings[1].Name);
            Assert.AreEqual(1, warnings[1].Types.Count);
            Assert.AreEqual(0, warnings[1].Issues.Count);
        }

        /// <summary>
        /// アセンブリ警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("アセンブリ警告情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateAssemblyWarningsWithNull()
        {
            List<AssemblyWarning> warnings = FxCopReport.CreateAssemblyWarnings(null);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// アセンブリ警告情報リスト作成メソッドに空の Targets ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("アセンブリ警告情報リスト作成メソッドに空の Targets ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateAssemblyWarningsWithEmptyNode()
        {
            string content = "<Targets></Targets>";
            XmlNode node = CreateXmlNode(content);
            List<AssemblyWarning> warnings = FxCopReport.CreateAssemblyWarnings(node);
            Assert.AreEqual(0, warnings.Count);
        }

        /// <summary>
        /// アセンブリ警告情報リスト作成メソッドに Target を１つだけ含む Targets ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("アセンブリ警告情報リスト作成メソッドに Target を１つだけ含む Targets ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateAssemblyWarningsWithSingleAssembly()
        {
            string content =
                @"<Targets>" +
                @"    <Target Name='C:\tmp\asm.dll'>" +
                @"        <Modules>" +
                @"            <Module>" +
                @"                <Messages>" +
                @"                    <Message TypeName='type' Category='category' CheckId='checkid' Status='Active'>" +
                @"                        <Issue Level='Error' Path='C:\tmp\src.cs' Line='100'>issue message</Issue>" +
                @"                    </Message>" +
                @"                </Messages>" +
                @"            </Module>" +
                @"        </Modules>" +
                @"    </Target>" +
                @"</Targets>";
            XmlNode node = CreateXmlNode(content);
            List<AssemblyWarning> warnings = FxCopReport.CreateAssemblyWarnings(node);
            Assert.AreEqual(1, warnings.Count);
            Assert.AreEqual(@"C:\tmp\asm.dll", warnings[0].FileName);
            Assert.AreEqual("asm.dll", warnings[0].BaseName);
            Assert.AreEqual(0, warnings[0].Namespaces.Count);
            Assert.AreEqual(1, warnings[0].Issues.Count);
        }

        /// <summary>
        /// アセンブリ警告情報リスト作成メソッドに Target を複数含む Targets ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("アセンブリ警告情報リスト作成メソッドに Target を複数含む Targets ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateAssemblyWarningsWithMultiAssembly()
        {
            string content =
                @"<Targets>" +
                @"    <Target Name='C:\tmp\asm1.dll'/>" +
                @"    <Target Name='C:\tmp\asm2.dll'>" +
                @"        <Modules>" +
                @"            <Module>" +
                @"                <Namespaces>" +
                @"                    <Namespace Name='namespace1'/>" +
                @"                </Namespaces>" +
                @"            </Module>" +
                @"            <Module>" +
                @"                <Namespaces>" +
                @"                    <Namespace Name='namespace2'/>" +
                @"                    <Namespace Name='namespace3'/>" +
                @"                </Namespaces>" +
                @"            </Module>" +
                @"        </Modules>" +
                @"    </Target>" +
                @"</Targets>";
            XmlNode node = CreateXmlNode(content);
            List<AssemblyWarning> warnings = FxCopReport.CreateAssemblyWarnings(node);
            Assert.AreEqual(2, warnings.Count);
            Assert.AreEqual(@"C:\tmp\asm1.dll", warnings[0].FileName);
            Assert.AreEqual("asm1.dll", warnings[0].BaseName);
            Assert.AreEqual(0, warnings[0].Namespaces.Count);
            Assert.AreEqual(0, warnings[0].Issues.Count);
            Assert.AreEqual(@"C:\tmp\asm2.dll", warnings[1].FileName);
            Assert.AreEqual("asm2.dll", warnings[1].BaseName);
            Assert.AreEqual(3, warnings[1].Namespaces.Count);
            Assert.AreEqual(0, warnings[1].Issues.Count);
        }

        /// <summary>
        /// FxCop ルール情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop ルール情報リスト作成メソッドに null を渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateRulesWithNull()
        {
            List<FxCopRule> rules = FxCopReport.CreateRules(null);
            Assert.AreEqual(0, rules.Count);
        }

        /// <summary>
        /// FxCop ルール情報リスト作成メソッドに空の Rules ノードを渡した場合、空のリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop ルール情報リスト作成メソッドに空の Rules ノードを渡した場合、空のリストが作成されることをテストします。")]
        public void TestCreateRulesWithEmptyNode()
        {
            string content = "<Rules></Rules>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopRule> rules = FxCopReport.CreateRules(node);
            Assert.AreEqual(0, rules.Count);
        }

        /// <summary>
        /// FxCop ルール情報リスト作成メソッドに Rule を１つだけ含む Rules ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop ルール情報リスト作成メソッドに Rule を１つだけ含む Rules ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateRulesWithSingleRule()
        {
            string content =
                           @"<Rules>" +
                           @"    <Rule TypeName=""typename"" Category=""category"" CheckId=""checkid"">" +
                           @"        <Name>name</Name>" +
                           @"        <Description>description</Description>" +
                           @"        <Resolution>resolution</Resolution>" +
                           @"        <Owner>owner</Owner>" +
                           @"        <Url>rul</Url>" +
                           @"        <Email>email</Email>" +
                           @"        <MessageLevel Certainty=""95"">name</MessageLevel>" +
                           @"        <File Name=""file.dll"" Version=""1.0.0.0""/>" +
                           @"    </Rule>" +
                           @"</Rules>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopRule> rules = FxCopReport.CreateRules(node);
            Assert.AreEqual(1, rules.Count);
            Assert.AreEqual("checkid", rules[0].CheckId);
            Assert.AreEqual("category", rules[0].Category);
            Assert.AreEqual("name", rules[0].Name);
            Assert.AreEqual("description", rules[0].Description);
        }

        /// <summary>
        /// FxCop ルール情報リスト作成メソッドに Rule を複数含む Rules ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。
        /// </summary>
        [Test]
        [Description("FxCop ルール情報リスト作成メソッドに Rule を複数含む Rules ノードを渡した場合、XML の内容を反映したリストが作成されることをテストします。")]
        public void TestCreateRulesWithMultiRules()
        {
            string content =
                           @"<Rules>" +
                           @"    <Rule TypeName=""typename1"" Category=""category1"" CheckId=""checkid1"">" +
                           @"        <Name>name1</Name>" +
                           @"        <Description>description1</Description>" +
                           @"    </Rule>" +
                           @"    <Rule TypeName=""typename2"" Category=""category2"" CheckId=""checkid2"">" +
                           @"        <Name>name2</Name>" +
                           @"        <Description>description2</Description>" +
                           @"    </Rule>" +
                           @"</Rules>";
            XmlNode node = CreateXmlNode(content);
            List<FxCopRule> rules = FxCopReport.CreateRules(node);
            Assert.AreEqual(2, rules.Count);
            Assert.AreEqual("checkid1", rules[0].CheckId);
            Assert.AreEqual("category1", rules[0].Category);
            Assert.AreEqual("name1", rules[0].Name);
            Assert.AreEqual("description1", rules[0].Description);
            Assert.AreEqual("checkid2", rules[1].CheckId);
            Assert.AreEqual("category2", rules[1].Category);
            Assert.AreEqual("name2", rules[1].Name);
            Assert.AreEqual("description2", rules[1].Description);
        }

        /// <summary>
        /// 指定された XML を表す文字列を基に、<see cref="System.Xml.XmlNode"/> オブジェクトを作成します。
        /// </summary>
        /// <param name="content">XML を表す文字列。</param>
        /// <returns>指定された XML を表す文字列を基に作成した <see cref="System.Xml.XmlNode"/> オブジェクト。</returns>
        private XmlNode CreateXmlNode(string content)
        {
            XmlDocument document = new XmlDocument();
            document.LoadXml(content);
            return document.DocumentElement;
        }
    }
}
