﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using Sasa.QualityTools.DrFx.Explorer.Rule;
using System;
using System.Xml;

namespace Sasa.QualityTools.DrFx.Explorer.Tests.Unit.Rule
{
    [TestFixture]
    public class FxCopRuleLoaderTest
    {
        /// <summary>
        /// XML をパースして、FxCopRule オブジェクトが生成できることを確認します。
        /// </summary>
        [Test]
        public void TestCreateRule()
        {
            string content =
                "<Rule TypeName='typename' Category='category' CheckId='CA9999'>" +
                "    <Name>name</Name>" +
                "    <Description>description</Description>" +
                "    <Url>url</Url>" +
                "    <Resolution>resolution</Resolution>" +
                "    <Email>email</Email>" +
                "    <MessageLevel Certainty='99'>messagelevel</MessageLevel>" +
                "    <FixCategories>fixcategories</FixCategories>" +
                "    <Owner>owner</Owner>" +
                "</Rule>";
            XmlNode node = Parse(content);

            FxCopRule actual = FxCopRuleLoader.CreateRule(node);

            Assert.That(actual.Name, Is.EqualTo("name"));
            Assert.That(actual.CheckId, Is.EqualTo("CA9999"));
            Assert.That(actual.Category, Is.EqualTo("category"));
            Assert.That(actual.TypeName, Is.EqualTo("typename"));
        }

        [Test]
        [ExpectedException(typeof(InvalidOperationException))]
        public void TestCreateRuleWithContentNoName()
        {
            string content =
                "<Rule TypeName='typename' Category='category' CheckId='CA9999'>" +
                "    <Description>description</Description>" +
                "    <Url>url</Url>" +
                "    <Resolution>resolution</Resolution>" +
                "    <Email>email</Email>" +
                "    <MessageLevel Certainty='99'>messagelevel</MessageLevel>" +
                "    <FixCategories>fixcategories</FixCategories>" +
                "    <Owner>owner</Owner>" +
                "</Rule>";
            XmlNode node = Parse(content);

            FxCopRule actual = FxCopRuleLoader.CreateRule(node);
        }

        [Test]
        [ExpectedException(typeof(InvalidOperationException))]
        public void TestCreateRuleWithContentNoCheckId()
        {
            string content =
                "<Rule TypeName='typename' Category='category'>" +
                "    <Name>name</Name>" +
                "    <Description>description</Description>" +
                "    <Url>url</Url>" +
                "    <Resolution>resolution</Resolution>" +
                "    <Email>email</Email>" +
                "    <MessageLevel Certainty='99'>messagelevel</MessageLevel>" +
                "    <FixCategories>fixcategories</FixCategories>" +
                "    <Owner>owner</Owner>" +
                "</Rule>";
            XmlNode node = Parse(content);

            FxCopRule actual = FxCopRuleLoader.CreateRule(node);
        }

        [Test]
        [ExpectedException(typeof(InvalidOperationException))]
        public void TestCreateRuleWithContentNoCategory()
        {
            string content =
                "<Rule TypeName='typename' CheckId='CA9999'>" +
                "    <Name>name</Name>" +
                "    <Description>description</Description>" +
                "    <Url>url</Url>" +
                "    <Resolution>resolution</Resolution>" +
                "    <Email>email</Email>" +
                "    <MessageLevel Certainty='99'>messagelevel</MessageLevel>" +
                "    <FixCategories>fixcategories</FixCategories>" +
                "    <Owner>owner</Owner>" +
                "</Rule>";
            XmlNode node = Parse(content);

            FxCopRule actual = FxCopRuleLoader.CreateRule(node);
        }

        [Test]
        [ExpectedException(typeof(InvalidOperationException))]
        public void TestCreateRuleWithContentNoTypeName()
        {
            string content =
                "<Rule Category='category' CheckId='CA9999'>" +
                "    <Name>name</Name>" +
                "    <Description>description</Description>" +
                "    <Url>url</Url>" +
                "    <Resolution>resolution</Resolution>" +
                "    <Email>email</Email>" +
                "    <MessageLevel Certainty='99'>messagelevel</MessageLevel>" +
                "    <FixCategories>fixcategories</FixCategories>" +
                "    <Owner>owner</Owner>" +
                "</Rule>";
            XmlNode node = Parse(content);

            FxCopRule actual = FxCopRuleLoader.CreateRule(node);
        }
        
        /// <summary>
        /// XML コンテンツをパースします。
        /// </summary>
        /// <param name="content">パースしたい XML コンテンツ。</param>
        /// <returns>XML コンテンツをパースして得られた XML ノードオブジェクト。</returns>
        private XmlNode Parse(string content)
        {
            XmlDocument document = new XmlDocument();
            document.LoadXml(content);
            return document.DocumentElement;
        }
    }
}
