﻿//  Copyright (c) 2010-2012, www.dvgtu.org
//  All rights reserved.

//  Redistribution and use in source and binary forms, with or without modification, 
//  are permitted provided that the following conditions are met:

//  * Redistributions of source code must retain the above copyright notice, 
//    this list of conditions and the following disclaimer. 
//  * Redistributions in binary form must reproduce the above copyright notice, 
//    this list of conditions and the following disclaimer in the documentation 
//    and/or other materials provided with the distribution. 

//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
//  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
//  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
//  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
//  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
//  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
//  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
//  POSSIBILITY OF SUCH DAMAGE.
// $Id: ProfileForm.cs 128 2011-11-11 13:08:43Z takeshich $

using System;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Reflection;
using System.IO;
using OpenMetaverse;
using OpenMetaverse.Assets;
using dvgtuViewer.SLNetworkComm;
using CSJ2K;

namespace dvgtuViewer
{
	/// <summary>
	/// Description of ProfileForm.
	/// </summary>
	public partial class ProfileForm : WeifenLuo.WinFormsUI.Docking.DockContent
	{
		
		private DvgtuInstance dvgtuInstance;
		private GridClient Client{ get { return dvgtuInstance.Client; } }
		private SLNetCom NetCom{ get { return dvgtuInstance.Netcom; } }
		
		private UUID AvatarID;
		private UUID PartnerID;
		private string FullAvatarName;
		
		private TextureDownloadCallback TextureDownloadCallback;

		
		public ProfileForm(DvgtuInstance dvgtuInstance)
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			
			//
			// TODO: Add constructor code after the InitializeComponent() call.
			//
			this.dvgtuInstance = dvgtuInstance;
			this.Load += new EventHandler(ProfileForm_Load);
		}

		void ProfileForm_Load(object sender, EventArgs e)
		{
			//NetCom.NetcomSync = this;
			
			GetProfile(AvatarID);
			lblProgress.Visible = false;
			progressBar1.Visible = false;
			BtnWebView.Visible = false;
			
			Client.Avatars.AvatarPropertiesReply += new EventHandler<AvatarPropertiesReplyEventArgs>(Client_Avatars_AvatarPropertiesReply);
			Client.Avatars.UUIDNameReply += new EventHandler<UUIDNameReplyEventArgs>(Client_Avatars_UUIDNameReply);
			Client.Assets.ImageReceiveProgress += new EventHandler<ImageReceiveProgressEventArgs>(Client_Assets_ImageReceiveProgress);
			
			TextureDownloadCallback += new TextureDownloadCallback(ProfileImageTextureDownloadCallback);
		}
		
		void ProfileForm_Disposed(object sender,EventArgs e){
			Client.Avatars.AvatarPropertiesReply -= new EventHandler<AvatarPropertiesReplyEventArgs>(Client_Avatars_AvatarPropertiesReply);
			Client.Avatars.UUIDNameReply -= new EventHandler<UUIDNameReplyEventArgs>(Client_Avatars_UUIDNameReply);
			Client.Assets.ImageReceiveProgress -= new EventHandler<ImageReceiveProgressEventArgs>(Client_Assets_ImageReceiveProgress);
		}

		void Client_Assets_ImageReceiveProgress(object sender, ImageReceiveProgressEventArgs e)
		{
			if (InvokeRequired)
            {
                Invoke(new MethodInvoker(() => Client_Assets_ImageReceiveProgress(sender, e)));
                return;
            }
			
			int pct = 0;
            if (e.Total > 0)
            {
                pct = (e.Received * 100) / e.Total;
            }
            
            if (pct < 0 || pct > 100)
            {
                return;
            }
            
            lblProgress.Visible = true;
            progressBar1.Visible = true;
            
            lblProgress.Text = String.Format("{0} of {1}KB ({2}%)", (int)e.Received / 1024, (int)e.Total / 1024, pct);
            progressBar1.Value = pct;
            
            if (pct == 100)
            {
            	lblProgress.Hide();
            	progressBar1.Hide();
            }
		}
		
//		public void SetClient(GridClient client){
//			Client = client;
//		}
//		
//		public void SetNetCom(SLNetCom netCom){
//			NetCom = netCom;
//		}
		
		public void SetAvatarID(UUID avatarID){
			AvatarID = avatarID;
		}
		
		void Client_Avatars_UUIDNameReply(object sender, UUIDNameReplyEventArgs e)
		{
			string AvatarName;
			string PartnerAvatarName;
			if (InvokeRequired){
				Invoke(new MethodInvoker(() => Client_Avatars_UUIDNameReply(sender,e)));
				return;
			}	
						
			if (e.Names.TryGetValue(AvatarID,out AvatarName)){
				txtFullName.Text = AvatarName;
				this.Text = AvatarName;
				FullAvatarName = AvatarName;
			}
			
			if (PartnerID != UUID.Zero){
				if (e.Names.TryGetValue(PartnerID,out PartnerAvatarName)){
					txtPartner.Text = PartnerAvatarName;
				}	
			}
			
		}

		void Client_Avatars_AvatarPropertiesReply(object sender, AvatarPropertiesReplyEventArgs e)
		{
			if (InvokeRequired){
				Invoke(new MethodInvoker(() => Client_Avatars_AvatarPropertiesReply(sender,e)));
				return;
			}
			
			string AboutText = e.Properties.AboutText;
			string BornOn = e.Properties.BornOn;
			string ProfileURL = e.Properties.ProfileURL;
			UUID ProfileImageID = e.Properties.ProfileImage;
			
			PartnerID = e.Properties.Partner;
			
			txtAbout.Text = AboutText;
			txtBornOn.Text = BornOn;
			if (ProfileURL != string.Empty){
				txtUrl.Text = ProfileURL;
				BtnWebView.Visible = true;
			}
			
			
			
			Client.Assets.RequestImage(ProfileImageID,OpenMetaverse.ImageType.Normal,TextureDownloadCallback);
			Client.Avatars.RequestAvatarName(PartnerID);
		}
		
		/// <summary>
		/// GetProfile :AvatarIDからプロファイル情報を取得する
		/// </summary>
		/// <param name="AvatarID"></param>
		/// 
		private void GetProfile(UUID AvatarID){
			
			Client.Avatars.RequestAvatarProperties(AvatarID);
			Client.Avatars.RequestAvatarName(AvatarID);
		
		}
		
		void ProfileForm_Close(object sender, EventArgs e)
		{
//			Assembly asm = Assembly.GetEntryAssembly();
//			string fullPath = asm.Location;
//
//			//フル・パスからディレクトリ・パス部分を抽出する
//			string dirPath = Path.GetDirectoryName(fullPath);
//			string path =  dirPath + "\\DockSettings.xml";
//			DockPanel.SaveAsXml(path);
		}

		
		
		void ProfileImageTextureDownloadCallback(TextureRequestState state, AssetTexture assetTexture)
		{
			try
			{
				if (state != TextureRequestState.NotFound || state != TextureRequestState.Timeout)
				{

					picProfileImage.Image = J2kImage.FromBytes(assetTexture.AssetData);
					//for debug
//					Assembly asm = Assembly.GetEntryAssembly();
//					string fullPath = asm.Location;
//				
//					//フル・パスからディレクトリ・パス部分を抽出する
//					string dirPath = Path.GetDirectoryName(fullPath);
//					string path =  dirPath + "\\ProfileImage.jpg";
//					Bitmap bitmap = new Bitmap(J2kImage.FromBytes(assetTexture.AssetData));
//					bitmap.Save(path);
				}

			}
			catch(Exception ex)
			{
				Logger.Log("ProfileForm_ProfileImageTextureDownloadCallback:" + ex.Message,OpenMetaverse.Helpers.LogLevel.Error);
			}
		}
		
		
		
		
		void BtnCloseClick(object sender, EventArgs e)
		{
			this.Close();
		}
		
		void BtnFriendshipClick(object sender, EventArgs e)
		{
			Client.Friends.OfferFriendship(AvatarID);
		}
		
		void BtnTeleportClick(object sender, EventArgs e)
		{
			Client.Self.SendTeleportLure(AvatarID,Client.Network.CurrentSim.Name + "に来て!");
		}
		
		void BtnWebViewClick(object sender, EventArgs e)
		{
			Process.Start(txtUrl.Text);
		}
		
		void BtnSendIMClick(object sender, EventArgs e)
		{
			dvgtuInstance.chatForm.IMPreparate(AvatarID,FullAvatarName);
			this.Close();
		}
	}
}
