<?php
// vim: foldmethod=marker
/**
 *  Ethna_DB_MySQL.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: 97ff48161c82a774753c46e85cc0c76a345cb619 $
 */

// {{{ Ethna_DB_MySQL
/**
 *  MySQL用、データベースアクセスドライバ
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_MySQL extends Ethna_DB
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  string  $dsn DSN
     *  @param  bool    $persistent 持続接続設定
     */
    function Ethna_DB_MySQL($dsn, $persistent)
    {
        parent::Ethna_DB($dsn, $persistent);
    }

    /**
     *  DBに接続する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function connect()
    {
        $host = ($this->_dsninfo['socket'])
              ? $this->_dsninfo['socket']
              : $this->_dsninfo['hostspec'];
        if ($this->_dsninfo['port']) {
            $host .= ":$port";
        }
        $user = $this->_dsninfo['username'];
        $pass = $this->_dsninfo['password'];
        $db = NULL;
        if ($this->_persistent) {
            $db = mysql_pconnect($host, $user, $pass);
        } else { 
            $db = mysql_connect($host, $user, $pass);
        } 
        if ($db === false) {
            return Ethna::raiseError(
                       "could not connect database: $host, $user, $pass"
                   );
        }

        $dbname = $this->_dsninfo['database'];
        $select_db_result = mysql_select_db(
                                $dbname,
                                $db
                            );
        if ($select_db_result === false) {
            $this->_db = NULL;
            return Ethna::raiseError(
                       "could not select MySQL database: $dbname"
                   );
        }
        $this->_db = $db;
        return 0;
    }

    /**
     *  DB接続を切断する
     *
     *  @access public
     */
    function disconnect()
    {
        if ($this->_persistent) {
            return 0;  //  持続的接続は閉じない
        } 
        $result = mysql_close($this->_db);
        if ($result === false) {
            return Ethna::raiseError(
                "could not close Connection!: "
              . $this->_dsn
            );
        }
        $this->_db = NULL;
        return 0;
    }

    /**
     *  DBトランザクションを開始する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function begin()
    {
        return $this->__transact_query('begin');
    }

    /**
     *  DBトランザクションを中断する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function rollback()
    {
        return $this->__transact_query('rollback');
    }

    /**
     *  DBトランザクションを終了する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function commit()
    {
        return $this->__transact_query('commit');
    }

    /**
     *  SQL ステートメントを実行する準備を行い、
     *  Ethna_Statementオブジェクトを返します。
     *
     *  Ethna_Statement::execute() メソッドによ
     *  って実行される SQL ステートメントを準備
     *  します。 SQL ステートメントは、文が実行
     *  されるときに実際の値に置き換えられます。
     *
     *  (ステートメントにパラメータを指定する場合、
     *   ? で指定すること)
     *  @access public
     *  @param  $sql  実行を準備するSQL
     *  @return mixed 成功時にEthna_DB_Statement
     *                Ethna_Error:エラー
     */
    function prepare($sql)
    {
        $res = NULL;
        $stmt = new Ethna_DB_MySQL_Statement($this);
        $stmt->setLogger($this->_logger);

        //
        //   MySQLの手続き型APIでは、prepared stmt
        //   向けのAPIがないので、常にエミュレートされます
        //

        $stmt->setOption('__sql', trim($sql));
        $stmt->setOption('__emulation_mode', true);
        return $stmt; 
    }

    /**
     *  SQL ステートメントを実行し、作用した行数を返します。
     *  SELECT 文からは結果を返しません。SELECT 文を指定した
     *  場合は常に0が返ります。
     *
     *  @access public
     *  @param  $sql    実行するSQL
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function exec($sql)
    {
        $sql = ltrim($sql);
        if (preg_match('/^SELECT/i', $sql)) {
            return 0;
        }
        $this->_logger->log(LOG_DEBUG, $sql);
        $result = mysql_query($sql, $this->_db);
        if (is_resource($result)) {
            return Ethna::raiseError(
                "this method can not return resource: "
              . $this->_dsn
              . " query: $sql",
                E_DB_QUERY 
            );
        } 
        if ($result === false) {
             return Ethna::raiseError(
                "could not exec your sql: "
              . $this->_dsn
              . " query: $sql",
                E_DB_QUERY 
            );
        }
        return mysql_affected_rows($this->_db); 
    }

    /**
     *  直前のINSERTによるIDを取得する
     *
     *  @access public
     *  @param  string  (MySQLでは不要)$name データベース依存のパラメータ
     *  @return mixed   int:直近のINSERTにより生成されたID null:エラー
     */
    function getInsertId($name = NULL)
    {
        return mysql_insert_id($this->_db);
    }

    /**
     *  文字列をエスケープする 
     *
     *  @access public
     *  @param  string  $value  取得対象のテーブル名
     *  @return string  エスケープ済みの値
     */
    function escape($value)
    {
        return mysql_real_escape_string($value, $this->_db);
    }

    /**
     *  テーブル定義情報を取得する
     *
     *  @access public
     *  @param  string  $table_name テーブル名 
     *  @return mixed   array: PEAR::DBに準じたメタデータ
     *                  Ethna_Error::エラー
     */
    function getMetaData($table_name)
    {
        //
        //  TODO:  implement this method.
        //
    }

    /**
     *  データベースのタイプを取得する
     *
     *  @access public
     *  @return string  データベースのタイプ
     */
    function getType()
    {
        return 'mysql';
    }

    /**
     *  DBトランザクション関連のクエリを送信
     *  する内部関数です。
     *
     *  @access private 
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __transact_query($query)
    {
        if (!$this->isValid()) {
             return Ethna::raiseError(
                "Not Connected: "
              . $this->_dsn
            );
        }
        $this->_logger->log(LOG_DEBUG, $query);
        $result = mysql_query($query, $this->_db);
        if ($result === false) {
            $error = mysql_error($this->_db);
              return Ethna::raiseError(
                "$query transation failed: "
              . $this->_dsn
              . " $error",
                E_DB_QUERY 
            );
        }
        return 0; 
    }
}

// {{{ Ethna_DB_MySQL_Statement
/**
 *  MySQL ドライバの一部
 *
 *  実行前にはプリペアドステートメント。
 *  実行後には関連する結果セットを表す
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_MySQL_Statement extends Ethna_DB_Statement
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  Ethna_DB $db データベース接続 
     */
    function Ethna_DB_MySQL_Statement(&$db)
    {
        parent::Ethna_DB_Statement($db);
    } 

    /**
     *  プリペアドステートメント実行結果から
     *  次の行を取得します。 
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse 
     */
    function fetchRow()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            return mysql_fetch_row($this->_result);
        case DB_FETCHMODE_ASSOC:
            return mysql_fetch_assoc($this->_result);
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
    }

    /**
     *  プリペアドステートメント実行結果の
     *  結果セットを「全て」配列で返します
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse
     */
    function fetchAll()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        $allresult = array();
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            while ($row = mysql_fetch_row($this->_result)) {
                array_push($allresult, $row);
            }
            break;
        case DB_FETCHMODE_ASSOC:
            while ($row = mysql_fetch_assoc($this->_result)) {
                array_push($allresult, $row);
            }
            break;
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
        return $allresult;
    }

    /**
     *  直近の DELETE, INSERT, UPDATE 文によっ
     *  て作用した行数を返します。 
     *
     *  @access public
     *  @return int  作用した行数
     *               SELECT の場合は 0
     */
    function affectedRows()
    {
        return mysql_affected_rows($this->_db); 
    }

    /**
     *  プリペアドステートメントのエミュレー
     *  ションロジックです。
     *
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? として指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __emulatePstmt($sql, $param = array())
    {
        $prepare_sql = $this->__getEmulatePstmtSQL($sql, $param);
        $result = mysql_query($prepare_sql, $this->_db);
        if ($result === false) {
            $error = mysql_error($this->_db);
            return Ethna::raiseError(
                "could not exec your sql: "
              . " query: $prepare_sql"
              . " detail: $error ",
                E_DB_QUERY 
            );
        }
        $this->_result = $result;
        return 0;                
    }
} 

?>
