<?php
// vim: foldmethod=marker
/**
 *  Ethna_AppObject.php
 *
 *  @author     Masaki Fujimoto <fujimoto@php.net>
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: fbfcbd440dcf5f06cbbf05fa48fb207b9ff70f70 $
 */

// {{{ Ethna_AppObject
/**
 *  アプリケーションオブジェクトのベースクラス
 *  
 *  アプリケーションオブジェクトのインスタンスひとつが
 *  当該テーブルの1レコードを表現する
 *
 *  ※  このクラスの仕様は ActiveRecord を模倣した上で、
 *     アクションフォームとの親和性を高める仕組みを用意する。
 *     但しmigration はない。
 *  @see http://ar.rubyonrails.org/
 *
 *  @author     Masaki Fujimoto <fujimoto@php.net>
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 *  @TODO       アクションフォームを短いルーチンで保存させる
 *              ルーチンまたはインターフェイス
 */
class Ethna_AppObject
{
    /**#@+
     *  @access private
     */

    /*
     *  @var   string   このオブジェクトが表現するテーブル名
     *                  データベースに対応するテーブル名を指定 
     *                  デフォルトはクラス名を小文字にしたもの。異なる場合のみ指定
     */
    var $_table_name; 

    /*
     *  @var   mixed    このクラスが表現するテーブルが所属するデータベースの接続キー
     */
    var $_db_key = ''; 

    /**
     *  @var    array   テーブルのリレーション定義。
     *                  
     *   var $relations = array(
     *                      //   1対1の関係であることを示す
     *                      //   値がスカラーの場合は、結びつくキーとなる
     *                      //   カラムの値は id と見做される
     *                      'has_one' => 'foo',   //  foo テーブルと1対1の関係にある
     *                      'has_one' => array(
     *                                     'tbl' => 'foo',    //fooテーブルと1対1の関係
     *                                     'key' => 'foo_key',//fooテーブルとfoo_keyカラムで結ぶ
     *                                   ),
     *                      //   1対多の関係であることを示す
     *                      //   値がスカラーの場合は、結びつくキーとなる
     *                      //   カラムの値は id と見做される
     *                      'has_many' => 'bar',
     *                      'has_many' => array(
     *                                     'tbl' => 'bar',    //barテーブルと1対多の関係
     *                                     'key' => 'bar_key',//barテーブルとbar_keyカラムで結ぶ
     *                                     //   多対多の関係の場合に、中間に挟むテーブル名を
     *                                     //   指定する。値がスカラーの場合は、結びつくキーと
     *                                     //   なるカラムの値は id と見做される
     *                                     'through' =>  array(
     *                                                     'tbl' => 'baz',    //bazが中間テーブル
     *                                                     'key' => 'baz_key',//baz_keyカラムで結ぶ
     *                                                   ),
     *                                     ),
     *                      //   結び付く対象となるテーブルの場合、
     *                      //   belongs_to に結びつく先のテーブル名を指定
     *                      'belongs_to' => array(
     *                                        'tbl' => 'baz',
     *                                        'key' => 'bar_key',//barテーブルとbar_keyカラムで結ぶ
     *                                      ),
     *                    );
     */
    var $_relations = array();

    /**
     *   @var  array  テーブルのカラム定義。テーブルのカラム定義を記述します。 
     *                この定義はデータベースから取得され、キャッシュされます。
     *   $_columns = array(
     *        'column_name' => array( 
     *           //    SQL の取得結果を別の名前で取得したい場合、
     *           //    この名前を指定する。ユーザーがオーバーライド可能
     *           'as'        => $alias_col_name,
     *           //    表示名。テンプレート等への表示に使う 
     *           //    カラムの値と違う場合にユーザーがオーバライド可能。
     *           'name'      => $display_name,
     *           //    INSERT時に値が指定されなかった場合に
     *           //    挿入されるデフォルト値。デフォルトはNULLだが、
     *           //    ユーザーがオーバーライド可能
     *           'default'   => $default_val, 
     *           
     *           //    以下は内部で使われるカラム定義
     *           //    クエリ生成に利用される。オーバライドしてはいけない
     *           'primary'   => $primary,     //  primary key なら true
     *           'seq'       => $seq,         //  sequence 型なら  true  
     *           'unique'    => $key,         //  uniqueキーならtrue
     *           'type'      => $type,        //  データ型
     *           'required'  => $required,    //  NOT NULL なら true
     *       ),
     *   );
     */
    var $_columns = array();

    /** @var    array    カラム値の更新前の値  */
    var $_orig_values = NULL;

    /** @var    array    カラム値の現在の値  */
    var $_cur_values = NULL;

    /** @var    boolean  取得(初期設定)したカラム値が更新されたか否か */
    var $_is_dirty = false;
 
    /** @var    boolean  取得したカラム値が読み取り専用か否か */
    var $_is_read_only = false;

    /** @var    boolean  プライマリーキーを持っているか否か */
    var $_has_key = false;

    /** @var    boolean  アプリケーションオブジェクトの全機能を使えるか否か */
    var $_is_valid = false;

    /** @var    int      テーブル定義のカラム定義キャッシュ有効期間(sec) */
    var $_def_cache_lifetime = 86400;  //  デフォルトは 1 day.

    /** @var    Ethna_DB  データベース接続オブジェクト */
    var $_db = NULL;

    /** @var    mixed    ロガーオブジェクト */
    var $_logger = NULL;

    /**#@-*/

    // {{{ Ethna_AppObject
    /**
     *  Ethna_AppObjectクラスのコンストラクタ
     *
     *  @access public
     *  @param  array   $values     レコードの初期値。カラム名をキーとし、
     *                              カラムの値を値として指定した配列
     */
    function Ethna_AppObject($values = NULL)
    {
        //    テーブル名を初期化
        //    デフォルトはクラス名を小文字にしたもの
        if (empty($this->_table_name)) {
            $this->_table_name = strtolower(get_class());
        }
        
        //   コンストラクタで指定されたカラム値の初期値を設定
        if (is_array($values)) {
            foreach ($values as $k => $v) {
                $this->$k = $v;
            } 
            $this->_orig_values = $values;
            $this->_cur_values = $values;
        }
    }
    // }}}

    // {{{ setDB 
    /**
     *  データベース接続オブジェクトを設定する
     *
     *  @access public
     *  @param  Ethna_DB  $db  データベース接続オブジェクト
     *  @return mixed  0:正常終了 エラーの場合はEthna_Error
     */
    function setDB($db = NULL)
    {
        if (!empty($this->_db) && is_subclass_of($this->_db, 'Ethna_DB')) {
            return 0;   //  既に設定済み
        }

        if (empty($db) && class_exists('Ethna_DB_Util')) {
            $db = Ethna_DB_Util::getDB($this->_db_key);
            if (Ethna::isError($db)) {
                $errmsg = get_class() . ": AppObject init failed -> "
                        . $db->getMessage();
                return Ethna::raiseError($errmsg, E_DB_GENERAL);
            }
        } else {
            //  Ethna_DB の継承クラスでなければならない
            if (!is_subclass_of($db, 'Ethna_DB')) {
                return Ethna::raiseError(
                           "invalid database object", E_GENERAL
                       );
            }
        }
        $this->_db = $db;
        $this->_logger = $db->getLogger();
        $this->_is_valid = $this->isValid();

        return 0;
    }
    // }}}

    // {{{ create
    /**
     *  指定された初期値でデータベースに値を保存します。
     *  保存した後のアプリケーションオブジェクトを返します
     *
     *  @access public
     *  @param  array  $values     レコードの初期値。カラム名をキーとし、
     *                             カラムの値を値として指定した配列
     *                             配列は複数指定可能
     *  @param  boolean $use_trans トランザクションを使用するか否か
     *  @return mixed 作成されたアプリケーションオブジェクト。初期値を複数
     *                指定した場合はアプリケーションオブジェクトの配列
     */
    function create($values = array(), $use_trans = false)
    {
        $class = get_class($this);
        $trans_result = false;
        $trans_error = false;
        $ret = NULL;

        if (is_array($values)) {
            //    必要であればトランザクション開始
            if ($use_trans) {
                $trans_result = $this->begin();
                if (Ethna::isError($trans_result)) {
                    Ethna::raiseNotice(
                        'transaction begin failed :'
                      . $trans_result->getMessage()
                    );
                    $trans_error = true;
                }
            }
            if (array_key_exists('0', $values)) {
                //  複数配列で指定した場合  カラム名には数値
                //  は指定できないのでこのような区別が可能
                $ret = array();
                foreach ($values as $record) {
                    $instance = new $class($record);
                    $trans_result = $instance->save(); 
                    if (Ethna::isError($trans_result)) {
                        Ethna::raiseNotice(
                            $trans_result->getMessage()
                        );
                        $trans_error = true;
                    }
                    $ret[] = $instance;
                } 
            } else {
                //    配列を単独で指定した場合
                $instance = new $class($record);
                $trans_result = $instance->save(); 
                if (Ethna::isError($trans_result)) {
                    Ethna::raiseNotice(
                        $trans_result->getMessage()
                    );
                    $trans_error = true;
                }
                $ret = $instance;
            }
            //    必要であればトランザクション終了
            if ($use_trans) {
                if ($trans_error) {
                    $trans_result = $this->rollback();
                } else {
                    $trans_result = $this->commit();
                }
                if (Ethna::isError($trans_result)) {
                    Ethna::raiseNotice(
                        'transaction [commit|rollback] failed :'
                      . $trans_result->getMessage()
                    );
                }
            }
        } else {
            //    引数は配列で指定しなければならないが、
            //    万が一スカラーであった場合は警告を出し、
            //    空のインスタンスを作成して返す
            Ethna::raiseNotice('Ethna_AppObject#create parameter must be array');
            $ret = new $class();
        }
        return $ret;
    }

    // {{{ begin 
    /**
     *  トランザクションを開始する
     *  Ethna_DB#begin と同義です。 
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function begin()
    {
        if (!$this->__isConnected()) {
            return Ethna::raiseError('Not connected to Database');
        }
        return $this->_db->begin();
    }
    // }}}

    // {{{ commit 
    /**
     *  トランザクションをコミットする
     *  Ethna_DB#commit と同義です。 
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function commit()
    {
        if (!$this->__isConnected()) {
            return Ethna::raiseError('Not connected to Database');
        }
        return $this->_db->commit();
    }
    // }}}

    // {{{ rollback 
    /**
     *  トランザクションをロールバックする
     *  Ethna_DB#rollback と同義です。 
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function rollback()
    {
        if (!$this->__isConnected()) {
            return Ethna::raiseError('Not connected to Database');
        }
        return $this->_db->rollback();
    }
    // }}}

    // {{{ isValid
    /**
     *  アプリケーションオブジェクトの全機能を使える
     *  か否かを返す
     *
     *  @access public
     *  @return bool    true:有効 false:無効
     */
    function isValid()
    {
        //    データベース接続ができ、カラム定義が設定されていて、
        //    プライマリーキーがあること が条件となる
        //    リレーション設定は必須ではない
        if ($this->_isConnected() && !empty($this->_columns)) {
            if (is_array($this->_columns)) {
                foreach ($this->_columns as $def) {
                    if (array_key_exists('primary', $def)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
    // }}}

    // {{{ getTblName
    /**
     *  このオブジェクトと結びついたテーブル名を返す 
     *
     *  @access public
     *  @return array   テーブル名 
     */
    function getTblName()
    {
        return $this->_table_name;
    }
    // }}}

    // {{{ getColDef
    /**
     *  カラム定義を返す 
     *
     *  @access public
     *  @return array   カラム定義
     */
    function getColDef()
    {
        return $this->_columns;
    }
    // }}}

    // {{{ getRelDef
    /**
     *  リレーション定義を返す 
     *
     *  @access public
     *  @return array   リレーション定義
     */
    function getRelDef()
    {
        return $this->_relations;
    }
    // }}}

    // {{{ find
    /**
     *  指定した条件で検索を行う
     *
     *  @access public
     *  @param  mixed  $options プライマリーキーの値または検索条件
     */
    function find($options)
    {
    }
    // }}}

    // {{{ findFirst
    /**
     *  指定した条件で検索を行うが、結果の最初の要素のみを設定する
     *
     *  @access public
     *  @param  mixed  $condition 検索条件
     */
    function findFirst($condition)
    {
    }
    // }}}

    // {{{ findAll
    /**
     *  指定した条件で検索を行い、結果を全て取得する
     *
     *  @access public
     *  @param  mixed  $condition 検索条件
     */
    function findAll($condition)
    {
    }
    // }}}

    // {{{ findBy
    /**
     *  指定したカラム名を条件にして検索を行う 
     *
     *  @access public
     *  @param  string  $colname  条件とするカラム名
     *  @param  mixed   $value    条件とする値
     */
    function findBy($colname, $value)
    {
    }
    // }}}

    // {{{ findOrCreateBy
    /**
     *  指定したカラム名を条件にして検索を行うが、存在しなければ
     *  指定した初期値でレコードを作成する
     *
     *  @access public
     *  @param  mixed   $colname  条件とするカラム名
     *  @param  mixed   $value    条件とする値
     *  @param  array   $init_val INSERTする初期値。 
     */
    function findOrCreateBy($colname, $value, $init_val)
    {
    }
    // }}}

    // {{{ findBySql
    /**
     *  指定したSQL を元にして検索を行う。
     *  Ethna_DB#query と同義です。 
     *
     *  @access public
     *  @param  string $sql    実行するSQL
     *  @param  mixed  $param  パラメータ名
     *                 (? の場合は数値、:name 形式の場合は :name)
     *                 をキーとし、対応する値を値とする配列
     *  @param  int    $fetchmode  フェッチモード
     *  @return mixed  成功時にEthna_DB_Statement
     *                 Ethna_Error: エラー
     */
    function findBySql($sql, $params)
    {
    }
    // }}}

    // {{{ save
    /**
     *  設定された値をデータベースに保存する(INSERT or UPDATE)
     *
     *  TODO: アクションフォームとの親和性を高くするため、アクションフォーム
     *        の値を直接保存できるようにする
     *  @access public
     */
    function save()
    {
    }
    // }}}

    // {{{ replace
    /**
     *  オブジェクトを置換する
     *
     *  MySQLのREPLACE文に相当する動作を行う(INSERT で重複エラーが発生したら
     *  UPDATE を行う)
     */
    function replace()
    {
    }
    // }}}

    // {{{ update
    /**
     *  指定したカラム名を条件にして検索を行う 
     *
     *  @access public
     *  @param  mixed   $id       条件とするプライマリーキーの値
     *  @param  array   $attrs    条件とする値の配列
     *  @return mixed   成功すれば更新した行数 Ethna_Error:エラー
     */
    function update($id, $attrs)
    {
    }
    // }}}
   
    // {{{ updateAttr
    /**
     *  特定の行の特定のカラムのみを更新する
     *  プライマリーキーの値が既に設定されていることが前提です。
     *  そうでなければエラーになるので注意してください。
     *
     *  @access public
     *  @param  mixed   $name  条件とするカラム名
     *  @param  array   $value  
     *  @return mixed   成功すれば更新した行数 Ethna_Error:エラー
     */
    function updateAttr($name, $value)
    {
    }
    // }}}

    // {{{ updateAll
    /**
     *  指定された条件に当てはまるレコードを全て更新する
     *
     *  @access public
     *  @param  array   $condition  更新条件 
     *  @return mixed   成功すれば削除した行数 Ethna_Error:エラー
     */
    function updateAll($condition)
    {
    }
    // }}}

    // {{{ delete 
    /**
     *  指定されたプライマリーキーの値でレコードを削除する
     *
     *  @access public
     *  @param  mixed   $id  条件とするプライマリーキーの値
     *  @return mixed   成功すれば削除した行数 Ethna_Error:エラー
     */
    function delete($id)
    {
    }
    // }}}

    // {{{ deleteAll
    /**
     *  指定された条件でレコードをテーブルのレコードを削除する
     *
     *  @access public
     *  @param  array   $condition  削除条件 
     *  @return mixed   成功すれば削除した行数 Ethna_Error:エラー
     */
    function deleteAll($condition)
    {
    }
    // }}}

    // {{{  toAF 
    /**
     *  設定されたカラム値をアクションフォームに設定する
     *
     *  @param  Ethna_ActionForm $af アクションフォーム 
     *                               NULL の場合は現在のアクションフォーム
     *  @access public
     */
    function toAF($af = NULL)
    {
    }
    // }}}

    // {{{ __isConnected
    /**
     *  内部のDBオブジェクトが接続済みか否かを返す
     *
     *  @access private
     *  @return boolean   true: 接続済み false: 未接続
     */
    function __isConnected()
    {
        if (!is_subclass_of($this->_db, 'Ethna_DB') || !$this->_db->isValid()) {
            return false; 
        }
        return true;
    }

    // {{{ __init
    /**
     *  Ethna_AppObject の完全な初期化を行う
     *  コンストラクタでエラーを投げない代わりに、必須の初期化作業
     *  をここで行う。呼出側では必要に応じてこれを呼ぶことで、エラー
     *  処理をギリギリまで遅延させる
     *
     *  @access private
     *  @param  array   $options  初期化オプション
     *  @return mixed   0: 正常終了 Ethna_Error:エラー 
     */
    function __init($options = array())
    {
        //   データベース接続を初期化        
        $result = $this->setDB();
        if (Ethna::isError($result)) {
            $errmsg = get_class() . ": AppObject init failed -> "
                    . $result->getMessage();
            return Ethna::raiseError($errmsg, E_DB_GENERAL);
        }
        if (isset($options['dbonly'])) {
            return 0;
        }

        //    テーブルのカラム定義を初期化
        //    この値はキャッシュされる
        if (empty($this->_columns)) {
            $columns = $this->_getColumnDef();
            if (Ethna::isError($columns)) {
                return $columns;
            }
            $this->_columns = $columns;
        }         
        $this->_is_valid = $this->isValid();

        return 0; 
    } 
    // }}}

    // {{{ _getColumnDef
    /**
     *  カラム定義を取得します。キャッシュされている場合は、
     *  そこから取得します。
     *
     *  @access private
     *  @return array   カラム定義
     */
    function _getColumnDef()
    {
        //    キャッシュをチェックし、値を取得する
        $table_name = $this->getTblName();
        $cache_manager =& Ethna_CacheManager::getInstance('localfile');
        $cache_manager->setNamespace('ethna_app_object');
        $cache_key = md5($this->_db->getDSN() . '-' . $table_name);

        $columns = array();
        if ($cache_manager->isCached($cache_key, $this->_def_cache_lifetime)) {
            //    cache hit!
            $columns = $cache_manager->get($cache_key,
                                           $this->_def_cache_lifetime);
            if (Ethna::isError($columns)) {
                return Ethna::raiseError(
                           "could not column definition from cache -> "
                         . $columns->getMessage(),
                           E_DB_GENERAL
                       );
            } 
        } else {
            //    そもそもデータベース接続がなければ検索できない
            if (is_null($this->_db)) {
                return Ethna::raiseError("no database connection", E_DB_GENERAL);
            }
    
            //
            // getMetaData メソッドでは、必ず以下のデータが返ってくる
            //
            // 'required' => [true|false] // NOT NULLかどうか
            // 'type' => ...    //  データベースの型名
            // 'unique' => [true|false] //  unique 制約があるか
            // 'primary' => [true|false] // primary keyか否か 
            // 'seq' => [true|false] // seq(auto increment) か否か 
            //
            $columns = $this->_db->getMetaData($table_name);
            if(Ethna::isError($columns)){
                return $columns;
            }
            
            //   データベースから取得した値はキャッシュする
            $cache_manager->set($cache_key, $columns);
        } 

        //   ユーザー定義のカラム定義をマージする
        //   ここはなぜキャッシュしないかと言えば、ソースの書き換え
        //   は即座に反映させるべきだからである
        $columns = array_merge_recursive($columns, $this->_columns);

        return $columns;
    }
    // }}}

    // {{{ _clearCache
    /**
     *  キャッシュデータを削除する
     *
     *  @access private
     */
    function _clearCache()
    {
        $class_name = strtolower(get_class($this));
        foreach (array('_ETHNA_APP_OBJECT_CACHE',
                       '_ETHNA_APP_MANAGER_OL_CACHE',
                       '_ETHNA_APP_MANAGER_OPL_CACHE',
                       '_ETHNA_APP_MANAGER_OP_CACHE') as $key) {
            if (array_key_exists($key, $GLOBALS)
                && array_key_exists($class_name, $GLOBALS[$key])) {
                unset($GLOBALS[$key][$class_name]);
            }
        }
    }
    // }}}
}
// }}}
?>
