<?php
// vim: foldmethod=marker
/**
 *  Ethna_DB.php
 *
 *  @author     Masaki Fujimoto <fujimoto@php.net>
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: 69cc5628eef966850abb0a95700510f5dea0c11a $
 */

if (!defined('DB_FETCHMODE_ASSOC')) {
    define('DB_FETCHMODE_ASSOC', 1);
}

// {{{ Ethna_DB
/**
 *  データベースアクセスドライバ抽象クラス
 *
 *  実装は各ドライバに委ねられる
 *
 *  @author     Masaki Fujimoto <fujimoto@php.net>
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB
{
    /**#@+
     *  @access private
     */

    /** @var    resource   データベース依存 接続ハンドル  */
    var $_db = NULL;  

    /** @var    array   トランザクション管理スタック */
    var $_transaction = array();

    /** @var    object  Ethna_Logger    ログオブジェクト */
    var $_logger;

    /** @var    string  DSN */
    var $_dsn;

    /** @var    array   DSN (DB::parseDSN()の返り値) */
    var $_dsninfo;

    /** @var    bool    持続接続フラグ */
    var $_persistent;

    /**#@-*/

    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  string  $dsn DSN
     *  @param  bool    $persistent 持続接続設定
     */
    function Ethna_DB($dsn, $persistent)
    {
        $this->_dsn = $dsn;
        $this->_persistent = $persistent;
        $this->_dsninfo = $this->parseDSN($dsn);
    }

    /**
     *  ロガーを設定する
     *  このインターフェイスによって、Ethna の
     *  ことを知らなくても理論上はデータベース
     *  インターフェイスやORMを利用可能にする
     *
     *  @access public
     *  @param  object $logger ロガーオブジェクト
     *  Note: 理論上は、log($level, $msg) を実装している
     *  クラスのインスタンスであればよい
     */
    function setLogger($logger)
    {
        $this->_logger = $logger;
    }

    /**
     *  ロガーを取得する
     *
     *  @access public
     *  @return object ロガーオブジェクト
     */
    function getLogger()
    {
        return $this->_logger;
    }

    /**
     *  DBに接続する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function connect()
    {
    }

    /**
     *  DB接続を切断する
     *
     *  @access public
     */
    function disconnect()
    {
    }

    /**
     *  DB接続状態を返す
     *
     *  @access public
     *  @return bool    true:正常(接続済み) false:エラー/未接続
     */
    function isValid()
    {
        if (is_resource($this->_db) && !empty($this->_db)) {
            return true;
        }
        return false; 
    }

    /**
     *  DBトランザクションを開始する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function begin()
    {
    }

    /**
     *  DBトランザクションを中断する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function rollback()
    {
    }

    /**
     *  DBトランザクションを終了する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function commit()
    {
    }
   
    /**
     *  SQL ステートメントを実行する準備を行い、
     *  Ethna_Statementオブジェクトを返します。
     *
     *  Ethna_Statement::execute() メソッドによ
     *  って実行される SQL ステートメントを準備
     *  します。 SQL ステートメントは、文が実行
     *  されるときに実際の値に置き換えられます。
     *
     *  (ステートメントにパラメータを指定する場合、
     *   ? で指定すること)
     *  @access public
     *  @param  $sql  実行を準備するSQL
     *  @return mixed 成功時にEthna_DB_Statement
     *                Ethna_Error:エラー
     */
    function prepare($sql)
    {
    }

    /**
     *  SQL ステートメントを実行し、作用した行数を返します。
     *  SELECT 文からは結果を返しません。SELECT 文を指定した
     *  場合は常に0が返ります。
     *
     *  @access public
     *  @param  $sql    実行するSQL
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function exec($sql)
    {
    }

    /**
     *  SQL ステートメントを実行し、結果セットを
     *  Ethna_DB_Statement オブジェクトとして返します。
     *  場合は常に0が返ります。
     *
     *  @access public
     *  @param  string $sql    実行するSQL
     *                         (ステートメントにパラメータを指定する場合、
     *                          ? で指定すること)
     *  @param  mixed  $param  パラメータの配列。? の数と一致しなければ
     *                         なりません。
     *  @param  int    $fetchmode  フェッチモード
     *  @return mixed  成功時にEthna_DB_Statement
     *                 Ethna_Error: エラー
     */
    function query($sql, $param = array(), $fetchmode = DB_FETCHMODE_ASSOC)
    {
        $stmt = $this->prepare($sql);
        if (Ethna::isError($stmt)) {
            return $stmt;
        }
        $stmt->setFetchMode($fetchmode);
        $result = $stmt->exec($sql, $param);
        if (Ethna::isError($result)) {
            return $result;
        }
        return $stmt;
    }

    /**
     *  直前のINSERTによるIDを取得する
     *  返す値はデータベース依存です。
     *
     *  @access public
     *  @param  string  $name データベース依存のパラメータ
     *  @return mixed   int:直近のINSERTにより生成されたID null:エラー
     */
    function getInsertId($name = NULL)
    {
        return NULL;
    }

    /**
     *  文字列をエスケープする 
     *
     *  @access public
     *  @param  string  $value  エスケープ対象の値 
     *  @return string  エスケープ済みの値
     */
    function escape($value)
    {
        return $value;
    }

    /**
     *  バイナリデータをデータベースに保存できる形式に
     *  エンコードします
     *
     *  @access public
     *  @param  string  $value  エスケープ対象の値 
     *  @return string  エスケープ済みの値
     */
    function blobEncode($value)
    {
        return $value;
    }

    /**
     *  データベースに保存されたエスケープ済みバイナリ
     *  データを元のデータに復元します。
     *
     *  @access public
     *  @param  string  $value  復元対象の値 
     *  @return string  復元済みの値
     */
    function blobDecode($value)
    {
        return $value;
    }

    /**
     *  バイナリデータを指定したテーブルのカラムに保存します。
     *  BLOBデータを保存する場合は、該当カラムにNULLをINSERT
     *  してからこのメソッドを使用してください。
     *
     *  @access public
     *  @param  string  $table    保存対象のテーブル名  
     *  @param  string  $column   保存対象のカラム名  
     *  @param  string  $value    保存する値  
     *  @param  string  $where    保存するための条件  
     *  @param  string  $blobtype BLOBのタイプ(CLOB|BLOB)
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function updateBlob($table, $column, $value, $where, $blobtype='BLOB')
    {
        $escaped_value = $this->escape($value);
        return $this->exec("UPDATE $table SET $column=$escaped_value WHERE $where");
    }

    /**
     *  バイナリデータを指定したファイルから読み出し、指定した
     *  テーブルのカラムに保存します。
     *  BLOBデータを保存する場合は、該当カラムにNULLをINSERT
     *  してからこのメソッドを使用してください。
     *
     *  @access public
     *  @param  string  $table    保存対象のテーブル名  
     *  @param  string  $column   保存対象のカラム名  
     *  @param  string  $path     保存するデータが格納されたファイル
     *  @param  string  $where    保存するための条件  
     *  @param  string  $blobtype BLOBのタイプ(CLOB|BLOB)
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function updateBlobFile($table, $column, $path, $where, $blobtype='BLOB')
    {
        if (!file_exists_ex($path)) {
            return Ethna::raiseError("file does not exist : $path"); 
        }
        $contents = file_get_contents($path, true);
        return $this->updateBlob($table, $column, $contents, $where, $blobtype);
    }

    /**
     *  テーブル定義情報を取得する
     *
     *  <code>
     *    'colname' => array(
     *                     'required' => [true|false] // NOT NULLかどうか
     *                     'type' => ...    //  データベースの型名
     *                     'unique' => [true|false] //  unique 制約があるか
     *                     'primary' => [true|false] // primary keyか否か 
     *                     'seq' => [true|false] // seq(auto increment) か否か 
     *                 )
     *  </code>
     *  @access public
     *  @param  string  $table_name  取得対象のテーブル名
     *  @return mixed   array: PEAR::DBに準じたメタデータ
     *                  Ethna_Error::エラー
     */
    function getMetaData($table_name)
    {
        //   このメソッドはAppObject
        //   との連携に必要。
    }

    /**
     *  DSNを取得する
     *
     *  @access public
     *  @return string  DSN
     */
    function getDSN()
    {
        return $this->_dsn;
    }

    /**
     *  DSN(パース済み)を取得する
     *
     *  @access public
     *  @param  string $key DSN設定キー 
     *  @return mixed パース済みDSN情報, $key が指定された場合は対応する情報
     */
    function getDSNInfo($key = NULL)
    {
        if (is_null($key) == false) {
            return (isset($this->_dsninfo[$key]))
                 ? $this->_dsninfo[$key]
                 : NULL;
        }
        return $this->_dsninfo;
    }

    /**
     *  データベース依存の接続ハンドルを取得する 
     *
     *  このAPIはユーザーが使用しても意味がありません。
     *  生のハンドルを使用すると、ユーザー側のロジックが
     *  データベース依存になってしまうからです。
     *
     *  @access public
     *  @return resource  データベース依存の接続ハンドル 
     */
    function getRawDB()
    {
        return $this->_db;
    }

    /**
     *  データベースのタイプを取得する
     *
     *  @access public
     *  @return string  データベースのタイプ
     */
    function getType()
    {
    }

    /**
     * データソースを解析する
     *
     * DSNに対して追加のキーがある場合は、URI の
     * クエリストリング形式でDSNの最後に追加できます。
     *
     * 指定できるDSNの完全な形式は以下の通りです。
     * <code>
     *  phptype(dbsyntax)://username:password@protocol+hostspec/database?option=8&another=true
     * </code>
     *
     * 以下のように、様々なバリエーションが指定可能です。
     * <code>
     *  phptype://username:password@protocol+hostspec:110//usr/db_file.db?mode=0644
     *  phptype://username:password@procotol(proto_opts)/database_name
     *  phptype://username:password@hostspec/database_name
     *  phptype://username:password@hostspec
     *  phptype://username@hostspec
     *  phptype://hostspec/database
     *  phptype://hostspec
     *  phptype(dbsyntax)
     *  phptype
     * </code>
     *
     * @param string $dsn 解析するデータソース名
     *
     * @return 次のキーを格納した連想配列
     *  + phptype:  PHP で使われるデータベースバックエンド (mysql, odbc etc.)
     *  + dbsyntax: SQLの構文などに関して指定するデータベース(odbcで使用)
     *  + protocol: 通信に使うプロトコル (tcp, unix etc.) デフォルトは tcp
     *  + hostspec: ホスト名 (hostname[:port]) protocol が tcp のときだけ解釈されます
     *  + port    : ポート番号 protocol が tcp の場合だけ解釈されます
     *  + socket  : unixドメインソケットのパス名 protocol が unix の場合だけ hostspec の部分が socket のパス名と見做されます。
     *  + database: データベースサーバ上で使用するデータベース名
     *  + username: ログインするユーザ名
     *  + password: パスワード
     *  + ? 以降にオプションをクエリストリング形式で指定可能
     */
    function parseDSN($dsn)
    {
        $parsed = array(
            'phptype'  => false,
            'dbsyntax' => false,
            'username' => false,
            'password' => false,
            'protocol' => false,
            'hostspec' => false,
            'port'     => false,
            'socket'   => false,
            'database' => false,
            //  上記以外でも、 ? のあとに追加したユーザー定義のオプション
            //  のキーと値が格納される
        );

        //   dsn が配列なら既存のものとマージするのみ
        //   但し dbsyntax がない場合はそれを補う
        if (is_array($dsn)) {
            $dsn = array_merge($parsed, $dsn);
            if (!$dsn['dbsyntax']) {
                $dsn['dbsyntax'] = $dsn['phptype'];
            }
            return $dsn;
        }

        //   :// で分割
        if (($pos = strpos($dsn, '://')) !== false) {
            $str = substr($dsn, 0, $pos);  // "://" より以前
            $dsn = substr($dsn, $pos + 3); // "://" より以後 
        } else {
            $str = $dsn;  //  "://" が ない場合は全体をとる
            $dsn = null;
        }

        // $str => phptype(dbsyntax) にマッチさせ、それぞれを取得
        // dbsyntax は必須ではない。ない場合は phptype と同じになる
        if (preg_match('|^(.+?)\((.*?)\)$|', $str, $arr)) {
            $parsed['phptype']  = $arr[1];
            $parsed['dbsyntax'] = !$arr[2] ? $arr[1] : $arr[2];
        } else {
            $parsed['phptype']  = $str;
            $parsed['dbsyntax'] = $str;
        }

        if ($dsn === false || !count($dsn)) {  // "hoge://" で終わった場合
            return $parsed;
        }

        // username と password 部分を（'@' があれば）取得
        // $dsn => username:password@protocol+hostspec/database?opt=val
        if (($at = strrpos($dsn, '@')) !== false) {
            $str = substr($dsn, 0, $at);   //  '@' で分割
            $dsn = substr($dsn, $at + 1);  //  '@' より後が残る
            if (($pos = strpos($str, ':')) !== false) {
                $parsed['username'] = rawurldecode(substr($str, 0, $pos));
                $parsed['password'] = rawurldecode(substr($str, $pos + 1));
            } else {
                //  ':' がない場合 '@' より前全体をユーザ名と解釈する
                $parsed['username'] = rawurldecode($str);
            }
        }

        //  protocol と hostspec を見つける
        //  $dsn => protocol+hostspec/database?opt=val 
        if (preg_match('|^([^(]+)\((.*?)\)/?(.*?)$|', $dsn, $match)) {
            //   以下の形式の場合
            // $dsn => protocol(proto_opts)/database 
            // $dsn => protocol(proto_opts)database
            $proto      = $match[1]; // protocol の部分
            $proto_opts = $match[2] ? $match[2] : false; // proto_opts の部分
            $dsn        = $match[3]; // database?opt=val が残る
        } else {
            // $dsn => protocol+hostspec/database (old format)
            if (strpos($dsn, '+') !== false) {
                list($proto, $dsn) = explode('+', $dsn, 2);
            }
            if (strpos($dsn, '/') !== false) {
                //  proto_opts には '/' 以前が残る(hostspecの部分)
                list($proto_opts, $dsn) = explode('/', $dsn, 2);
            } else {
                $proto_opts = $dsn; // '+' も '/' もない場合は全体が残る 
                $dsn        = null;
            }
        }

        // protocol と hostspec (proto_opts) の値を処理する
        // protocol は tcp または unix に対応していない
        $parsed['protocol'] = (!empty($proto)) ? $proto : 'tcp';
        $proto_opts         = rawurldecode($proto_opts);
        if ($parsed['protocol'] == 'tcp') {
            //  port が解釈されるのは protocol が tcp で、
            //  かつ hostspec or (proto_opts) の部分に ':' がある場合のみ
            if (strpos($proto_opts, ':') !== false) {
                list($parsed['hostspec'], $parsed['port']) = explode(':', $proto_opts);
            } else {
                $parsed['hostspec'] = $proto_opts;
            }
        } elseif ($parsed['protocol'] == 'unix') {
            //   procotol が 'unix' の場合は、
            //   hostspec または (proto_opt) 部分をソケット名と見做す
            $parsed['socket'] = $proto_opts;
        }

        // データベース名とそれ以後のオプション引数を解釈 
        // $dsn => database?opt=val
        if ($dsn) {
            if (($pos = strpos($dsn, '?')) === false) {
                // ? がないため、dsn の残りをすべて
                // データベース名と見做す
                $parsed['database'] = rawurldecode($dsn);
            } else {
                // ? より前をデータベース名と見なし、
                // ? の後のオプション引数を解釈する
                // /database?param1=value1&param2=value2
                $parsed['database'] = rawurldecode(substr($dsn, 0, $pos));
                $dsn                = substr($dsn, $pos + 1);
                if (strpos($dsn, '&') !== false) {
                    $opts = explode('&', $dsn);
                } else { // database?param1=value1
                    $opts = array($dsn);
                }
                foreach ($opts as $opt) {
                    list($key, $value) = explode('=', $opt);
                    if (!isset($parsed[$key])) {
                        // パラメータのオーバーライドは許可しない
                        // 早い者勝ちとする
                        $parsed[$key] = rawurldecode($value);
                    }
                }
            }
        }
        return $parsed;
    }
}
// }}}
?>
