<?php
// vim: foldmethod=marker
/**
 *  Ethna_DB_Statement.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: 99b2e2a077b26debe4326b4ae55b13d4eea97ca0 $
 */

//  フェッチモード
/*
 *   １行をカラム名をキーとして、対応する値を値とする
 *   連想配列として返します 
 *
 *   例： array(
 *            'colname1' => 'value1',
 *            'colname2' => 'value2',
 *        );
 */
if (!defined('DB_FETCHMODE_ASSOC')) {
    define('DB_FETCHMODE_ASSOC', 1);
}
/*
 *   １行を、0からはじまるカラムの番号をキーとして、
 *   対応する値の連想配列として返します 
 *
 *   例： array(
 *            '0' => 'value1',
 *            '1' => 'value2',
 *        );
 */
define('DB_FETCHMODE_NUM',   2);


// {{{ Ethna_DB_Statement
/**
 *  実行前にはプリペアドステートメント。
 *  実行後には関連する結果セットを表す抽象クラス
 *
 *  実装は各ドライバに委ねられる
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_Statement
{
    /**#@+
     *  @access protected
     */

    /** @var  Ethna_DB データベース接続オブジェクト  */
    var $_ethdb;

    /** @var  resource 各ドライバ依存 データベース接続  */
    var $_db;

    /** @var  object  ロガーオブジェクト  */
    var $_logger;

    /** @var  mixed    結果セット */
    var $_result = NULL; 
     
    /** @var  mixed    バインドする値  */
    var $_bind_vals = array(); 

    /** @var  boolean  プリペアドステートメントがサポートされているか否か  */
    var $_pstmt_supported = false;

    /** @var  int  フェッチモード  */
    var $_fetch_mode = DB_FETCHMODE_ASSOC;

    /** @var  mixed    ドライバ依存オプション  */
    var $_options = array(); 

    /**#@-*/

    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  Ethna_DB $db データベース接続 
     */
    function Ethna_DB_Statement(&$db)
    {
        $this->_ethdb = $db;
        $this->_db = $db->getRawDB();
        $this->_bind_vals = array();
    } 

    /**
     *  ロガーを設定する
     *
     *  @access public
     *  @param  object $logger ロガーオブジェクト
     *  Note: 理論上は、log($level, $msg) を実装している
     *  クラスのインスタンスであればよい
     */
    function setLogger($logger)
    {
        $this->_logger = $logger;
    }

    /**
     *  プリペアドステートメントに値をバインドする
     *
     *  @access public
     *  @param  mixed  $param  パラメータ名
     *  @param  mixed  $value  パラメータに対応する値
     *  @param  mixed  $value  パラメータの型
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function bind($param, $value, $type = NULL)
    {
        //    未使用  PDO のラッパーとする為
        //    PHP5 向けに予約されている
        //    @see http://jp.php.net/manual/ja/pdostatement.bindvalue.php
    }

    /**
     *  プリペアドステートメントを実行します。
     *  プリペアドステートメントがサポートされて
     *  いないドライバでは、動作がエミュレートされます。
     *  この場合、動作が速くなるわけではないので
     *  注意して下さい。
     *
     *  @access public
     *  @param  string $sql 実行するSQL (パラメータは ? で指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function exec($sql, $params = array())
    {
        if ($this->_pstmt_supported) {
            return $this->__emulatePstmt($sql, $params);
        }
        return $this->__exec($sql, $params);
    }

    /**
     *  exec メソッドの実際の実行ロジックです。
     *  
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? で指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __exec($sql, $params = array())
    {
        return Ethna::raiseError(
                   'prepared statement is not supported by this driver',
                   E_DB_GENERAL
               );
    }
    
    /**
     *  プリペアドステートメント実行結果から
     *  次の行を取得します。 
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse 
     */
    function fetchRow()
    {
    }
     
    /**
     *  プリペアドステートメント実行結果の
     *  結果セットを「全て」配列で返します
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse
     */
    function fetchAll()
    {
    }

    /**
     *  直近の DELETE, INSERT, UPDATE 文によっ
     *  て作用した行数を返します。 
     *
     *  @access public
     *  @return int  作用した行数
     *               SELECT の場合は 0
     */
    function affectedRows()
    {
    }

    /**
     *  フェッチモードを設定します 
     *
     *  @access public
     *  @param  int $mode フェッチモード  
     */
    function setFetchMode($mode = DB_FETCHMODE_ASSOC)
    {
        $this->_fetch_mode = $mode;
    }

    /**
     *  ドライバ依存のオプションの値を設定します。
     *
     *  @access public
     *  @param  string $name オプション名
     *  @param  mixed  $value オプション値
     */
    function setOption($name, $value)
    {
        if (!strcmp($name, '__emulation_mode')) {
           $this->_pstmt_supported = $value;
        } 
        $this->_options[$name] = $value;
    }

    /**
     *  ドライバ依存のオプションの値を取得します。
     *
     *  @access public
     *  @param  string $name オプション名
     *  @return mixed  $value オプション値
     */
    function getOption($name)
    {
        if (isset($this->_options[$name])) {
            return $this->_options[$name];
        }
        return NULL;
    }

    /**
     *  プリペアドステートメントのエミュレーションを行う
     *  ための SQL を取得する共通ロジックです。パラメータは
     *  適切にエスケープされます。
     *
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? として指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return string エスケープ済みのSQL
     */
    function __getEmulatePstmtSQL($sql, $param = array())
    {
        $sql_parts = explode('?', $sql);
        $prepare_sql = '';
        if (count($sql_parts) > 1) {
            $count = 1;
            $param_count = count($param);
            foreach ($sql_parts as $part) {
                if (empty($part)) {
                    continue;
                }
                if ($count > $param_count) {
                    $prepare_sql .= $part;
                    break;
                }    

                $idx = $count - 1;
                $val = isset($param[$idx])
                     ? $param[$idx]
                     : NULL;
                if (is_numeric($val) || is_float($val)) {
                    $escaped_val = "'" . $this->_ethdb->escape($val) . "'";
                    $prepare_sql .= ($part . $escaped_val . ' ');
                }
                if (is_null($val)) {
                    $prepare_sql .= ($part . ' NULL ');
                }
                if (is_string($val)) {
                    $escaped_val = "'" . $this->_ethdb->escape($val) . "'";
                    $prepare_sql .= ($part . $escaped_val . ' ');
                }
                if (is_bool($val)) {
                    $val = ($val) ? 1 : 0;
                    $prepare_sql .= ($part . $val . ' ');
                }
                //
                //    we do not BLOB Support here.
                //
                $count++;
            }
        } else {
            $prepare_sql = $sql;
        }
        return $prepare_sql;
    }

    /**
     *  プリペアドステートメントのエミュレー
     *  ションロジックです。
     *
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? で指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __emulatePstmt($sql, $param = array())
    {
    }
}
// }}}
?>
