/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.action;

import java.io.Serializable;
import java.text.NumberFormat;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.entity.Document;
import jp.sf.fess.form.IndexForm;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.LabelTypeHelper;
import jp.sf.fess.service.SearchService;

import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.util.RequestUtil;

public class IndexAction implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final int DEFAULT_PAGE_SIZE = 20;

    public static final long DEFAULT_START_COUNT = 0;

    @ActionForm
    @Resource
    protected IndexForm indexForm;

    @Resource
    protected SearchService searchService;

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected LabelTypeHelper labelTypeHelper;

    public List<Document> documentItems;

    public String pageSize;

    public String currentPageNumber;

    public String allRecordCount;

    public String allPageCount;

    public boolean existNextPage;

    public boolean existPrevPage;

    public String currentStartRecordNumber;

    public String currentEndRecordNumber;

    public List<String> pageNumberList;

    public String execTime;

    public List<Map<String, String>> labelItems;

    @Execute(validator = false)
    public String index() {
        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        return "index.jsp";
    }

    protected String doSearch() {
        String query;
        if (StringUtil.isBlank(indexForm.query)) {
            if (StringUtil.isBlank(indexForm.labelTypeValue)) {
                // redirect to index page
                indexForm.query = null;
                return "index?redirect=true";
            } else {
                query = "label:\"" + indexForm.labelTypeValue + "\"";
            }
        } else {
            if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
                query = indexForm.query + " label:\""
                        + indexForm.labelTypeValue + "\"";
            } else {
                query = indexForm.query;
            }
        }

        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        // init pager
        if (StringUtil.isBlank(indexForm.start)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            try {
                Long.parseLong(indexForm.start);
            } catch (NumberFormatException e) {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                int num = Integer.parseInt(indexForm.num);
                if (num > 100) {
                    // max page size
                    indexForm.num = "100";
                }
            } catch (NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        long startTime = System.currentTimeMillis();
        documentItems = searchService.selectList(query, Integer
                .parseInt(indexForm.start), Integer.parseInt(indexForm.num));
        long execTime = System.currentTimeMillis() - startTime;

        NumberFormat nf = NumberFormat.getInstance(RequestUtil.getRequest()
                .getLocale());
        nf.setMaximumIntegerDigits(2);
        nf.setMaximumFractionDigits(2);
        try {
            this.execTime = nf.format(((double) execTime) / 1000);
        } catch (Exception e) {
        }

        Beans.copy(documentItems, this).includes("pageSize",
                "currentPageNumber", "allRecordCount", "allPageCount",
                "existNextPage", "existPrevPage", "currentStartRecordNumber",
                "currentEndRecordNumber", "pageNumberList").execute();

        return "search.jsp";
    }

    @Execute(validator = false)
    public String search() {
        return doSearch();
    }

    @Execute(validator = false)
    public String prev() {
        return doMove(-1);
    }

    @Execute(validator = false)
    public String next() {
        return doMove(1);
    }

    @Execute(validator = false)
    public String move() {
        return doMove(0);
    }

    protected String doMove(int move) {
        int pageSize = DEFAULT_PAGE_SIZE;
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                pageSize = Integer.parseInt(indexForm.num);
            } catch (NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        if (StringUtil.isBlank(indexForm.pn)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            Integer pageNumber = Integer.parseInt(indexForm.pn);
            if (pageNumber != null && pageNumber > 0) {
                pageNumber = pageNumber + move;
                if (pageNumber < 1) {
                    pageNumber = 1;
                }
                indexForm.start = String.valueOf((pageNumber - 1) * pageSize);
            } else {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }

        return doSearch();
    }

    protected boolean isMobile() {
        if (browserTypeHelper == null) {
            return false;
        }

        return browserTypeHelper.isMobile();
    }

    public List<Map<String, String>> getLabelTypeItems() {
        return labelTypeHelper.getLabelTypeItems();
    }

    public boolean isDisplayLabelTypeItems() {
        return !labelTypeHelper.getLabelTypeItems().isEmpty();
    }

}