/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.helper.RoleQueryHelper;

import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.util.RequestUtil;

public class QueryHelperImpl implements QueryHelper, Serializable {

    private static final long serialVersionUID = 1L;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected RoleQueryHelper roleQueryHelper;

    public String[] supportedFields = new String[] { "url", "host", "site",
            "title", "content", "contentLength", "lastModified", "mimetype",
            "label" };

    /* (non-Javadoc)
     * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
     */
    public String build(String query) {
        String baseQuery = buildQuery(query);
        if (StringUtil.isBlank(baseQuery)) {
            return "";
        }

        if (browserTypeHelper == null && roleQueryHelper == null) {
            return baseQuery;
        }

        StringBuilder queryBuf = new StringBuilder(255);
        queryBuf.append('(');
        queryBuf.append(baseQuery);
        queryBuf.append(')');

        if (browserTypeHelper != null) {
            queryBuf.append(" AND type:\"");
            queryBuf.append(browserTypeHelper.getBrowserType());
            queryBuf.append('"');
        }

        if (roleQueryHelper != null) {
            List<String> roleList = roleQueryHelper.build();
            if (!roleList.isEmpty()) {
                queryBuf.append(" AND (");
                boolean isFirst = true;
                for (String role : roleList) {
                    if (isFirst) {
                        isFirst = false;
                    } else {
                        queryBuf.append(" OR ");

                    }
                    queryBuf.append("role:\"");
                    queryBuf.append(role);
                    queryBuf.append('"');
                }
                queryBuf.append(')');
            }
        }

        return queryBuf.toString();
    }

    protected String buildQuery(String query) {
        String[] values = splitQuery(query);
        if (values.length == 0) {
            return "";
        }

        // set queries to request
        RequestUtil.getRequest().setAttribute(Constants.QUERIES, values);

        StringBuilder queryBuf = new StringBuilder(255);
        StringBuilder titleBuf = new StringBuilder(255);
        StringBuilder contentBuf = new StringBuilder(255);
        for (String value : values) {
            boolean nonPrefix = false;
            // check prefix
            for (String field : supportedFields) {
                String prefix = field + ":";
                if (value.startsWith(prefix)
                        && value.length() != prefix.length()) {
                    if (queryBuf.length() > 0) {
                        queryBuf.append(" AND ");
                    }
                    queryBuf.append(prefix);
                    queryBuf.append('"');
                    queryBuf.append(value.substring(prefix.length()));
                    queryBuf.append('"');
                    nonPrefix = true;
                    break;
                }
            }

            if (!nonPrefix) {
                // title
                if (titleBuf.length() > 0) {
                    titleBuf.append(" AND ");
                }
                titleBuf.append("title:\"");
                titleBuf.append(value);
                titleBuf.append('"');

                // content
                if (contentBuf.length() > 0) {
                    contentBuf.append(" AND ");
                }
                contentBuf.append("content:\"");
                contentBuf.append(value);
                contentBuf.append('"');
            }
        }

        if (titleBuf.length() > 0) {
            boolean append = false;
            if (queryBuf.length() > 0) {
                append = true;
            }
            if (append) {
                queryBuf.append(" AND (");
            }
            queryBuf.append('(').append(titleBuf.toString()).append(") OR (")
                    .append(contentBuf.toString()).append(')');
            if (append) {
                queryBuf.append(')');
            }
        }
        return queryBuf.toString();
    }

    protected String[] splitQuery(String query) {
        List<String> valueList = new ArrayList<String>();
        StringBuilder buf = new StringBuilder();
        boolean quoted = false;
        for (int i = 0; i < query.length(); i++) {
            char c = query.charAt(i);
            switch (c) {
            case '"':
                quoted = !quoted;
                break;
            case ' ':
                if (quoted) {
                    buf.append(c);
                } else {
                    if (buf.length() > 0) {
                        valueList.add(buf.toString());
                    }
                    buf = new StringBuilder();
                }
                break;
            default:
                buf.append(c);
                break;
            }
        }
        if (buf.length() > 0) {
            valueList.add(buf.toString());
        }
        return valueList.toArray(new String[valueList.size()]);
    }
}
